<?php

namespace App\Authorization\Authorizer;

use App\Authorization\Exception\AuthorizationDeniedException;
use App\Authorization\Exception\EntityNotFoundException;
use App\Authorization\Session\UserSessionInterface;
use InvalidArgumentException;
use Order;

final class OrderEntityAuthorizer extends EntityAuthorizer
{
    /** @var Order */
    private $orderModel;

    public function __construct($orderModel)
    {
        $this->orderModel = $orderModel;
    }

    /**
     * @throws AuthorizationDeniedException
     */
    public function authorizeAppAdmin(UserSessionInterface $userSession, $entityId): bool
    {
        return $this->denyAuthorization();
    }

    /**
     * @throws AuthorizationDeniedException
     */
    public function authorizeCarrier(UserSessionInterface $userSession, $entityId): bool
    {
        return $this->denyAuthorization();
    }

    /**
     * @throws AuthorizationDeniedException
     * @throws EntityNotFoundException
     */
    public function authorizeCountryAdmin(UserSessionInterface $userSession, $entityId): bool
    {
        $order = $this->fetchOrder($entityId);

        if ($userSession->company() !== $order->company_code) {
            return $this->denyAuthorization();
        }

        if (!in_array($order->user_id, $userSession->countryUsers(), true)) {
            return $this->denyAuthorization();
        }

        return $this->grantAuthorization();
    }

    /**
     * @throws AuthorizationDeniedException
     * @throws EntityNotFoundException
     */
    public function authorizeCustomer(UserSessionInterface $userSession, $entityId): bool
    {
        $order = $this->fetchOrder($entityId);

        if (!in_array($order->customer_id, $userSession->customers(), true)) {
            return $this->denyAuthorization();
        }

        return $this->grantAuthorization();
    }

    /**
     * @throws AuthorizationDeniedException
     * @throws EntityNotFoundException
     */
    public function authorizeUser(UserSessionInterface $userSession, $entityId): bool
    {
        $order = $this->fetchOrder($entityId);

        if ($userSession->user() !== $order->user_id) {
            return $this->denyAuthorization();
        }

        return $this->grantAuthorization();
    }

    /**
     * @throws InvalidArgumentException
     * @throws EntityNotFoundException
     */
    private function fetchOrder($entityId)
    {
        // @todo refactor controllers to not handle request without orderId, remove this code then
        /**
         * In controllers code is often ran when no orderId has been supplied. This causes unexpected behaviour. We
         * choose to throw an exception if the order is not found, so application is stopped from running and causing
         * unexpected behaviour.
         */
        if ($entityId === null) {
            throw new InvalidArgumentException(
                'Fail if no order ID, to prevent code from running without ID in Orders controller',
                404
            );
        }

        /**
         * Code in controllers is not very structured. In ideal situation we inject the order, but code is too
         * unstructured, so we inject this authorizer with the model and fetch the entity in this authorizer
         */
        $orderResult = $this->orderModel->getOrderForAuthorization($entityId);

        /**
         * If no order can be found we throw an exception, this is because we are fetching the order inside the
         * authorizer. Normally the controller would handle this.
         */
        if ($orderResult->num_rows() === 0) {
            throw new EntityNotFoundException('Order not found');
        }

        return $orderResult->row();
    }
}
