<?php

namespace App\Authorization\Session;

use App\Authorization\Exception\UserTypeInvalidException;
use InvalidArgumentException;

final class UserSession implements UserSessionInterface
{
    public const USER_TYPE_APP_ADMIN = 'app_admin';
    public const USER_TYPE_CARRIER = 'Carrier';
    public const USER_TYPE_COUNTRY_ADMIN = 'Country Admin';
    public const USER_TYPE_CUSTOMER = 'Customer';
    public const USER_TYPE_USER = 'Truck_Owner';

    private const SESSION_KEY_BUSINESS_TYPE = 'business_type';
    private const SESSION_KEY_COMPANY_CODE = 'company_code';
    private const SESSION_KEY_COUNTRY_USER_IDS = 'country_user_ids';
    private const SESSION_KEY_CUSTOMER_ID = 'cust_id';
    private const SESSION_KEY_SUB_CUSTOMER_IDS = 'sub_cust';
    private const SESSION_KEY_USER_ID = 'user_id';

    private const CODEIGNITER_SESSION_REQUIRED_FIELDS = [
        self::SESSION_KEY_COMPANY_CODE,
        self::SESSION_KEY_USER_ID,
        self::SESSION_KEY_BUSINESS_TYPE
    ];

    private $businessType;
    private $company;
    private $countryUsers;
    private $customer;
    private $subCustomers;
    private $user;

    private function __construct(array $sessionData)
    {
        $this->businessType = $sessionData[self::SESSION_KEY_BUSINESS_TYPE] ?? null;
        $this->company = $sessionData[self::SESSION_KEY_COMPANY_CODE] ?? null;
        $this->user = $sessionData[self::SESSION_KEY_USER_ID] ?? null;
        $this->countryUsers = $sessionData[self::SESSION_KEY_COUNTRY_USER_IDS] ?? [];
        $this->customer = $sessionData[self::SESSION_KEY_CUSTOMER_ID] ?? null;
        $this->subCustomers = $sessionData[self::SESSION_KEY_SUB_CUSTOMER_IDS] ?? [];
    }

    public function company(): string
    {
        return $this->company;
    }

    public function countryUsers(): array
    {
        $attachedCountryUserIds = [];
        $attachedCountryUserIds[] = $this->user;

        if (!empty($this->countryUsers) && is_array($this->countryUsers)) {
            $attachedCountryUserIds = array_merge(
                $attachedCountryUserIds,
                $this->countryUsers
            );
        }

        return $attachedCountryUserIds;
    }

    /**
     * @throws InvalidArgumentException
     * @throws UserTypeInvalidException
     */
    public static function createFromCodeIgniterSessionData(array $sessionData): self
    {
        if (!empty(array_diff(self::CODEIGNITER_SESSION_REQUIRED_FIELDS, array_keys($sessionData)))) {
            throw new InvalidArgumentException('Session is missing required fields');
        }

        foreach (self::CODEIGNITER_SESSION_REQUIRED_FIELDS as $sessionKey) {
            if (empty($sessionData[$sessionKey])) {
                throw new InvalidArgumentException('Session is missing required fields');
            }
        }

        if (!in_array(
            $sessionData[self::SESSION_KEY_BUSINESS_TYPE],
            [
                self::USER_TYPE_APP_ADMIN,
                self::USER_TYPE_CARRIER,
                self::USER_TYPE_CUSTOMER,
                self::USER_TYPE_USER,
                self::USER_TYPE_COUNTRY_ADMIN
            ],
            true
        )) {
            throw new UserTypeInvalidException();
        }

        return new self($sessionData);
    }

    public function customers(): array
    {
        $customerIds = [];

        if (!empty($this->customer)) {
            $customerIds[] = $this->customer;
        }

        if (!empty($this->subCustomers) && is_array($this->subCustomers)) {
            $customerIds = array_merge($customerIds, $this->subCustomers);
        }

        return $customerIds;
    }

    public function user()
    {
        return $this->user;
    }

    public function userType(): string
    {
        return $this->businessType;
    }
}
