<?php

namespace App\DriverCollection;

use App\ApiRequest\DataPreprocessing\BadRequestException;
use App\ApiRequest\DataPreprocessing\TimezoneNotSpecifiedException;
use App\EventsQueue\AbstractEventController;
use App\Generators\RandomOrderSequence;
use Exception;
use App\Generators\RandomString;

class DriverCollection
{

    /**
     * @var AbstractEventController
     */
    private $CI;

    public function __construct(AbstractEventController $CI)
    {
        $this->CI = $CI;
    }

    /**
     * @throws BadRequestException
     * @throws TimezoneNotSpecifiedException
     * @throws Exception
     */
    public function create(array $data, $driver_id, string $user_id, string $imei): array
    {
        [$pickupAddressLine1, $plat, $plng, $pickupPin, $pickupCountry, $pickupCity] = $this->processAddressData(
            $data['pickup']
        );
        [$dropAddressLine1, $dlat, $dlng, $dropPin, $dropCountry, $dropCity] = $this->processAddressData(
            $data['destination']
        );

        if (empty($pickupAddressLine1) || empty($dropAddressLine1)) {
            throw new BadRequestException("Please enter proper locations");
        }

        $pickup_partyid = $data['pickup_pary_id'] ?? "";
        $drop_partyid = $data['delivery_pary_id'] ?? "";
        $cust_id = $data['customer_id'] ?? "";

        if ($pickup_partyid == "") {
            $pickup_partyid = $cust_id;
        }
        if ($drop_partyid == "") {
            $drop_partyid = $cust_id;
        }

        $curdt = date('Y-m-d H:i:s');
        $pickupDate = $curdt;
        $dropDate = $curdt;

        $minutesTime = NULL;
        $distanceKM = NULL;
        if($plat != NULL && $plng !=NULL && $dlat != NULL && $dlng !=NULL) {
            $chkdist = distancemetrixship($plat, $plng, $dlat, $dlng);
            if ($chkdist['duration'] > 0) {
                $duration = $chkdist['duration'];
                $tm = strtotime($curdt);
                $dropDate = date("Y-m-d H:i:s", $tm + $duration);
                $minutesTime = $chkdist['disttext'];
                $distanceKM = round((($chkdist['duration'] / 40) * 60), 2) . ' min';
            }
        }
        $vendor_id = "";
        $vehicle_id = "";
        $quantity = 0;
        $valueOfGoods = 0;
        $shipmentOption = "LTL";

        $uoms = $data['uoms'] ?? [];
        $references = $data['references'] ?? [];

        foreach ($uoms as $cnt) {
            $cnt = (array)$cnt;
            if ($cnt['value'] > 0) {
                $quantity += $cnt['value'];
            }
        }
        $weight = $volume = 1;
        $custinfo = $this->CI->db->select("name,phone,code,location,email_id,company_code,branch_code")->get_where(
            "tb_customers",
            ["id" => $cust_id],
            1,
            0
        );
        if (!in_array(0, [$pickup_partyid, $drop_partyid])) {
            $partyNames = $this->CI->common->gettbldata("id IN ($pickup_partyid, $drop_partyid)", "id,name", "tbl_party_master", 0, 0);
            foreach ($partyNames as $eachRow) {
                if ($eachRow['id'] == $pickup_partyid) {
                    $pickupName = $eachRow["name"];
                } else {
                    $dropName = $eachRow["name"];
                }
            }
        }
        $cmpname = $custinfo->row()->name;
        $consigneeName = $custinfo->row()->name;
        $consigneeCode = $custinfo->row()->code;
        $customer_area = $custinfo->row()->location;
        $consigneeEmail = $custinfo->row()->email_id;
        $consigneePhone = $custinfo->row()->phone;
        $company_code = $custinfo->row()->company_code;
        $branchCode = $custinfo->row()->branch_code;

        $vendorchk = $this->CI->db->select("vehicle_id")->get_where(
            "tbl_assigned_drivers",
            ["driver_id" => $driver_id, "status" => 1],
            1,
            0
        );
        if ($vendorchk->num_rows() > 0) {
            [$vehicle_id, $vendor_id] = $this->extractCarrierAndVehicle($vendorchk, $vendor_id);
        } else {
            $vendorchk = $this->CI->db->select("vehicle_id")->get_where(
                "tb_vehicles_drivers",
                ["driver_id" => $driver_id, "status" => 1],
                1,
                0
            );
            if ($vendorchk->num_rows() > 0) {
                [$vehicle_id, $vendor_id] = $this->extractCarrierAndVehicle($vendorchk, $vendor_id);
            }
        }

        if ($vendor_id == "" || $vehicle_id == "") {
            throw new BadRequestException("No vehicle assigned to you");
        }

        $vehicle_type = "Truck";
        $vehinfo = $this->CI->db->query(
            "SELECT t.trucktype FROM tb_trucktypes t,tb_trucks_data d WHERE t.id=d.truck_type AND d.id=? LIMIT 1",
            [$vehicle_id]
        );
        if ($vehinfo->num_rows() > 0) {
            $vehicle_type = $vehinfo->row()->trucktype;
        }

        $user = $this->CI->db->select("id,country_code")
            ->get_where("tb_users", ["id" => $user_id], 1, 0)
            ->row_array();
        $tz = $this->getUserTimeZone($user['country_code']);
        $hrs = $tz['hrs'];
        $country_code = $tz['phone_code'];
        $product = $branch_code = $freight_term = $freight_termname = $logicalreceiver = $physicalreceiver = $physicalsender = $logicalsender = "";
        $modeoftransport = 1;
        $rcentship = $this->CI->db->query(
            "SELECT shipid,domainname,company_code,branch_code,product,freight_term,freight_termname,logicalreceiver,
                    physicalreceiver,physicalsender,logicalsender,modeoftransport
                FROM tb_shipments
                WHERE company_code = ?
                ORDER BY id DESC LIMIT 1" , [$company_code]
        );
        if ($rcentship->num_rows() > 0) {
            $product = $rcentship->row()->product;
            $branch_code = $rcentship->row()->branch_code;
            $freight_term = $rcentship->row()->freight_term;
            $freight_termname = $rcentship->row()->freight_termname;
            $logicalreceiver = $rcentship->row()->logicalreceiver;
            $physicalreceiver = $rcentship->row()->physicalreceiver;
            $physicalsender = $rcentship->row()->physicalsender;
            $logicalsender = $rcentship->row()->logicalsender;
            $modeoftransport = $rcentship->row()->modeoftransport;
        }
        [$ord, $shiptxn] = RandomOrderSequence::generate($this->CI->db, $user_id, $country_code);

        $ttansmode = "Full Truck Load";
        $chkshmt = $this->CI->db->select("id")->get_where("tb_shipments", ["shipid" => $shiptxn], 1, 0);
        if ($chkshmt->num_rows() == 0) {
            $shipdatains = [
                'shipid' => $shiptxn,
                'txnid' => $shiptxn,
                'trucktype' => $ttansmode,
                'product' => $product,
                'pickupcnt' => 1,
                'dropcnt' => 1,
                'unitspec' => 1,
                'insertusr' => $cust_id,
                'carrier' => $vendor_id,
                'insertuserdate' => $curdt,
                'enddate' => $dropDate,
                'insdate' => $curdt,
                'upddate' => $dropDate,
                'reason' => 'SHIPMENT',
                'purpose' => 'SEND INTEGRATION',
                'ship_object' => 'SHIPMENT',
                'logdate' => $curdt,
                'status' => 1,
                'createdon' => $curdt,
                'updatedon' => $curdt,
                'transport_mode' => $shipmentOption,
                'txncode' => $ord,
                'domainname' => $branch_code,
                'company_code' => $company_code,
                'branch_code' => $branch_code,
                'freight_term' => $freight_term,
                'freight_termname' => $freight_termname,
                'logicalreceiver' => $logicalreceiver,
                'physicalreceiver' => $physicalreceiver,
                'physicalsender' => $physicalsender,
                'logicalsender' => $logicalsender,
                'modeoftransport' => $modeoftransport
            ];
            $sship_id = $this->CI->common->insertTableData('tb_shipments', $shipdatains);
        } else {
            $sship_id = $chkshmt->row()->id;
        }

        if ($sship_id <= 0) {
            throw new BadRequestException("Shipment Already exists!");
        }

        $orddata = [
            'order_id' => $ord,
            'pickup_datetime' => $pickupDate,
            'pickup_endtime' => $pickupDate,
            'delivery_datetime' => $dropDate,
            'drop_endtime' => $dropDate,
            'pickup_company' => $pickupName ?: $cmpname,
            'delivery_company' => $dropName ?: $cmpname,
            'pickup_country' => $pickupCountry,
            'delivery_country' => $dropCountry,
            'pickup_city' => $pickupCity,
            'delivery_city' => $dropCity,
            'pickup_pincode' => $pickupPin,
            'delivery_pincode' => $dropPin,
            'pickup_address1' => $pickupAddressLine1,
            'delivery_address1' => $dropAddressLine1,
            'quantity' => $quantity,
            'weight' => $weight,
            'volume' => $volume,
            'goods_value' => $valueOfGoods,
            'transport_mode' => $shipmentOption,
            'dedicated_vehicle' => 0,
            'vehicle_type' => $vehicle_type,
            'user_id' => $user_id,
            'customer_id' => $cust_id,
            'vendor_id' => $vendor_id,
            'customer_name' => $consigneeName,
            'customer_code' => $consigneeCode,
            'customer_area' => $customer_area,
            'customer_phone' => $consigneePhone,
            'customer_email' => $consigneeEmail,
            'vendor_name' => "",
            'vendor_code' => "",
            'vendor_area' => "",
            'vendor_phone' => "",
            'vendor_email' => "",
            'plat' => $plat,
            'plng' => $plng,
            'dlat' => $dlat,
            'dlng' => $dlng,
            'product' => $product,
            'shipment_id' => $sship_id,
            'pickup_custid' => $cust_id,
            'pickup_partyid' => $pickup_partyid,
            'drop_custid' => $cust_id,
            'drop_partyid' => $drop_partyid,
            'status' => 2,
            'updatedon' => $curdt,
            "created_source" => 1,
            "shipmentid" => $shiptxn,
            'logicalreceiver' => $logicalreceiver,
            'physicalreceiver' => $physicalreceiver,
            'physicalsender' => $physicalsender,
            'logicalsender' => $logicalsender,
            'modeoftransport' => $modeoftransport,
            'company_code' => $company_code,
            'branch_code' => $branchCode
        ];

        $ord_id = $this->CI->common->insertTableData("tb_orders", $orddata);

        $this->createCargoDetails($uoms, $quantity, $weight, $volume, $user_id, $curdt, $ord_id);
        $this->createOrderReferences($references, $ord_id, $curdt);
        $this->createOrderParties($user_id, $ord_id, $pickup_partyid, $ord, $curdt, $drop_partyid, $shiptxn);

        $shipment = [
            'user_id' => $user_id,
            'stime' => date("H:i", strtotime($curdt)),
            'etime' => date("H:i", strtotime($dropDate)),
            'splace' => $pickupAddressLine1,
            'slat' => $plat,
            'slng' => $plng,
            'eplace' => $dropAddressLine1,
            'elat' => $dlat,
            'elng' => $dlng,
            'scity' => $pickupCity,
            'dcity' => $dropCity,
            'zone_id' => 1,
            'empshift_start' => date("H:i", strtotime($curdt)),
            'empshift_end' => date("H:i", strtotime($dropDate)),
            'trip_type' => 0,
            'startdate' => $curdt,
            'enddate' => $dropDate,
            'shipment_name' => 1,
            'shipmentid' => $shiptxn,
            'shipment_id' => $sship_id,
            'customer_id' => $cust_id,
            'transport_mode' => $shipmentOption,
            'vendor_id' => $vendor_id,
            'carrier_type' => 0,
            'txnid' => $ord,
            'weight' => $weight,
            'volume' => $volume,
            'units' => count($uoms),
            'domainname' => $branch_code,
            'vehicle_type' => $vehicle_type,
            'company_code' => $company_code,
            'branch_code' => $branch_code,
            'status' => 1,
            'created_on' => $curdt,
            'updated_on' => $curdt
        ];
        $ship_id = $this->CI->common->insertTableData("tb_shifts", $shipment);
        /*update order table*/
        $this->CI->common->updatetbledata("tb_orders", ["shift_id" => $ship_id], ["id" => $ord_id]);

        $stopleg1_id = "";
        $stopleg2_id = "";

        $chkstoplegs = $this->CI->db->select("id")->get_where(
            "tb_shiporder_stops",
            [
                'stopname' => $pickupAddressLine1,
                'stopcity' => $pickupCity,
                'shipment_id' => $ship_id,
                'txncode' => $ord,
                'stoptype' => 'P'
            ],
            1,
            0
        );
        if ($chkstoplegs->num_rows() == 0) {
            $shiporder = [
                'stopname' => $pickupAddressLine1,
                'plat' => $plat,
                'plng' => $plng,
                'stopcity' => $pickupCity,
                'address' => $pickupAddressLine1,
                'stoptype' => 'P',
                'stopstatus' => 0,
                'shipment_id' => $ship_id,
                'ordernumber' => 1,
                'startdate' => $curdt,
                'enddate' => $curdt,
                'weight' => $weight,
                'volume' => $volume,
                'shipmentstopid' => 0,
                'serviceprovider' => "LMV",
                'ship_units' => count($uoms),
                'txncode' => $ord,
                'status' => 1,
                'created_on' => $curdt,
                'updated_on' => $curdt
            ];
            $stopleg1_id = $this->CI->common->insertTableData("tb_shiporder_stops", $shiporder);
        }
        $chkstoplegs1 = $this->CI->db->select("id")->get_where(
            "tb_shiporder_stops",
            [
                'stopname' => $dropAddressLine1,
                'stopcity' => $dropCity,
                'shipment_id' => $ship_id,
                'txncode' => $ord,
                'stoptype' => 'D'
            ],
            1,
            0
        );
        if ($chkstoplegs1->num_rows() == 0) {
            $shiporder = [
                'stopname' => $dropAddressLine1,
                'plat' => $dlat,
                'plng' => $dlng,
                'stopcity' => $dropCity,
                'address' => $dropAddressLine1,
                'stoptype' => 'D',
                'stopstatus' => 0,
                'shipment_id' => $ship_id,
                'ordernumber' => 2,
                'startdate' => $dropDate,
                'enddate' => $dropDate,
                'weight' => $weight,
                'volume' => $volume,
                'shipmentstopid' => 0,
                'serviceprovider' => "LMV",
                'ship_units' => count($uoms),
                'txncode' => $ord,
                'status' => 1,
                'created_on' => $curdt,
                'updated_on' => $curdt
            ];
            $stopleg2_id = $this->CI->common->insertTableData("tb_shiporder_stops", $shiporder);
        }
        if ($stopleg1_id == "" || $stopleg2_id == "") {
            throw new BadRequestException("Source/Destination Locations Are Wrong");
        }

        $chkemp = $this->CI->db->select("id")->get_where(
            "tb_employee",
            [
                'assoc_id' => $ord,
                'order_id' => $ord,
                'shift_id' => $ship_id,
                'stop_id' => $stopleg1_id,
                'drop_stopid' => $stopleg2_id
            ],
            1,
            0
        );
        if ($chkemp->num_rows() == 0) {
            $stops = [
                'assoc_id' => $ord,
                'pickup' => $pickupAddressLine1,
                'plat' => $plat,
                'plng' => $plng,
                'drop' => $dropAddressLine1,
                'dlat' => $dlat,
                'dlng' => $dlng,
                'pickup_city' => $pickupCity,
                'drop_city' => $dropCity,
                'pickup_datetime' => $curdt,
                'drop_datetime' => $dropDate,
                'name' => 'Goods',
                'phone' => "",
                'address' => $pickupAddressLine1,
                'user_id' => $user_id,
                'password' => RandomString::generate(10),
                'status' => 1,
                'createdon' => $curdt,
                'updatedon' => $curdt,
                'material_id' => 0,
                'capacity' => $quantity,
                'information' => "",
                'shipment_weight' => $weight,
                'shipment_volume' => $volume,
                'ship_type' => 'P',
                'customer_id' => $cust_id,
                'vendor_id' => $vendor_id,
                'shipment_id' => $sship_id,
                'startdate' => $curdt,
                'enddate' => $dropDate,
                'shift_id' => $ship_id,
                'stop_order' => 1,
                'drop_order' => 2,
                'basic_stop' => 1,
                'stop_id' => $stopleg1_id,
                'drop_stopid' => $stopleg2_id,
                'order_id' => $ord,
                'pkgitemid' => 'Goods',
                'no_of_pkgs' => count($uoms),
                'domainname' => $branch_code,
                "accepted" => 1
            ];
            $this->CI->common->insertTableData("tb_employee", $stops);
        }
        /*assign to carrier*/
        $veharr = [
            "user_id" => $user_id,
            "shft_id" => $ship_id,
            "carrier_id" => $vendor_id,
            "vehicle_id" => $vehicle_id
        ];
        $chkveh = $this->CI->db->select("id")->get_where("tb_shft_veh", $veharr, 1, 0);
        if ($chkveh->num_rows() == 0) {
            $insveh = [
                "user_id" => $user_id,
                "shft_id" => $ship_id,
                "carrier_id" => $vendor_id,
                "vehicle_id" => $vehicle_id,
                "register_number" => "",
                "status" => 1,
                'created_on' => $curdt,
                'updated_on' => $curdt
            ];
            $shipvehid = $this->CI->common->insertTableData("tb_shft_veh", $insveh);
        } else {
            $shipvehid = $chkveh->row()->id;
            $updveh = ["status" => 1, 'updated_on' => $curdt];
            $this->CI->db->where($veharr)->update("tb_shft_veh", $updveh);
        }

        $this->defineShipmentOrder($user_id, $shipvehid, $ship_id, $curdt);

        $tripId = $this->acceptDriverCreatedCollection(
            $ship_id,
            $curdt,
            $vehicle_id,
            $driver_id,
            $imei,
            $pickupAddressLine1,
            $ord_id,
            $plat,
            $plng
        );

        $result['orderSequence'] = $ord;
        $result['stopLegId'] = $stopleg1_id;
        $result['hrs'] = $hrs;

        $result['doc_types'] = $this->CI->db->select('id,type_name')
            ->get_where('tb_document_types', ['status' => 1])
            ->result_array();

        $result['trips'] = [
            'id' => $tripId,
            'shift_id' => $ship_id,
            'user_id' => $user_id,
            'driver_id' => $driver_id,
            'vehicle_id' => $vehicle_id,
            'shipweight' => $quantity,
            'imei' => $imei,
            'stime' => date('H:i', strtotime($curdt)),
            'etime' => date('H:i', strtotime($dropDate)),
            'trip_type' => 0,
            'saddress' => $pickupAddressLine1,
            'daddress' => $dropAddressLine1,
            'splace' => $pickupCity,
            'eplace' => $dropCity,
            'scity' => $pickupCity,
            'dcity' => $dropCity,
            'slat' => $plat ?? "",
            'slng' => $plng ?? "",
            'elat' => $dlat ?? "",
            'elng' => $dlng ?? "",
            'shipmentid' => $shiptxn,
            'shift_veh_id' => $shipvehid,
            'astatus' => 0,
            'sno' => 1,
            'distance' => $distanceKM,
            'duration' => $minutesTime,
            'startdate' => strtotime($curdt),
            'enddate' => strtotime($dropDate),
            'pod' => '',
            'signature' => ''
        ];

        return ['data' => $result];
    }

    private function processAddressData(array $address): array
    {
        $addressLine1 = $address['formatted_address'] ?? null;
        $latitude = $address['geometry']['location']['lat'] ?? null;
        $longitude = $address['geometry']['location']['lng'] ?? null;
        $address_components = $address['address_components'] ?? null;
        $area = $pickupCity = $pickupCountry = $pickupPin = '';
        foreach ($address_components as $res1) {
            if ($res1['types'][0] == "locality") {
                $area = $res1['long_name'];
            }
            if ($res1['types'][0] == "administrative_area_level_2") {
                $pickupCity = $res1['long_name'];
            }
            if ($res1['types'][0] == "country") {
                $pickupCountry = $res1['long_name'];
            }
            if ($res1['types'][0] == "postal_code") {
                $pickupPin = $res1['long_name'];
            }
        }
        if ($pickupCity == '') {
            $pickupCity = $area;
        }

        return [$addressLine1, $latitude, $longitude, $pickupPin, $pickupCountry, $pickupCity];
    }

    private function extractCarrierAndVehicle(object $vendorchk, $vendor_id): array
    {
        $vehicle_id = $vendorchk->row()->vehicle_id;
        $tbltrucks = $this->CI->db->select("vendor_id")->get_where(
            "tb_vendor_vehicles",
            ["vehicle_id" => $vehicle_id, "status" => 1],
            1,
            0
        );

        if ($tbltrucks->num_rows() > 0) {
            $vendor_id = $tbltrucks->row()->vendor_id;
        } else {
            $tbltrucks = $this->CI->db->select("vendor_id")->get_where(
                "tb_trucks_data",
                ["id" => $vehicle_id, "vendor_id >" => 0, "status" => 1],
                1,
                0
            );

            if ($tbltrucks->num_rows() > 0) {
                $vendor_id = $tbltrucks->row()->vendor_id;
            }
        }
        return [$vehicle_id, $vendor_id];
    }

    private function getUserTimeZone($countryCode): array
    {
        $getqry = $this->CI->db->query(
            "SELECT country_code, cntry_timezone, cntry_hrs, currency, phone_code
                    FROM tbl_country_master
                    WHERE (country_code = ? OR country_name = ?)
                      AND status = 1
                    LIMIT 1",
            [$countryCode, $countryCode]
        );
        $res = [
            "currency" => "SGD",
            "timezone" => "Asia/Singapore",
            "hrs" => "+08.00",
            "country" => "SG",
            "phone_code" => "65"
        ];
        if ($getqry->num_rows() > 0) {
            $res = [
                "currency" => $getqry->row()->currency,
                "timezone" => $getqry->row()->cntry_timezone,
                "hrs" => $getqry->row()->cntry_hrs,
                'country' => $getqry->row()->country_code,
                'phone_code' => $getqry->row()->phone_code
            ];
        }
        return $res;
    }

    private function createCargoDetails(
        $uoms,
        $quantity,
        int $weight,
        $volume,
        $user_id,
        $curdt,
        $ord_id
    ): void {
        $stackable = 0;
        if (count($uoms) > 0) {
            foreach ($uoms as $ct) {
                $ct = (array)$ct;
                if ($ct['value'] != "") {
                    $cargo_whr = [
                        'cargo_type' => $ct['name'],
                        'quantity' => $quantity,
                        'length' => $weight,
                        'length_unit' => 'M',
                        'width' => $weight,
                        'width_unit' => 'M',
                        'height' => $weight,
                        'height_unit' => 'M',
                        'weight' => $weight,
                        'weight_unit' => 'Kg',
                        'volume' => $volume,
                        'volume_unit' => 'cbm',
                        'stackable' => $stackable
                    ];
                    $cargo_ins = [
                        'cargo_type' => $ct['name'],
                        'goods_description' => $ct['name'],
                        'quantity' => $quantity,
                        'length' => $weight,
                        'length_unit' => 'M',
                        'width' => $weight,
                        'width_unit' => 'M',
                        'height' => $weight,
                        'height_unit' => 'M',
                        'weight' => $weight,
                        'weight_unit' => 'Kg',
                        'volume' => $volume,
                        'volume_unit' => 'cbm',
                        'stackable' => $stackable,
                        'createdby' => $user_id,
                        'createdon' => $curdt
                    ];
                    $checkparty = $this->CI->db->select("id")->get_where("tb_cargo_details", $cargo_whr, 1, 0);
                    if ($checkparty->num_rows() == 0) {
                        $cargo_id = $this->CI->common->insertTableData("tb_cargo_details", $cargo_ins);
                    } else {
                        $cargo_id = $checkparty->row()->id;
                    }
                    $ins = ['cargo_type' => $ct['name'], 'order_id' => $ord_id, 'status' => 1, 'createdon' => $curdt];
                    $this->CI->common->insertTableData("tb_order_cargotypes", $ins);
                    $cargodata = [
                        'order_id' => $ord_id,
                        'cargo_id' => $cargo_id,
                        'handling_unit' => $ct['id'],
                        'length' => $weight,
                        'width' => $weight,
                        'height' => $weight,
                        'weight' => $weight,
                        'quantity' => $quantity,
                        'status' => 1,
                        'createdon' => $curdt
                    ];
                    $this->CI->common->insertTableData("tb_order_cargodetails", $cargodata);
                }
            }
        }
    }

    private function createOrderReferences($references, $ord_id, $curdt): void
    {
        if (count($references) > 0) {
            foreach ($references as $ref) {
                if ($ref != null) {
                    $ref = (array)$ref;
                    if ($ref['value'] != "" && $ref['reference_id'] != "") {
                        $ins = [
                            'order_id' => $ord_id,
                            'reference_id' => $ref['reference_id'],
                            'ref_value' => $ref['value'],
                            'status' => 1,
                            'createdon' => $curdt
                        ];
                        $this->CI->common->insertTableData("tb_order_references", $ins);
                    }
                }
            }
        }
    }

    /**
     * @throws BadRequestException
     */
    private function createOrderParties(
        $user_id,
        $ord_id,
        $pickup_partyid,
        string $ord,
        $curdt,
        $drop_partyid,
        string $shiptxn
    ): void {
        $shipperparty_id = 1;
        $consignee_party_id = 3;
        $chkparty = $this->CI->common->gettblrowdata(
            ["name" => "Shipper", "user_id" => $user_id, "status" => 1],
            "id",
            "tbl_party_types",
            0,
            0
        );
        if (count($chkparty) > 0) {
            $shipperparty_id = $chkparty['id'];
        }
        $chkparty = $this->CI->common->gettblrowdata(
            ["name" => "Consignee", "user_id" => $user_id, "status" => 1],
            "id",
            "tbl_party_types",
            0,
            0
        );
        if (count($chkparty) > 0) {
            $consignee_party_id = $chkparty['id'];
        }
        $checkparty = $this->CI->db->select("id")->get_where(
            "tb_order_parties",
            ['order_id' => $ord_id, 'party_id' => $pickup_partyid, 'party_type' => $shipperparty_id],
            1,
            0
        );
        if ($checkparty->num_rows() == 0) {
            $partyarr = [
                'order_id' => $ord_id,
                'order_number' => $ord,
                'party_id' => $pickup_partyid,
                'party_type' => $shipperparty_id,
                'status' => 1,
                'createdon' => $curdt
            ];
            $this->CI->common->insertTableData("tb_order_parties", $partyarr);
        }
        $checkparty = $this->CI->db->select("id")->get_where(
            "tb_order_parties",
            ['order_id' => $ord_id, 'party_id' => $drop_partyid, 'party_type' => $consignee_party_id],
            1,
            0
        );
        if ($checkparty->num_rows() == 0) {
            $partyarr = [
                'order_id' => $ord_id,
                'order_number' => $ord,
                'party_id' => $drop_partyid,
                'party_type' => $consignee_party_id,
                'status' => 1,
                'createdon' => $curdt
            ];
            $this->CI->common->insertTableData("tb_order_parties", $partyarr);
        }
        $checkshipment = $this->CI->db->select("id")->get_where(
            "tb_shifts",
            ['shipmentid' => $shiptxn, "status" => 1],
            1,
            0
        );
        if ($checkshipment->num_rows() != 0) {
            throw new BadRequestException("Already exists!");
        }
    }

    private function defineShipmentOrder(
        $user_id,
        $shipvehid,
        $ship_id,
        $curdt
    ): void {
        $veharr = ["user_id" => $user_id, "shft_veh_id" => $shipvehid];
        $chkveh = $this->CI->db->select("id")->get_where("tb_shft_veh_emp", $veharr, 1, 0);
        if ($chkveh->num_rows() == 0) {
            $getemp = $this->CI->db->select("id,pickup_datetime,drop_datetime")->get_where(
                "tb_employee",
                ["shift_id" => $ship_id]
            );
            if ($getemp->num_rows() > 0) {
                $pri = 1;
                foreach ($getemp->result() as $gt) {
                    $insveh1 = [
                        "user_id" => $user_id,
                        "shft_veh_id" => $shipvehid,
                        "emp_id" => $gt->id,
                        "priority" => $pri,
                        "pickup_time" => $gt->pickup_datetime,
                        'created_on' => $curdt,
                        'updated_on' => $curdt,
                        "status" => 1,
                        "drop_time" => $gt->drop_datetime
                    ];
                    $this->CI->common->insertTableData("tb_shft_veh_emp", $insveh1);
                    $pri++;
                }
            }
        } else {
            $getemp = $this->CI->db->select("id,pickup_datetime,drop_datetime")->get_where(
                "tb_employee",
                ["shift_id" => $ship_id]
            );
            if ($getemp->num_rows() > 0) {
                $pri = 1;
                foreach ($getemp->result() as $gt) {
                    $veharr = ["user_id" => $user_id, "shft_veh_id" => $shipvehid, "emp_id" => $gt->id];
                    $chkveh = $this->CI->db->select("id")->get_where("tb_shft_veh_emp", $veharr, 1, 0);
                    if ($chkveh->num_rows() == 0) {
                        $insveh1 = [
                            "user_id" => $user_id,
                            "shft_veh_id" => $shipvehid,
                            "emp_id" => $gt->id,
                            "priority" => $pri,
                            "pickup_time" => $gt->pickup_datetime,
                            'created_on' => $curdt,
                            'updated_on' => $curdt,
                            "status" => 1,
                            "drop_time" => $gt->drop_datetime
                        ];
                        $this->CI->common->insertTableData("tb_shft_veh_emp", $insveh1);
                    } else {
                        $veharr11 = ["id" => $chkveh->row()->id];
                        $updveh1 = [
                            "priority" => $pri,
                            "pickup_time" => $gt->pickup_datetime,
                            'updated_on' => $curdt,
                            "status" => 1,
                            "drop_time" => $gt->drop_datetime
                        ];
                        $this->CI->db->where($veharr11)->update("tb_shft_veh_emp", $updveh1);
                    }
                    $pri++;
                }
            }
        }
    }

    private function acceptDriverCreatedCollection(
        $ship_id,
        $curdt,
        $vehicle_id,
        $driver_id,
        $imei,
        $pickupAddressLine1,
        $ord_id,
        $plat,
        $plng
    ) {
        $chk = $this->CI->db->select("id")->get_where(
            "tb_stop_status",
            ["shipment_id" => $ship_id, "status_id" => 9],
            1,
            0
        );
        if ($chk->num_rows() == 0) {
            $ins = [
                "shipment_id" => $ship_id,
                "stop_id" => 0,
                "stop_detail_id" => 0,
                "stop_type" => "",
                "trip_id" => 0,
                "status_id" => 9,
                "status_code" => "0100",
                "status" => 1,
                "reason" => "Coming from E-Booking",
                "createdon" => $curdt
            ];
            $this->CI->db->insert("tb_stop_status", $ins);
        }
        /*accept shipment*/
        $chktrip = $this->CI->db->select("id")->get_where(
            "tb_trips",
            ["shift_id" => $ship_id, "vehicle_id" => $vehicle_id, "driver_id" => $driver_id],
            1,
            0
        );
        if ($chktrip->num_rows() == 0) {
            $tripsdata = [
                "shift_id" => $ship_id,
                "vehicle_id" => $vehicle_id,
                "driver_id" => $driver_id,
                "stime" => $curdt,
                "start_imei" => $imei,
                "splace" => $pickupAddressLine1,
                "created_on" => $curdt,
                "updated_on" => $curdt,
                "status" => 1
            ];
            $tid = $this->CI->common->insertTableData("tb_trips", $tripsdata);
        } else {
            $tid = $chktrip->row()->id;
        }

        $this->CI->common->updatetbledata("tb_orders",
            ['trip_id' => $tid],
            ['id' => $ord_id]
        );

        $chqry = $this->CI->db->select("id")->get_where(
            "tb_stop_status",
            ["shipment_id" => $ship_id, "stop_id" => 0, "stop_detail_id" => 0, "trip_id" => $tid, "status_id" => 10],
            1,
            0
        );
        if ($chqry->num_rows() == 0) {
            $insarry = [
                "shipment_id" => $ship_id,
                "stop_id" => 0,
                "stop_detail_id" => 0,
                "stop_type" => '',
                "trip_id" => $tid,
                "status_id" => 10,
                "latitude" => $plat,
                "longitude" => $plng,
                "status" => 1,
                "status_code" => "0212",
                "reason" => "From Mobile",
                "vehicle_id" => $vehicle_id,
                "driver_id" => $driver_id,
                "createdon" => $curdt
            ];
            $this->CI->db->insert("tb_stop_status", $insarry);
        }

        return $tid;
    }
}
