<?php

namespace App\EventsQueue;

use CI_DB_mysqli_driver;

final class EventQueueManager
{
    /**
     * @var CI_DB_mysqli_driver
     */
    private $db;

    public function __construct(CI_DB_mysqli_driver $db)
    {
        $this->db = $db;
    }

    public function startTransaction()
    {
        $this->db->trans_start();
    }

    public function completeTransaction()
    {
        $this->db->trans_complete();
    }

    public function markAsComplete(string $id)
    {
        $this->db->where(['id' => $id])->update(
            EventQueue::TABLE,
            ['status' => EventStatus::COMPLETED]
        );
    }

    public function getNotStartedEvents(string $eventType)
    {
        $query = $this->db->select('`id`, `event_name`, `payload`')
            ->from(EventQueue::TABLE)
            ->where(['status' => EventStatus::NOT_STARTED])
            ->like('event_name', $eventType, 'after')
            ->order_by('created_on', 'ASC')
            ->limit(500);

        return $query->get();
    }

    public function getNotFinishedEvents(string $eventType): object
    {
        $query = $this->db->select('`id`, `event_name`, `payload`')
            ->from(EventQueue::TABLE)
            ->where_in('status', [EventStatus::STARTED, EventStatus::PENDING])
            ->where('updated_on < DATE_SUB(NOW(), INTERVAL 20 MINUTE)')
            ->like('event_name', $eventType, 'after')
            ->order_by('created_on', 'ASC')
            ->limit(500);

        return $query->get();
    }

    public function markAsStarted(string $id)
    {
        $this->db->where(['id' => $id])
            ->update(EventQueue::TABLE, ['status' => EventStatus::STARTED]);
    }

    public function markAsPending(array $ids)
    {
        $this->db->where_in('id', $ids)
            ->update(EventQueue::TABLE, ['status' => EventStatus::PENDING]);
    }

    public function markAsNotStarted(string $id)
    {
        $this->db->where(['id' => $id])->update(
            EventQueue::TABLE,
            ['status' => EventStatus::NOT_STARTED]
        );
    }

    public function markAsFailed(string $id, string $reason = null)
    {
        $this->db->where(['id' => $id])
            ->update(
                EventQueue::TABLE,
                [
                    'status' => EventStatus::FAILED,
                    'result' => $reason
                ]
            );
    }

    public function setTimeTaken(string $id, string $timeTakenByJob)
    {
        $this->db->where(['id' => $id])->update(
            EventQueue::TABLE,
            ['time_taken' => $timeTakenByJob]
        );
    }
}
