<?php

namespace App\Mobile\V1\Trips\Actions;

use App\ApiRequest\DataPreprocessing\BadRequestException;
use App\Mobile\V1\Trips\Actions\Model\Action;
use App\Mobile\V1\Trips\Actions\Model\ActionDetails;
use App\Mobile\V1\Trips\Actions\Model\ActionDocuments;
use App\Mobile\V1\Trips\Actions\Model\ShipmentStatus;
use App\Mobile\V1\Trips\Actions\Request\ActionRequestValidation;
use App\Mobile\V1\Trips\Exception\ShipmentIdsNotProvidedForAuthorizationException;
use App\Mobile\V1\Trips\Exception\StopTypeNotValidException;
use App\Mobile\V1\Trips\Exception\ShipmentDoesNotBelongToAuthorizationException;
use App\Mobile\V1\Trips\Model\Merchandise;

/**
 * Process Request and return Response.
 * POST method only
 */
class ActionRequest extends ActionDetails
{
    const METHOD_CREATE = 'create';
    const METHOD_UPDATE = 'update';

    /** @var Service\ShipmentStatus */
    private $shipmentStatus;

    /** @var DecodeVirtualId */
    private $decodeVirtualId;

    public function __construct(\CI_DB_mysqli_driver $db)
    {
        parent::__construct($db);
        $this->shipmentStatus = new Service\ShipmentStatus(new ShipmentStatus());
        $this->decodeVirtualId = new DecodeVirtualId();
    }

    /**
     * @param string $method
     * @param array $request
     * @param array $shipmentIds
     * @param string $curtz
     * @return array
     * @throws BadRequestException
     * @throws ShipmentDoesNotBelongToAuthorizationException
     * @throws ShipmentIdsNotProvidedForAuthorizationException
     * @throws StopTypeNotValidException
     */
    public function request(string $method, array $request, array $shipmentIds, string $curtz)
    {
        $id = $request['id'] ?? 0;
        $merchandise = [];
        $additionalValues = [];

        $this->validateRequest($method, $request);
        $actionIdentificators = $this->decodeVirtualId->decodeVirtualId($request);

        $request['trip_id'] = $actionIdentificators['trip_id'] ?? 0;
        $request['stop_id'] = $actionIdentificators['stop_id'] ?? 0;
        $request['type'] = $actionIdentificators['type'] ?? '';
        $request['order_id'] = $actionIdentificators['order_id'] ?? null;

        if ($method == ActionRequest::METHOD_UPDATE) {
            $id = $this->update($request, $id, $shipmentIds, $additionalValues);
        } else {
            $id = $this->create($request, $shipmentIds, $additionalValues);
        }

        if ($id) {
            $documents = $request['details']['documents'] ?? [];
            $actionDocuments = new ActionDocuments($this->db);
            $actionDocuments->add($request, $additionalValues, $documents);
        }

        if (in_array($request['type'], Merchandise::getValidActionTypes())) {
            $merchandise = Merchandise::getItems($this->db, $additionalValues['order_id']);
        }

        return $this->findByActionId($id ?? 0, $merchandise, $curtz);
    }

    /**
     * @param string $date
     * @return string
     */
    public static function formatDateForResponse(string $date) : string
    {
        return preg_replace('/(.*):/', '\1', date('c', strtotime($date)));
    }

    /**
     * @param string $date
     * @return bool|string
     */
    public static function formatDateForInsert(string $date)
    {
        return date('Y-m-d H:i:s', strtotime($date));
    }

    /**
     * @param array $request
     * @param array $shipmentIds
     * @param array $additionalValues
     * @return int
     * @throws BadRequestException
     * @throws ShipmentDoesNotBelongToAuthorizationException
     * @throws ShipmentIdsNotProvidedForAuthorizationException
     * @throws StopTypeNotValidException
     */
    private function create(array $request, array $shipmentIds, array &$additionalValues) : int
    {
        $action = new Action($this->db);
        $additionalValues = $action->getAdditionalDataForUpdate($request['stop_id'], $request['trip_id']);

        $this->validateShipment($additionalValues, $shipmentIds);
        $this->validateStopType($additionalValues, $request);

        $statusData = $this->shipmentStatus->getStatusData($request['type'], $additionalValues['stop_type']);

        $actionId = $this->validateAction(
            $request['stop_id'],
            $request['trip_id'],
            $additionalValues['order_id'],
            $statusData['status_id'] ?? null,
            $statusData['status_code'] ?? null
        );

        if (!$actionId) {
            $id = $action->insert($request, $additionalValues);

            //insert second record for discrepancy
            if (isset($request['details']['discrepancy']) && ($request['details']['discrepancy'] != '' && $request['details']['discrepancy'] !== null)) {
                $id = $action->insert($request, $additionalValues, true);
            }

            //inject second record if required
            $this->shipmentStatus->injectStatus($id, $request['type'], $additionalValues['stop_type']);
        } else {
            $action = new Action($this->db, $actionId);
            $request['id'] = $actionId;

            if (isset($request['details']['discrepancy']) && ($request['details']['discrepancy'] != '' && $request['details']['discrepancy'] !== null)) {
                $action->update($request, $additionalValues, true);
            } else {
                $action->update($request, $additionalValues);
            }

            $id = $action->getId();
        }

        return $id;
    }

    /**
     * @param array $request
     * @param int $id
     * @param array $shipmentIds
     * @param array $additionalValues
     * @return int
     * @throws BadRequestException
     * @throws ShipmentDoesNotBelongToAuthorizationException
     * @throws ShipmentIdsNotProvidedForAuthorizationException
     * @throws StopTypeNotValidException
     */
    private function update(array $request, int $id, array $shipmentIds, array &$additionalValues) : int
    {
        $action = new Action($this->db, $id);
        $additionalValues = $action->getAdditionalDataForUpdate($request['stop_id'], $request['trip_id']);

        $this->validateShipment($additionalValues, $shipmentIds);
        $this->validateStopType($additionalValues, $request);

        if (isset($request['details']['discrepancy']) && ($request['details']['discrepancy'] != '' && $request['details']['discrepancy'] !== null)) {
            $action->update($request, $additionalValues, true);
        } else {
            $action->update($request, $additionalValues);
        }

        return $action->getId();
    }

    /**
     * @param array $additionalValues
     * @param array $request
     * @throws StopTypeNotValidException
     */
    private function validateStopType(array $additionalValues, array $request)
    {
        if (in_array($request['type'], ActionType::$stopTypeIdentifiers)
            && $request['type'] != ActionType::$stopTypeIdentifiers[$additionalValues['stop_type']]) {
            throw new StopTypeNotValidException('Invalid stop type');
        }
    }

    /**
     * @param array $additionalValues
     * @param array $shipmentIds
     * @throws ShipmentIdsNotProvidedForAuthorizationException
     * @throws ShipmentDoesNotBelongToAuthorizationException
     */
    private function validateShipment(array $additionalValues, array $shipmentIds)
    {
        $shipmentId = $additionalValues['shipment_id'] ?? '';

        if (!in_array($shipmentId, $shipmentIds)) {
            throw new ShipmentDoesNotBelongToAuthorizationException('Shipment not found in token.');
        } elseif (empty($shipmentIds)) {
            throw new ShipmentIdsNotProvidedForAuthorizationException('Shipment IDs are missing');
        }
    }

    /**
     * if not action exists, then create one and dependency
     * create and copy created record with new status.
     *
     * @param int $stopId
     * @param int $tripId
     * @param int $orderId
     * @param int $statusId
     * @param string $statusCode
     * @return int
     */
    private function validateAction(int $stopId, int $tripId, int $orderId, int $statusId, string $statusCode) : int
    {
        $sql = "
                SELECT id FROM tb_stop_status
                WHERE stop_id = ? and trip_id = ? and order_id = ? and status_id = ? and status_code = ? and status = 1";

        $result = $this->db->query($sql, [$stopId, $tripId, $orderId, $statusId, $statusCode])->result_array();

        return $result[0]['id'] ?? 0;
    }

    /**
     * @param string $method
     * @param array $request
     * @throws BadRequestException
     */
    private function validateRequest(string $method, array $request)
    {
        if (!is_array($request) || empty($request)) {
            throw new BadRequestException('Invalid request');
        }

        $originalRequest = $request;
        $requestValidation = new ActionRequestValidation();
        $missingFields = $requestValidation->missingRequiredFields($request, $method);

        if (!empty($missingFields)) {
            throw new BadRequestException('Some required fields are missing: ' . $requestValidation->formatValidatedFields($missingFields));
        }

        $invalidFields = $requestValidation->validateDataType($originalRequest, $method);

        if (!empty($invalidFields)) {
            throw new BadRequestException('Some fields are invalid: ' . $requestValidation->formatValidatedFields($invalidFields));
        }
    }
}
