<?php

namespace App\Mobile\V1\Trips\Actions\Model;

use App\ApiRequest\DataPreprocessing\BadRequestException;
use App\Mobile\V1\Trips\Actions\ActionDiscrepancy;
use App\Mobile\V1\Trips\Actions\ActionRequest;

/**
 * @class Action
 * Insert or Update Action
 */
class Action
{
    /**
     * @var object
     */
    private $db;

    /**
     * @var int
     */
    private $id;

    /** @var \App\Mobile\V1\Trips\Actions\Service\ShipmentStatus */
    private $shipmentStatus;

    /**
     * @param \CI_DB_mysqli_driver $db
     * @param int $id
     */
    public function __construct(\CI_DB_mysqli_driver $db, int $id = 0)
    {
        $this->db = $db;
        $this->id = $id;
        $this->shipmentStatus = new \App\Mobile\V1\Trips\Actions\Service\ShipmentStatus(new ShipmentStatus());
    }

    /**
     * @return int
     */
    public function getId(): int
    {
        return $this->id;
    }

    /**
     * @param array $request
     * @param array $additionalValues
     * @param bool $discrepancy
     * @return int
     */
    public function insert(array $request, array $additionalValues, bool $discrepancy = false): int
    {
        $sql = "
                INSERT INTO tb_stop_status
                SET
                    order_id = ?,
                    shipment_id = ?,
                    stop_id = ?,
                    stop_detail_id = ?,
                    stop_type = ?,
                    trip_id = ?,
                    status_id = ?,
                    latitude = ?,
                    longitude = ?,
                    loc_name = ?,
                    status_code = ?,
                    status_stage = ?,
                    status = ?,
                    reason = ?,
                    vehicle_id = ?,
                    driver_id = ?,
                    sentknlogin = ?,
                    createdon = ?,
                    updatedon = ?
                ";

        $this->db->query($sql, $this->setQueryParameters($request, $additionalValues, $discrepancy));

        return $this->db->insert_id();
    }

    /**
     * @param array $request
     * @param array $additionalValues
     * @param bool $discrepancy
     * @return int
     * @throws BadRequestException
     */
    public function update(array $request, array $additionalValues, bool $discrepancy = false): int
    {
        if (!isset($request['id'])) {
            throw new BadRequestException('Update Action ID was not provided for update');
        }

        $sql = "
                UPDATE tb_stop_status
                SET
                    order_id = ?,
                    shipment_id = ?,
                    stop_id = ?,
                    stop_detail_id = ?,
                    stop_type = ?,
                    trip_id = ?,
                    status_id = ?,
                    latitude = ?,
                    longitude = ?,
                    loc_name = ?,
                    status_code = ?,
                    status_stage = ?,
                    status = ?,
                    reason = ?,
                    vehicle_id = ?,
                    driver_id = ?,
                    sentknlogin = ?,
                    createdon = ?,
                    updatedon = ?
                WHERE id = ?";

        $this->db->query($sql, $this->setQueryParameters($request, $additionalValues, $discrepancy, true));

        return $this->db->affected_rows();
    }

    /**
     * @param int $stopId
     * @param int $tripId
     * @return array
     */
    public function getAdditionalDataForUpdate(int $stopId, int $tripId): array
    {
        $stopData = $this->getStopData($stopId);
        $additionalData = $this->getTripData($tripId);

        return array_replace($stopData, $additionalData);
    }

    /**
     * @param array $request
     * @param array $additionalValues
     * @param bool $discrepancy
     * @param bool $update
     * @return array
     */
    private function setQueryParameters(array $request, array $additionalValues, bool $discrepancy, bool $update = false) : array
    {
        $params = [];

        $typeIdentifier = $request['type'] ?? 0; //gateIn, pickup, delivery, gateOut
        $stopType = $additionalValues['stop_type'] ?? 'P'; //P or D
        $stopId = $request['stop_id'] ?? 0;
        $discrepancyIdentifier = $request['details']['discrepancy'] ?? '';

        if ($discrepancy) {
            $statusData = $this->getDiscrepancyState($discrepancyIdentifier);
        } else {
            $statusData = $this->shipmentStatus->getStatusData($typeIdentifier, $stopType);
        }

        $statusId = $statusData['status_id'] ?? 0;
        $statusCode = $statusData['status_code'] ?? '0';
        $statusStage = $statusData['status_stage'] ?? '';
        $status = $statusData['status'] ?? 0;

        $currentDate = date("Y-m-d H:i:s");
        $date = $request['details']['date'] ?? $currentDate;
        $dateModified = $request['details']['date_modified'] ?? $currentDate;

        $params[] = $additionalValues['order_id'] ?? 0;
        $params[] = $additionalValues['shift_id'] ?? 0;
        $params[] = $stopId;
        $params[] = $additionalValues['stop_detail_id'] ?? 0;
        $params[] = $stopType;
        $params[] = $request['trip_id'] ?? 0;
        $params[] = $statusId;
        $params[] = $request['details']['position']['location']['lat'] ?? '0.0';
        $params[] = $request['details']['position']['location']['long'] ?? '0.0';
        $params[] = $additionalValues['loc_name'] ?? '';
        $params[] = $statusCode;
        $params[] = $statusStage;
        $params[] = $status;
        $params[] = $request['details']['comment'] ?? 'From Mobile';
        $params[] = $additionalValues['vehicle_id'] ?? 0;
        $params[] = $additionalValues['driver_id'] ?? 0;
        $params[] = 0;

        //createdon
        $params[] = ActionRequest::formatDateForInsert($date);

        //updatedon
        $params[] = ActionRequest::formatDateForInsert($dateModified);

        if ($update) {
            $params[] = $request['id'] ?? 0;
        }

        return $params;
    }

    /**
     * @param string $discrepancyIdentifier
     * @return array
     */
    private function getDiscrepancyState(string $discrepancyIdentifier) : array
    {
        return ActionDiscrepancy::$identifiers[$discrepancyIdentifier] ?? [];
    }

    /**
     * @param int $stopId
     * @return array
     */
    private function getStopData(int $stopId) : array
    {
        $sql = "
                SELECT
                    stop.id AS stop_id,
                    stop.stoptype AS stop_type,
                    stop.stopname AS loc_name
                FROM tb_shiporder_stops stop
                WHERE stop.id = ?";

        $result = $this->db->query($sql, [$stopId])->result_array();

        return $result[0] ?? [];
    }

    /**
     * @param int $tripId
     * @return array
     */
    private function getTripData(int $tripId): array
    {
        $sql = "
                SELECT
                       trip.id AS trip_id,
                       trip.shift_id,
                       ord.id AS order_id,
                       ord.shipmentid as shipment_id,
                       emp.id AS stop_detail_id,
                       trip.driver_id,
                       trip.vehicle_id
                FROM tb_employee emp
                JOIN tb_orders `ord` ON `ord`.order_id = emp.order_id
                JOIN tb_trips trip ON `ord`.trip_id = trip.id
                WHERE trip.id = ?
                LIMIT 1";

        $result = $this->db->query($sql, [$tripId])->result_array();

        return $result[0] ?? [];
    }
}
