<?php

namespace App\Mobile\V1\Trips\Actions\Model;

use App\ApiRequest\DataPreprocessing\BadRequestException;
use App\Mobile\V1\Trips\Actions\Enum\ActionType;
use App\Mobile\V1\Trips\Model\Location;
use App\Mobile\V1\Trips\Model\Merchandise;
use App\Mobile\V1\Trips\Model\Position;
use App\Mobile\V1\Trips\Model\StopAction;
use App\Mobile\V1\Trips\Model\StopActionDetails;
use Carbon\Carbon;

/**
 * @class ActionDetails
 * select action(s) queries
 */
class ActionDetails
{
    /**
     * @var object
     */
    protected $db;

    /**
     * @param \CI_DB_mysqli_driver $db
     */
    public function __construct(\CI_DB_mysqli_driver $db)
    {
        $this->db = $db;
    }

    /**
     * @param int $id
     * @param array $merchandise
     * @param string $curtz
     * @return array
     * @throws BadRequestException
     */
    public function findByActionId(int $id, array $merchandise, string $curtz)
    {
        $sql = $this->baseQuery();
        $sql .= " WHERE sts.id = ? LIMIT 1";

        $this->db->query("SET SESSION group_concat_max_len = 1000000");

        $result = $this->db->query($sql, [$id])->result_array();
        $data = is_array($result) ? $result : [];

        if (empty($data)) {
            throw new BadRequestException('Invalid Request');
        }

        return $this->formatDataStructure($data, $merchandise, $curtz, true);
    }

    /**
     * @param int $stopId
     * @param array $merchandise
     * @param string $curtz
     * @param int|null $limit
     * @return array
     */
    public function findByStopId(int $stopId, array $merchandise, string $curtz, $limit = null): array
    {
        $sql = $this->baseQuery();
        $sql .= " WHERE sts.stop_id = ?";
        $sql .= " ORDER BY sts.createdon DESC " . ($limit ? 'limit ' . $limit : '');

        $this->db->query("SET SESSION group_concat_max_len = 1000000");

        $result = $this->db->query($sql, [$stopId])->result_array();
        $data = is_array($result) ? $result : [];

        if (empty($data)) {
            return $data;
        }

        return $this->formatDataStructure($data, $merchandise, $curtz);
    }

    /**
     * @return string
     */
    private function baseQuery()
    {
        return "SELECT sts.id,

                    (CASE
                        WHEN sts.status_id = 1 and sts.stop_type = 'D' THEN concat(sts.trip_id,'_',sts.stop_id,'_','delivery','_',sts.order_id)
                        WHEN sts.status_id = 1 and sts.stop_type = 'P' THEN concat(sts.trip_id,'_',sts.stop_id,'_','pickup','_',sts.order_id)
                        WHEN sts.status_id = 2 THEN concat(sts.trip_id,'_',sts.stop_id,'_','gateIn')
                        WHEN sts.status_id = 3 THEN concat(sts.trip_id,'_',sts.stop_id,'_','gateOut')
                    ELSE
                        concat(sts.trip_id,'_',sts.stop_id,'_','other','_',sts.order_id)
                    END) as virtual_id,

                    (CASE
                        WHEN sts.status_id = 1 and sts.stop_type = 'D' THEN 'delivery'
                        WHEN sts.status_id = 1 and sts.stop_type = 'P' THEN 'pickup'
                        WHEN sts.status_id = 2 THEN 'gateIn'
                        WHEN sts.status_id = 3 THEN 'gateOut'
                    ELSE
                        ''
                    END) as type,

                    (CASE
                        WHEN sts.status_code = '0212' or sts.status_code = '0219' THEN 'toDo'
                        WHEN sts.status_code = '0420' and sts.stop_type = 'P' THEN 'toDo'
                        WHEN sts.status_code = '0191' and sts.stop_type = 'P' THEN 'done'
                        WHEN sts.status_code = '0500' and sts.stop_type = 'P' THEN 'done'
                        WHEN sts.status_code = '0192' and sts.stop_type = 'D' THEN 'toDo'
                        WHEN sts.status_code = '2300' and sts.stop_type = 'D' THEN 'done'
                        WHEN sts.status_code = '3000' and sts.stop_type = 'D' THEN 'done'
                        WHEN sts.status_code = '3001' THEN 'notDone'
                        WHEN sts.status_code = '3002' THEN 'notDone'
                        WHEN sts.status_code = '0501' THEN 'notDone'
                        WHEN sts.status_code = '0502' THEN 'partial'
                        WHEN sts.status_code = '0503' THEN 'partial'
                        WHEN sts.status_code = '0504' THEN 'notDone'
                        WHEN sts.status_code = '0505' THEN 'notDone'
                    ELSE
                        'toDo'
                    END) as state,

                    sts.createdon AS `date`,
                    sts.updatedon AS date_modified,

                    (SELECT CONCAT(GROUP_CONCAT(
                        JSON_OBJECT(id, imgpath)  SEPARATOR ',')
                    ) AS images
                    FROM tb_pod_uploads
                    WHERE stop_id = sts.stop_id and stop_type = sts.stop_type and order_id = sts.order_id and trip_id = sts.trip_id) AS documents,

                    IF(sts.latitude IS NULL OR sts.latitude = '' OR sts.latitude = '0.0', '0.0', sts.latitude) AS latitude,
                    IF(sts.longitude IS NULL OR sts.longitude = '' OR sts.longitude = '0.0', '0.0', sts.longitude) AS longitude,

                    sts.updatedon AS position_date,

                    (CASE
                        WHEN sts.status_code = '3001' THEN 'refused'
                        WHEN sts.status_code = '3002' THEN 'damaged'
                        WHEN sts.status_code = '0502' THEN 'damaged'
                        WHEN sts.status_code = '0503' THEN 'short'
                        WHEN sts.status_code = '0504' THEN 'siteClosed'
                        WHEN sts.status_code = '0505' THEN 'refused'
                    ELSE
                        ''
                    END) as discrepancy,

                    sts.comment as comment,
                    o.id as orderId,
                    sf.carrier_instructions as instructions,
                    shs.startdate as minDateTime,
                    shs.enddate as maxDateTime,
                    sts.status_code as status_code

                    FROM tb_stop_status sts

                    JOIN tb_trips tr ON tr.id = sts.trip_id
                    JOIN tb_shifts sf ON sf.id = tr.shift_id
                    JOIN tb_shiporder_stops shs ON shs.id = sts.stop_id
                    JOIN tb_orders o ON o.id = sts.order_id";
    }

    /**
     * @param array $resultItems
     * @return array
     */
    private function getDocuments(array $resultItems): array
    {
        $documentsToArray = json_decode('[' . $resultItems['documents'] . ']', true);
        return array_map('current', is_array($documentsToArray) ? $documentsToArray : []);
    }

    /**
     * @param array $data
     * @param array $merchandiseData
     * @param string $curtz
     * @param bool $singleAction
     * @return array
     */
    private function formatDataStructure(array $data, array $merchandiseData, string $curtz, bool $singleAction = false): array
    {
        $response = [];

        foreach ($data as $resultItems) {
            $documentItems = $this->getDocuments($resultItems);

            if (in_array($resultItems['type'], ['0', ''])) {
                continue;
            }

            $type = $resultItems['type'];

            $temp = new StopAction(
                $resultItems['id'],
                $resultItems['virtual_id'],
                $type,
                $resultItems['instructions'] ?? '',
                (new Carbon($resultItems['minDateTime']))->tz($curtz),
                (new Carbon($resultItems['maxDateTime']))->tz($curtz),
                new StopActionDetails(
                    $resultItems['state'],
                    (new Carbon($resultItems['date']))->tz($curtz),
                    (new Carbon($resultItems['date_modified']))->tz($curtz),
                    $documentItems,
                    new Position(
                        new Location($resultItems['latitude'], $resultItems['longitude']),
                        (new Carbon($resultItems['position_date']))->tz($curtz)
                    ),
                    $resultItems['discrepancy'],
                    $resultItems['comment'] ?? '',
                    $resultItems['status_code'] ?? null,
                ),
                ...array_map(function (array $merchandise) {return Merchandise::fromArray($merchandise);}, (in_array($type, Merchandise::getValidActionTypes()) ? $merchandiseData : []))
            );

            if ($singleAction) {
                $response = $temp->toArray();
            } else {
                $response[] = $temp->toArray();
            }

        }

        return $response;
    }
}
