<?php

namespace App\Mobile\V1\Trips\Actions\Model;

use App\Mobile\V1\Trips\Actions\ActionType;

/**
 * @class ActionDocuments
 */
class ActionDocuments
{
    /**
     * @var object
     */
    private $db;

    /**
     * @param \CI_DB_mysqli_driver $db
     * @param int $id
     */
    public function __construct(\CI_DB_mysqli_driver $db, int $id = 0)
    {
        $this->id = $id;
        $this->db = $db;
    }

    /**
     * @param array $request
     * @param array $additionalValues
     * @param \CI_DB_mysqli_driver $documents
     * @return bool
     */
    public function add($request, $additionalValues, $documents)
    {
        $insertIds = [];

        if (is_string($documents)) {
            $documents = $this->stringToArray($documents);
        }

        if (!is_array($documents)) {
            return false;
        }

        foreach ($documents as $document) {
            if ($this->exists($document)) {
                continue;
            }

            $insertIds[] = $this->insert($request, $additionalValues, $document);
        }

        if (count($documents) && !count($insertIds)) {
            return false;
        }

        return true;
    }

    /**
     * @param array $request
     * @param array $additionalValues
     * @param string $documentName
     * @return int
     */
    private function insert($request, $additionalValues, $documentName): int
    {
        $sql = "
                INSERT INTO tb_pod_uploads
                SET
                    trip_id = ?,
                    shipment_id = ?,
                    stop_id = ?,
                    stop_detail_id = ?,
                    stop_type = ?,
                    doc_type = ?,
                    imgpath = ?,
                    pdfpath = ?,
                    latitude = ?,
                    longitude = ?,
                    user_id = ?,
                    createdby = ?,
                    order_id = ?,
                    status_code = ?,
                    status = ?,
                    createdon = ?,
                    updatedon = ?,
                    receiver_name = ?
                ";

        $this->db->query($sql, $this->setQueryParameters($request, $additionalValues, $documentName));

        return $this->db->insert_id();
    }

    /**
     * @param array $request
     * @param array $additionalValues
     * @param string $documentName
     * @return array
     */
    private function setQueryParameters($request, $additionalValues, $documentName, $update = false)
    {
        $typeIdentifier = $request['type'] ?? 0; //gateIn, pickup, delivery, gateOut
        $stopType = $additionalValues['stop_type'] ?? 'P'; //P or D

        $params[] = $additionalValues['trip_id'] ?? 0;
        $params[] = $additionalValues['shift_id'] ?? 0;
        $params[] = $additionalValues['stop_id'] ?? 0;
        $params[] = $additionalValues['stop_detail_id'] ?? 0;
        $params[] = $stopType;
        $params[] = $this->getDocumentType($typeIdentifier, $stopType);
        $params[] = $documentName;
        $params[] = $documentName;
        $params[] = $request['details']['position']['location']['lat'] ?? '0.0';
        $params[] = $request['details']['position']['location']['long'] ?? '0.0';
        $params[] = 0;
        $params[] = 0;
        $params[] = $additionalValues['order_id'] ?? 0;
        $params[] = $stopType == 'D' ? 'DELIVERY_OK_PIC' : 'PICKUP_OK_PIC';
        $params[] = 1;

        //createdon
        if (!$update) {
            $params[] = date("Y-m-d H:i:s");
        }

        $params[] = date("Y-m-d H:i:s");
        $params[] = null;

        return $params;
    }

    /**
     * @param string $documentName
     * @return bool
     */
    private function exists($documentName)
    {
        $sql = "SELECT id from tb_pod_uploads WHERE imgpath = ?";

        $result = $this->db->query($sql, [$documentName])->result_array();
        $data = is_array($result) ? $result : [];

        return !empty($data);
    }

    /**
     * @param string $data
     * @return array
     */
    private function stringToArray($data)
    {
        $items = [];

        $data = preg_replace('[[\s\'\"\[\(\]\)array]]', '', $data);
        $temp = explode(",", $data);

        foreach ($temp as $v) {
            $items[] = $v;
        }

        return $items;
    }

    /**
     * @param string $typeIdentifier
     * @param string $stopType
     * @return int
     */
    private function getDocumentType(string $typeIdentifier, string $stopType) : int
    {
        if ($stopType == 'D') {
            $type = ActionType::$deliveryIdentifiers[$typeIdentifier] ?? 1;
        } else {
            $type = ActionType::$pickupIdentifiers[$typeIdentifier] ?? 1;
        }

        return (int)$type;
    }
}
