<?php

namespace App\Mobile\V1\Trips\Actions\Service;

use App\Mobile\V1\Trips\Actions\ActionType;
use App\Mobile\V1\Trips\Actions\Model;

final class ShipmentStatus {

    private const PICKUP_GATE_IN = 'pickup_gateIn';
    private const PICKUP = 'pickup';
    private const PICKUP_GATE_OUT = 'pickup_gateOut';
    private const IN_TRANSIT = 'in_transit';
    private const DELIVERY_GATE_IN = 'delivery_gateIn';
    private const DELIVERY = 'delivery';
    private const DELIVERY_GATE_OUT = 'delivery_gateOut';

    private const STOP_STATUS_AUTO_INSERT = [
        self::PICKUP_GATE_IN => [],
        self::PICKUP => [],
        self::PICKUP_GATE_OUT => [self::IN_TRANSIT],
        self::DELIVERY_GATE_IN => [],
        self::DELIVERY => [],
        self::DELIVERY_GATE_OUT => []
    ];

    private $stopStatuses = [
        self::PICKUP_GATE_IN => [
            'stop_type' => 'P',
            'status_id' => 2,
            'status' => 1,
            'status_code' => '0420',
            'status_stage' => 'Pickup Gate In',
            'reason' => 'From Mobile'
        ],
        self::PICKUP => [
            'stop_type' => 'P',
            'status_id' => 1,
            'status' => 1,
            'status_code' => '0500',
            'status_stage' => 'Pickup',
            'reason' => 'From Mobile'
        ],
        self::PICKUP_GATE_OUT => [
            'stop_type' => 'P',
            'status_id' => 3,
            'status' => 1,
            'status_code' => '0191',
            'status_stage' => 'Pickup Gate Out',
            'reason' => 'From Mobile'
        ],
        self::IN_TRANSIT => [
            'stop_type' => 'P',
            'status_id' => 4,
            'status' => 1,
            'status_code' => '1550',
            'status_stage' => 'In Transit',
            'reason' => 'From Mobile'
        ],
        self::DELIVERY_GATE_IN => [
            'stop_type' => 'D',
            'status_id' => 2,
            'status' => 1,
            'status_code' => '0192',
            'status_stage' => 'Delivery Gate In',
            'reason' => 'From Mobile'
        ],
        self::DELIVERY => [
            'stop_type' => 'D',
            'status_id' => 1,
            'status' => 1,
            'status_code' => '2300',
            'status_stage' => 'Delivery',
            'reason' => 'From Mobile'
        ],
        self::DELIVERY_GATE_OUT => [
            'stop_type' => 'D',
            'status_id' => 3,
            'status' => 1,
            'status_code' => '3000',
            'status_stage' => 'Delivery Gate Out',
            'reason' => 'From Mobile'
        ]
    ];

    private $shipmentStatusModel;

    /**
     * @param Model\ShipmentStatus $shipmentStatusModel
     */
    public function __construct(Model\ShipmentStatus $shipmentStatusModel)
    {
        $this->shipmentStatusModel = $shipmentStatusModel;
    }

    /**
     * If we have dependent status, then auto-inject (for web app)
     *
     * @param int $actionId The id of tb_stop_status
     * @param string $typeIdentifier gateIn, pickup, delivery, gateOut
     * @param string $stopType P or D
     *
     * @return array
     */
    public function injectStatus(int $actionId, string $typeIdentifier, string $stopType): array
    {
        $ids = [];
        $key = $this->getStopStatusesArrayKey($typeIdentifier, $stopType);
        $injectItems = self::STOP_STATUS_AUTO_INSERT[$key];

        foreach($injectItems as $injectItem) {
            $ids[] = $this->shipmentStatusModel->createRecordFromAction($actionId, $this->stopStatuses[$injectItem]);
        }

        return $ids;
    }

    /**
     * @param string $typeIdentifier gateIn, pickup, delivery, gateOut
     * @param string $stopType P or D
     *
     * @return array
     */
    public function getStatusData(string $typeIdentifier, string $stopType) : array
    {
        $key = $this->getStopStatusesArrayKey($typeIdentifier, $stopType);

        return $this->stopStatuses[$key] ?? [];
    }


    /**
     * @param string $typeIdentifier gateIn, pickup, delivery, gateOut
     * @param string $stopType P or D
     *
     * @return string The internal key for fetching Shipment Status Data
     */
    private function getStopStatusesArrayKey(string $typeIdentifier, string $stopType): string
    {
        $suffix = '';

        // if gateIn or GateOut
        if (!in_array($typeIdentifier, ActionType::$stopTypeIdentifiers)) {
            $suffix = '_' . $typeIdentifier;
        }

        return ActionType::$stopTypeIdentifiers[$stopType] . $suffix;
    }
}