<?php

namespace App\Mobile\V1\Trips;

use App\ApiRequest\GetPartyInformation;
use App\ShiftsData\TbStopStatusFields\Status;
use App\ShiftsData\TbStopStatusFields\StatusId;
use App\ShiftsData\TbStopStatusFields\StopType;
use App\Mobile\V1\Trips\Exception\ShipmentOrderStopsNotFoundException;

class GetShipmentStops
{
    /**
     * @param string $shiftId
     * @param string $curtz
     * @param \CI_DB_mysqli_driver $db
     * @param array $orderIds
     * @param array $orders
     * @return array
     * @throws ShipmentOrderStopsNotFoundException
     */
    public static function getShipmentStops(string $shiftId, array $orderIds, array $orders, string $curtz, \CI_DB_mysqli_driver $db)
    {
        $stops = self::getStops($curtz, $shiftId, $db);

        if (empty($stops)) {
            throw new ShipmentOrderStopsNotFoundException("Shipment order stops not found.");
        }

        $result = [];
        $location = [];

        $prevsts = "N";
        $place = "";

        foreach ($stops as $i => $stop) {
            $stopDetails = self::stopDetails($stop, $orderIds, $db);

            if (!$stopDetails) {
                continue;
            }

            $status = $prevsts == "C" ? $prevsts = "S" : "N";
            $countPickupStops = 0;
            $countDropStops = 0;
            $bookingIds = [];

            foreach ($stopDetails as $stopDetail) {
                $countPickupStops += (int)$stopDetail->cnt_stop_ids;
                $countDropStops += (int)$stopDetail->cnt_drop_stop_ids;

                if ($stopDetail->order_id) {
                    $bookingIds[] = $stopDetail->order_id;
                }
            }

            $countStopDetails = $countPickupStops + $countDropStops;
            $countGateOut = self::countGateOutDetails($countPickupStops, $stop, $orderIds, $db);
            $countStatus = self::countStatuses($stop, $orderIds, $db);

            if ($countGateOut === $countStopDetails) {
                $status = $prevsts = "C";
            } elseif ($countStatus > 0) {
                $status = $prevsts = "S";
            }

            $status = ($status == "N" && $i === 0) ? "S" : $status;

            list($place, $location, $destinationParty, $sourceParty, $destination, $source) = self::processAddressesInGetShipmentStops(
                $orders,
                $stop,
                $place,
                $location,
                $shiftId,
                $i,
                $db
            );

            $result[] = self::prepareShipmentStopResultWithPlaces(
                $stop,
                $status,
                $bookingIds,
                $place,
                $location,
                $source,
                $destination,
                $sourceParty,
                $destinationParty
            );
        }

        return $result;
    }

    /**
     * @param $orders
     * @param $orderShipType
     * @param $place
     * @param $location
     * @param $stopId
     * @param $shiftId
     * @param int $index
     * @param \CI_DB_mysqli_driver $db
     * @return array
     */
    private static function processAddressesInGetShipmentStops(
        $orders,
        $stop,
        $place,
        $location,
        $shiftId,
        int $index,
        \CI_DB_mysqli_driver $db
    ): array {
        list($place, $location, $destinationParty, $sourceParty, $destination, $source, $stopsarr) = self::processLocationsInGetShipmentStops(
            $orders,
            $stop,
            $place,
            $location,
            $shiftId
        );

        $partyaddr = GetPartyInformation::getPartyAddrByOrdStops($stopsarr, $index, $db);

        if (!empty($partyaddr)) {
            if (!empty($partyaddr['source'])) {
                $source = $partyaddr['source'];
                $sourceParty = $partyaddr['source_party'];
            }
            if (!empty($partyaddr['destination'])) {
                $destination = $partyaddr['destination'];
                $destinationParty = $partyaddr['dest_party'];
            }
        }

        return [$place, $location, $destinationParty, $sourceParty, $destination, $source];
    }

    /**
     * @param $orders
     * @param $orderShipType
     * @param $place
     * @param array $location
     * @param $stopId
     * @param $shiftId
     * @return array
     */
    private static function processLocationsInGetShipmentStops(
        array $orders,
        \stdClass $stop,
        string $place,
        array $location,
        int $shiftId
    ): array {
        $companyCode = "";

        if (empty($orders)) {
            $location = [
                'address' => '',
                'city' => '',
                'zip' => '',
                'country' => ''
            ];

            $place = '';
        }

        foreach ($orders as $order) {
            $order = (object)$order;
            $companyCode = $order->company_code;

            if ($stop->stoptype == "P") {
                $place = $order->pickup_company;
                $address = $order->pickup_address1;
                $city = $order->pickup_city;
                $zip = $order->pickup_pincode;
                $country = $order->pickup_country;
            } else if ($stop->stoptype == "D") {
                $place = $order->delivery_company;
                $address = $order->delivery_address1;
                $city = $order->delivery_city;
                $zip = $order->delivery_pincode;
                $country = $order->delivery_country;
            }

            $location = [
                'address' => $address,
                'city' => $city,
                'zip' => $zip,
                'country' => $country
            ];
        }

        $source = $destination = $sourceParty = $destinationParty = "";

        $stopsarr = [
            'sstop' => ($stop->stoptype == "P") ? $stop->id : 0,
            'dstop' => ($stop->stoptype == "D") ? $stop->id : 0,
            'shipment_id' => $shiftId,
            'cmpcode' => $companyCode
        ];

        return [
            $place,
            $location,
            $destinationParty,
            $sourceParty,
            $destination,
            $source,
            $stopsarr
        ];
    }

    /**
     * @param \stdClass $stop
     * @param \CI_DB_mysqli_driver $db
     * @param array $orderIds
     * @return \stdClass[]
     */
    private static function stopDetails(\stdClass $stop, array $orderIds, \CI_DB_mysqli_driver $db)
    {
        if (empty($orderIds)) {
            return null;
        }

        $bookingIds = array_values(array_flip($orderIds));
        $questionmarks = str_repeat("?,", count($bookingIds)-1) . "?";

        $sql = "
                SELECT order_id, IF(stop_id = ?, COUNT(*), 0) AS cnt_stop_ids, IF(drop_stopid = ?, COUNT(*), 0) AS cnt_drop_stop_ids
                FROM tb_employee
                WHERE (stop_id = ? OR drop_stopid = ?) AND shift_id = ? AND `status` = ?  AND order_id IN (" . $questionmarks . ")
                GROUP BY order_id
              ";

        $params = array_merge([$stop->id, $stop->id, $stop->id, $stop->id, $stop->shift_id, 1], $bookingIds);
        $result = $db->query($sql, $params)->result();

        return $result ?? [];
    }

    /**
     * @param string $curtz
     * @param int $shiftId
     * @param \CI_DB_mysqli_driver $db
     * @return \stdClass[]
     */
    private static function getStops(string $curtz, int $shiftId, \CI_DB_mysqli_driver $db) : array
    {
        $sql = "
        SELECT id, stopname, plat, plng, stopcity, stoptype, ordernumber,
            convertToClientTZ(startdate, ?) as startdate,
            convertToClientTZ(enddate, ?) as enddate,
            weight, volume, shipment_id as shift_id
        FROM tb_shiporder_stops
        WHERE shipment_id = ?
        ORDER BY ordernumber";

        $result = $db->query($sql, [$curtz, $curtz, $shiftId])->result();

        return $result ?? [];
    }

    /**
     * @param int $countStops,
     * @param \stdClass $stop
     * @param \CI_DB_mysqli_driver $db
     * @param array|null $orderIds
     * @return int
     */
    private static function countGateOutDetails(int $countStops, \stdClass $stop, array $orderIds, \CI_DB_mysqli_driver $db) : int
    {
        $db->from('tb_stop_status');
        $db->where("stop_id", $stop->id);
        $db->where("shipment_id", $stop->shift_id);
        $db->where("status", Status::ACTIVE);
        $db->where_in('order_id', $orderIds);

        if ($countStops > 0) {
            $db->where("status_id", StatusId::PICKUP_GATE_OUT);
            $db->where("stop_type", $stop->stoptype);
        } else {
            $db->where("status_id", StatusId::PICKUP);
            $db->where("stop_type", StopType::DROP);
        }

        return $db->count_all_results();
    }

    /**
     * @param \stdClass $stop
     * @param array $orderIds
     * @param \CI_DB_mysqli_driver $db
     * @return int
     */
    private static function countStatuses(\stdClass $stop, array $orderIds, \CI_DB_mysqli_driver $db) : int
    {
        $db->from('tb_stop_status');
        $db->where("stop_id", $stop->id);
        $db->where("shipment_id", $stop->shift_id);
        $db->where_in("order_id", $orderIds);

        return $db->count_all_results();
    }

    /**
     * @param string $stop_id
     * @param \stdClass $stop
     * @param string $status
     * @param array $bookingIds
     * @return array
     */
    private static function prepareShipmentStopResult(
        \stdClass $stop,
        string $status,
        array $bookingIds
    ): array {
        $currentDate = date("Y-m-d H:i:s");

        return [
            'id' => $stop->id,
            'name' => self::getStopName($stop),
            'plat' => $stop->plat ?? null,
            'plng' => $stop->plng ?? null,
            'ship_type' => $stop->stoptype ?? 'P',
            'shipment_weight' => $stop->weight ?? 0,
            'shipment_volume' => $stop->volume ?? 0,
            'startdate' => strtotime($stop->startdate ?? $currentDate),
            'enddate' => strtotime($stop->enddate ?? $currentDate),
            'priority' => $stop->ordernumber ?? 0,
            'status' => $status,
            'order_id' => $bookingIds
        ];
    }

    /**
     * @param \stdClass $stop
     * @param string $status
     * @param array $bookingIds
     * @param string $place
     * @param array $location
     * @param string $source
     * @param string $destination
     * @param string $source_party
     * @param string $dest_party
     * @return array
     */
    private static function prepareShipmentStopResultWithPlaces(
        \stdClass $stop,
        string $status,
        array $bookingIds,
        string $place,
        array $location,
        string $source,
        string $destination,
        string $source_party,
        string $dest_party
    ): array {
        return array_merge(static::prepareShipmentStopResult($stop, $status, $bookingIds), [
            'place' => $place,
            'location' => $location,
            'source' => $source,
            'destination' => $destination,
            'source_party' => $source_party,
            'dest_party' => $dest_party,
            'ordernumber' => $stop->ordernumber
        ]);
    }

    /**
     * @param \stdClass $stop
     * @return string
     */
    private static function getStopName(\stdClass $stop): string
    {
        $stopname = "";

        if (isset($stop->stopcity) && strlen($stop->stopcity) > 0) {
            $stopname .= ucfirst(str_replace("_", " ", $stop->stopcity)) . " - ";
        }
        if (isset($stop->stopname) && strlen($stop->stopname) > 0) {
            $stopname .= ucfirst(str_replace("_", " ", $stop->stopname));
        }

        return $stopname;
    }
}
