<?php

namespace App\Mobile\V1\Trips;

use App\Mobile\V1\Trips\Actions\ActionRequest;
use App\Mobile\V1\Trips\Actions\ActionType;
use App\Mobile\V1\Trips\Model\Merchandise;
use App\Mobile\V1\Trips\Model\SiteFactory;
use App\Mobile\V1\Trips\Model\Stop;
use App\Mobile\V1\Trips\Model\StopAction;
use App\Mobile\V1\Trips\Model\StopActionCollection;
use App\Mobile\V1\Trips\Model\Trip;
use App\ShipmentStops\OrderNotFound;
use Carbon\Carbon;

class MakeTripObject
{
    /**
     * @param int $shiftId
     * @param int $tripId
     * @param int|null $driverId
     * @param \CI_DB_mysqli_driver $db
     * @param string $curtz
     * @param string $shipmentId
     * @return Trip
     * @throws OrderNotFound
     * @throws \App\ShipmentStops\ShipmentOrderStopsNotFoundException
     */
    public static function makeTripObject(
        int $shiftId,
        int $tripId,
        ?int $driverId,
        \CI_DB_mysqli_driver $db,
        string $curtz,
        string $shipmentId
    ): Trip {
        $orders = Trip::getOrdersByTripId($tripId, $db);

        if (empty($orders)) {
            throw new OrderNotFound("Orders not found for shift id.");
        }

        $virtualMerchandises = [];
        $stops = [];

        $tripOrderIds = Trip::getOrderIdentifiers($orders);
        $orderAddresses = Trip::getOrderAddresses($orders);

        $tripStatus = GetTripStatus::getTripStatus($shiftId, $tripId, $db);
        $rawStops = GetShipmentStops::getShipmentStops($shiftId, $tripOrderIds, $orders, $curtz, $db);

        //add virtual merchandises
        foreach ($tripOrderIds as $tripOrderId) {
            $virtualMerchandises[$tripOrderId] = Merchandise::getItems($db, $tripOrderId);
        }

        foreach ($rawStops as $rawStop) {
            $bookingIds = $rawStop['order_id'];

            foreach ($bookingIds as $bookingId) {
                $orderId = self::getOrderId($tripOrderIds, $bookingId);

                $minArrivalCarbon = new Carbon($rawStop['startdate'], $curtz);
                $maxArrivalCarbon = new Carbon($rawStop['enddate'], $curtz);

                $actions = self::getActions($rawStop, $orderId, $tripId, $minArrivalCarbon, $maxArrivalCarbon, $virtualMerchandises, $curtz, $db);
                $site = SiteFactory::create((object)$orderAddresses[$orderId], $rawStop['ship_type']);

                $newStop = new Stop(
                    $rawStop['id'],
                    $rawStop['ordernumber'],
                    $site,
                    $minArrivalCarbon,
                    $maxArrivalCarbon,
                    ...$actions
                );

                if (count($stops) === 0 || $newStop->toArray() !== $stops[count($stops)-1]->toArray()) {
                    $stops[] = $newStop;
                }
            }
        }

        return new Trip(
            $shipmentId,
            $tripId,
            $tripStatus,
            $driverId,
            ...$stops
        );
    }

    /**
     * @param array $tripOrderIds
     * @param string $bookingId
     * @return int
     */
    private static  function getOrderId(array $tripOrderIds, string $bookingId) : int
    {
        $orderId = 0;

        foreach ($tripOrderIds as $orderId) {
            if (array_key_exists($bookingId, $tripOrderIds)) {
                $orderId = $tripOrderIds[$bookingId];
                break;
            }
        }

        return $orderId;
    }

    /**
     * @param array $rawStop
     * @param int $orderId
     * @param int $tripId
     * @param Carbon $minArrivalCarbon
     * @param Carbon $maxArrivalCarbon
     * @param array $virtualMerchandises
     * @param string $curtz
     * @param \CI_DB_mysqli_driver $db
     * @return StopAction[]
     * @throws Exception\ActionTypeNotValidException
     */
    private static function getActions(
        array $rawStop,
        int $orderId,
        int $tripId,
        Carbon $minArrivalCarbon,
        Carbon $maxArrivalCarbon,
        array $virtualMerchandises,
        string $curtz,
        \CI_DB_mysqli_driver $db
    ): array {
        $merchandiseItems = [];
        $actionRequest = new ActionRequest($db);

        //set regular/saved action merchandises
        if (in_array(ActionType::$stopTypeIdentifiers[$rawStop['ship_type']], Merchandise::getValidActionTypes())) {
            $merchandiseItems = Merchandise::getItems($db, $orderId);
        }

        //get regular/saved actions
        $actionsByStopId = $actionRequest->findByStopId($rawStop['id'], $merchandiseItems, $curtz);

        $actions = new StopActionCollection(array_map(
            function (array $action) use ($curtz): StopAction {
                return StopAction::fromArray($action, $curtz);
            }, $actionsByStopId
        ));

        return $actions->addVirtualStops(
            $rawStop['ship_type'],
            $tripId,
            $rawStop['id'],
            $minArrivalCarbon,
            $maxArrivalCarbon,
            $virtualMerchandises
        )->reorder()->toArray();
    }
}
