<?php

namespace App\Mobile\V1\Trips\Model;

use App\Mobile\V1\Trips\Actions\Enum\ActionType;
use App\ShiftsData\TbStopStatusFields\StopType;
use Carbon\Carbon;

/**
 * Immutable collection of Stop Actions.
 */
final class StopActionCollection
{
    /** @var array|StopAction[] Stop Actions */
    private array $stopActions;

    /**
     * @param array $stopActions Stop Actions to build collection of.
     */
    public function __construct(array $stopActions)
    {
        $this->stopActions = $stopActions;
    }

    /**
     * Adds virtual missing stop actions.
     *
     * @param string $stopType
     * @param int $tripId
     * @param int $stopId
     * @param Carbon $minArrival
     * @param Carbon $maxArrival
     * @param array $virtualMerchandises
     * @return $this
     */
    public function addVirtualStops(string $stopType, int $tripId, int $stopId, Carbon $minArrival, Carbon $maxArrival, array $virtualMerchandises): self
    {
        if (!in_array($stopType, [StopType::DROP, StopType::PICKUP])) {
            throw new \LogicException("Unknown stop type.");
        }

        $factory = new StopActionFactory();

        $missingStopActions = [];
        $transferType = $stopType === StopType::DROP ? ActionType::DELIVERY : ActionType::PICKUP;

        if (!$this->hasActionType(ActionType::GATE_IN)) {
            $missingStopActions[] = $factory->makeGateIn($tripId, $stopId, $minArrival, $maxArrival);
        }

        if (!$this->hasActionType($transferType)) {
            foreach ($virtualMerchandises as $merchandises) {
                $missingStopActions[] = $factory->makeTransfer($tripId, $stopId, $transferType, $minArrival, $maxArrival, $merchandises);
            }
        }

        if (!$this->hasActionType(ActionType::GATE_OUT)) {
            $missingStopActions[] = $factory->makeGateOut($tripId, $stopId, $minArrival, $maxArrival);
        }

        return new self(array_merge($this->stopActions, $missingStopActions));
    }

    /**
     * Returns stop actions in order: gateIns, transport, gateOuts
     *
     * @return $this
     */
    public function reorder(): self
    {
        $gateIns = [];
        $transfers = [];
        $gateOuts = [];

        foreach ($this->stopActions as $stopAction) {
            switch ($stopAction->getType()) {
                case ActionType::GATE_IN:
                    $gateIns[] = $stopAction;
                    break;
                case ActionType::GATE_OUT:
                    $gateOuts[] = $stopAction;
                    break;
                default:
                    $transfers[] = $stopAction;
            }
        }

        return new self(array_merge($gateIns, $transfers, $gateOuts));
    }

    /**
     * Returns stop actions as an array.
     *
     * @return array
     */
    public function toArray(): array
    {
        return $this->stopActions;
    }

    /**
     * Checks if collection already has action with given type.
     *
     * @param string $actionType
     * @return bool
     */
    private function hasActionType(string $actionType): bool
    {
        foreach ($this->stopActions as $action) {
            if ($action->isEqual($actionType)) {
                return true;
            }
        }

        return false;
    }
}
