<?php

namespace App\Mobile\V1\Trips\Model;

class Trip
{
    /**
     * @var string
     */
    private $status;
    /**
     * @var string|null
     */
    private $operator;
    /**
     * @var Stop[]
     */
    private $stops;
    /**
     * @var string
     */
    private $shipmentId;
    /**
     * @var int|null
     */
    private $id;

    public function __construct(string $shipmentId, ?int $id, string $status, ?string $operator, Stop ...$stops)
    {
        $this->shipmentId = $shipmentId;
        $this->status = $status;
        $this->operator = $operator;
        $this->stops = $stops;
        $this->id = $id;
    }

    public function toArray(): array
    {
        $stopsArr = [];
        foreach ($this->stops as $stop) {
            $stopsArr[] = $stop->toArray();
        }
        return [
            'id' => $this->id,
            'shipment_id' => $this->shipmentId,
            'status' => $this->status,
            'operator' => $this->operator,
            'stops' => $stopsArr,
        ];
    }

    /**
     * @return Stop[]
     */
    public function getStops(): array
    {
        return $this->stops;
    }

    /**
     * @param \CI_DB_mysqli_driver $db
     * @param string $shipmentId
     * @return bool
     */
    public static function exists($db, $shipmentId)
    {
        $db->from('tb_trips tr');
        $db->join('tb_shifts sf', 'sf.id = tr.shift_id', 'INNER');
        $db->where("sf.shipmentid", $shipmentId);

        return $db->count_all_results();
    }

    /**
     * @param \CI_DB_mysqli_driver $db
     * @param string $shipmentId
     * @param string $newStatusValue
     * @return mixed
     */
    public static function statusExists($db, $shipmentId, $newStatusValue)
    {
        $statusCode = TripStatusCode::getCodeByStatus($newStatusValue);

        $db->from('tb_stop_status sts');
        $db->join('tb_trips tr', 'sts.trip_id = tr.id', 'INNER');
        $db->join('tb_shifts sf', 'sf.id = tr.shift_id', 'INNER');
        $db->where("sf.shipmentid", $shipmentId);
        $db->where("sts.stop_id", 0);
        $db->where("sts.status_code", $statusCode);

        return $db->count_all_results();
    }

    /**
     * @param \CI_DB_mysqli_driver $db
     * @param int $tripId
     * @return string
     */
    public static function getShipmentId(\CI_DB_mysqli_driver $db, int $tripId): string
    {
        if (!$tripId) {
            return '';
        }

        $sql = "
                SELECT sf.shipmentid FROM tb_trips tr
                JOIN tb_shifts sf ON sf.id = tr.shift_id
                WHERE tr.id = ?
                LIMIT 1";

        $result = $db->query($sql, [$tripId])->result_array();

        return $result[0]['shipmentid'] ?? '';
    }

    /**
     * @param array $orderItems
     * @return array
     */
    public static function getOrderAddresses(array $orderItems): array
    {
        $address = [];
        $addresses = [];

        foreach ($orderItems as $orderItem) {
            $address['pickup_company'] = $orderItem['pickup_company'];
            $address['pickup_country'] = $orderItem['pickup_country'];
            $address['pickup_city'] = $orderItem['pickup_city'];
            $address['pickup_pincode'] = $orderItem['pickup_pincode'];
            $address['pickup_address1'] = $orderItem['pickup_address1'];
            $address['pickup_address2'] = $orderItem['pickup_address2'];
            $address['plat'] = $orderItem['plat'];
            $address['plng'] = $orderItem['plng'];
            $address['delivery_company'] = $orderItem['delivery_company'];
            $address['delivery_country'] = $orderItem['delivery_country'];
            $address['delivery_city'] = $orderItem['delivery_city'];
            $address['delivery_pincode'] = $orderItem['delivery_pincode'];
            $address['delivery_address1'] = $orderItem['delivery_address1'];
            $address['delivery_address2'] = $orderItem['delivery_address2'];
            $address['dlat'] = $orderItem['dlat'];
            $address['dlng'] = $orderItem['dlng'];

            $addresses[$orderItem['id']] = $address;
        }

        return $addresses;
    }

    /**
     * @param array $orderItems
     * @return array
     */
    public static function getOrderIdentifiers(array $orderItems): array
    {
        $identifiers = [];

        foreach ($orderItems as $orderItem) {
            $identifiers[$orderItem['booking_number']] = $orderItem['id'];
        }

        return $identifiers;
    }

    /**
     * @param int $tripId
     * @param \CI_DB_mysqli_driver $db
     * @return array
     */
    public static function getOrdersByTripId(int $tripId, \CI_DB_mysqli_driver $db): array
    {
        if (!$db || !$tripId) {
            return [];
        }

        $sql = "
                SELECT id,
                    order_id as booking_number,
                    pickup_company,
                    pickup_country,
                    pickup_city,
                    pickup_pincode,
                    pickup_address1,
                    pickup_address2,
                    plat,
                    plng,
                    delivery_company,
                    delivery_country,
                    delivery_city,
                    delivery_pincode,
                    delivery_address1,
                    delivery_address2,
                    dlat,
                    dlng,
                    company_code
                FROM tb_orders
                WHERE trip_id = ?";

        $result = $db->query($sql, [$tripId])->result_array();

        return $result ?? [];
    }

    /**
     * Get last status
     * @param \CI_DB_mysqli_driver $db
     * @param string $shipmentId
     * @return array
     */
    public static function getPreviousStatusData(\CI_DB_mysqli_driver $db, string $shipmentId): array
    {
        if (!$shipmentId) {
            return [];
        }

        $sql = "
                SELECT sts.id, sts.status_code, sts.status_id
                FROM tb_stop_status sts
                JOIN tb_trips tr ON sts.trip_id = tr.id
                JOIN tb_shifts sf ON sf.id = tr.shift_id
                WHERE sf.shipmentid = ? and stop_id = 0
                ORDER BY sts.createdon DESC
                LIMIT 1";

        $result = $db->query($sql, [$shipmentId])->result_array();

        $id = (isset($result[0]['id'])) ? $result[0]['id'] : 0;
        $statusCode = (isset($result[0]['status_code'])) ? $result[0]['status_code'] : '';
        $statusId = (isset($result[0]['status_id'])) ? $result[0]['status_id'] : 0;

        return [
            'id' => $id,
            'status_value' => TripStatusCode::getStatusByCode($statusCode),
            'status_id' => $statusId,
        ];
    }

    /**
     * @param \CI_DB_mysqli_driver $db
     * @param string $shipmentId
     * @return int (action id)
     */
    public static function start($db, $shipmentId)
    {
        $sql = "
                INSERT INTO tb_stop_status
                    (order_id,
                     shipment_id,
                     stop_id,
                     stop_detail_id,
                     stop_type,
                     trip_id,
                     status_id,
                     latitude,
                     longitude,
                     loc_name,
                     status_code,
                     status_stage,
                     status,
                     reason,
                     vehicle_id,
                     driver_id,
                     sentknlogin,
                     createdon,
                     updatedon)
                SELECT
                    sts.order_id,
                    sts.shipment_id,
                    sts.stop_id,
                    sts.stop_detail_id,
                    sts.stop_type,
                    sts.trip_id,
                    '" . TripStatusId::ACCEPTED_BY_DRIVER . "',
                    sts.latitude,
                    sts.longitude,
                    sts.loc_name,
                    '" . TripStatusCode::STARTED_CODE . "',
                    sts.status_stage,
                    sts.status,
                    'From Mobile',
                    sts.vehicle_id,
                    sts.driver_id,
                    sts.sentknlogin,
                    NOW(),
                    NOW()
                FROM tb_stop_status sts
                JOIN tb_trips tr ON sts.trip_id = tr.id
                JOIN tb_shifts sf ON sf.id = tr.shift_id
                WHERE sf.shipmentid = ? AND sts.status_code = ?
                ORDER BY sts.updatedon DESC
                LIMIT 1";

        $db->query($sql, [$shipmentId, TripStatusCode::NOT_STARTED_CODE]);

        return $db->insert_id();
    }

    /**
     * @param \CI_DB_mysqli_driver $db
     * @param string $shipmentId
     * @param string $newStatusValue
     * @return int (action id)
     */
    public static function update(\CI_DB_mysqli_driver $db, string $shipmentId, string $newStatusValue): int
    {
        $statusCode = TripStatusCode::getCodeByStatus($newStatusValue);
        $statusId = TripStatusId::getStatusIdByCode($statusCode);

        $sql = "
                INSERT INTO tb_stop_status SET
                    order_id = ?,
                    shipment_id = ?,
                    stop_id = ?,
                    stop_detail_id = ?,
                    stop_type = ?,
                    trip_id = ?,
                    status_id = ?,
                    latitude = ?,
                    longitude = ?,
                    loc_name = ?,
                    status_code = ?,
                    status_stage = ?,
                    status = ?,
                    reason = ?,
                    vehicle_id = ?,
                    driver_id = ?,
                    sentknlogin = ?,
                    createdon = ?,
                    updatedon = ?
                ";

        $additionalValues = self::getAdditionalValues($db, $shipmentId);

        $params = [
            $additionalValues['order_id'],
            $additionalValues['shift_id'],
            0,
            0,
            '',
            $additionalValues['trip_id'],
            $statusId,
            '',
            '',
            null,
            $statusCode,
            null,
            1,
            'From Mobile',
            $additionalValues['vehicle_id'],
            $additionalValues['driver_id'],
            0,
            date("Y-m-d H:i:s"),
            date("Y-m-d H:i:s")
        ];

        $db->query($sql, $params);

        return $db->insert_id();
    }

    /**
     * @param string $shipmentId
     * @param \CI_DB_mysqli_driver $db
     * @return array
     */
    public static function getAdditionalValues(\CI_DB_mysqli_driver $db, string $shipmentId): array
    {
        $sql = "
                SELECT
                       trip.id AS trip_id,
                       trip.shift_id,
                       ord.id AS order_id,
                       trip.driver_id,
                       trip.vehicle_id
                FROM tb_employee emp
                JOIN tb_orders `ord` ON `ord`.order_id = emp.order_id
                JOIN tb_trips trip ON `ord`.trip_id = trip.id
                WHERE ord.shipmentid = ?
                LIMIT 1";

        $result = $db->query($sql, [$shipmentId])->result_array();

        return $result[0] ?? [];
    }
}
