<?php

namespace App\MobileEvents\Payload;

use App\EventsQueue\Payload\EventPayload;
use App\MobileEvents\Payload\Position\Position;
use App\MobileEvents\Payload\Trip\RequiredTripIdentity;
use DateTimeImmutable;
use DateTimeZone;
use Exception;

final class AbortTripPayload implements EventPayload
{
    /** @var Position */
    private $position;

    /** @var RequiredTripIdentity */
    private $tripIdentity;

    /** @var string */
    private $createdSource;

    /** @var string */
    private $userId;

    /** @var DateTimeImmutable */
    private $currentDateTime;

    /** @var DateTimeZone */
    private $currentTimeZone;

    /** @var string */
    private $offset;

    /** @var string */
    private $legOrderId;

    /**
     * @param RequiredTripIdentity $tripIdentity
     * @param Position $position
     * @param string $createdSource
     * @param string $userId
     * @param DateTimeImmutable $currentDateTime
     * @param DateTimeZone $currentTimeZone
     * @param string $offset
     * @param string $legOrderId
     * 
     * @throws Exception
     */
    public function __construct(RequiredTripIdentity $tripIdentity, Position $position, string $createdSource, string $userId, DateTimeImmutable $currentDateTime, DateTimeZone $currentTimeZone, string $offset, string $legOrderId)
    {
        $this->tripIdentity = $tripIdentity;
        $this->position = $position;
        $this->createdSource = $createdSource;
        $this->userId = $userId;
        $this->currentDateTime = $currentDateTime;
        $this->currentTimeZone = $currentTimeZone;
        $this->offset = $offset;
        $this->legOrderId = $legOrderId;

        if ($this->validatePayload() === false) {
            throw new Exception('Reject trip is invalid');
        }
    }

    /**
     * Return payload internals into the array in format expected by external systems
     *
     * @return array
     */
    public function toArray(): array
    {
        return [
            "shipment_id" => $this->getTripIdentity()->getShiftId(),
            "trip_id" => $this->getTripIdentity()->getTripId(),
            "driver_id" => $this->getTripIdentity()->getDriverId(),
            "order_id" => $this->getLegOrderId(),
            "latitude" => $this->getPosition()->getLatitude(),
            "longitude" => $this->getPosition()->getLongitude(),
            "curtz" => $this->getCurrentTimeZone()->getName(),
            "hrs" => $this->getOffset()
        ];
    }

    private function validatePayload(): bool
    {
        return !empty($this->getUserId()) &&
            !empty($this->getCurrentTimeZone()) &&
            !empty($this->getOffset()) &&
            !empty($this->getLegOrderId());
    }

    /**
     * @return string
     */
    public function getCreatedSource(): string
    {
        return $this->createdSource;
    }

    /**
     * @return string
     */
    public function getUserId(): string
    {
        return $this->userId;
    }

    /**
     * @return DateTimeZone
     */
    public function getCurrentTimeZone(): DateTimeZone
    {
        return $this->currentTimeZone;
    }

    /**
     * @return string
     */
    public function getOffset(): string
    {
        return $this->offset;
    }

    /**
     * @return DateTimeImmutable
     */
    public function getCurrentDateTime(): DateTimeImmutable
    {
        return $this->currentDateTime;
    }

    /**
     * @return RequiredTripIdentity
     */
    public function getTripIdentity(): RequiredTripIdentity
    {
        return $this->tripIdentity;
    }

    /**
     * @return Position
     */
    public function getPosition(): Position
    {
        return $this->position;
    }

    /**
     * @return string
     */
    public function getLegOrderId(): string
    {
        return $this->legOrderId;
    }
}