<?php

namespace App\MobileEvents\Payload;

use App\EventsQueue\Payload\EventPayload;
use App\MobileEvents\Payload\Position\Position;
use App\MobileEvents\Payload\Trip\RequiredTripIdentity;
use App\MobileEvents\Payload\Trip\TripIdentity;
use DateTimeZone;
use Exception;

final class CloseTripPayload implements EventPayload
{
    /** @var bool */
    private $isStopUnitsPartiallyPicked;

    /** @var string */
    private $shipmentId;

    /** @var Position */
    private $position;

    /** @var RequiredTripIdentity */
    private $tripIdentity;

    /** @var DateTimeZone */
    private $currentTimeZone;

    /** @var string */
    private $hours;

    /** @var string */
    private $companyCode;

    /** @var string */
    private $dateTime;

    public function __construct(
        bool $isStopUnitsPartiallyPicked,
        string $shipmentId,
        RequiredTripIdentity $tripIdentity,
        Position $position,
        DateTimeZone $currentTimeZone,
        string $hours,
        string $companyCode,
        \DateTime $dateTime
    ) {
        $this->isStopUnitsPartiallyPicked = $isStopUnitsPartiallyPicked;
        $this->shipmentId = $shipmentId;
        $this->tripIdentity = $tripIdentity;
        $this->position = $position;
        $this->currentTimeZone = $currentTimeZone;
        $this->hours = $hours;
        $this->companyCode = $companyCode;
        $this->dateTime = $dateTime;

        if ($this->validatePayload() === false) {
            throw new Exception('Close Trip Payload is invalid');
        }
    }

    public function getTripIdentity(): TripIdentity
    {
        return $this->tripIdentity;
    }

    /**
     * @return Position
     */
    public function getPosition(): Position
    {
        return $this->position;
    }

    /**
     * Return payload internals into the array in format expected by external systems
     *
     * @return array
     */
    public function toArray(): array
    {
        return [
            "shipment_id" => $this->shipmentId,
            "trip_id" => $this->tripIdentity->getTripId(),
            "driver_id" => $this->tripIdentity->getDriverId(),
            "stop_id" => '',
            "order_id" => $this->tripIdentity->getOrderId(),
            "latitude" => $this->position->getLatitude(),
            "longitude" => $this->position->getLongitude(),
            "curtz" => $this->currentTimeZone->getName(),
            "hrs" => $this->hours,
            "company_code" => $this->companyCode,
            "dateTime" => $this->dateTime,
        ];
    }

    /**
     * Whether Stop Units were partially picked
     *
     * @return bool
     */
    public function isStopUnitsPartiallyPicked(): bool
    {
        return $this->isStopUnitsPartiallyPicked;
    }

    public function shouldNotifyOpenboxes(): bool
    {
        return $this->companyCode === "UKKN-CG";
    }

    /**
     * Checks if Payload has all required data
     *
     * @return bool
     */
    private function validatePayload(): bool
    {
        return !empty($this->tripIdentity) &&
            !empty($this->shipmentId) &&
            !empty($this->position) &&
            !empty($this->currentTimeZone) &&
            !empty($this->hours);
    }

    /**
     * @return string
     */
    public function getDateTime()
    {
        return $this->dateTime;
    }
}
