<?php

namespace App\MobileEvents\Payload;

use App\EventsQueue\Payload\EventPayload;
use App\MobileEvents\Payload\Milestone\MilestoneIdentity;
use App\MobileEvents\Payload\Position\Position;
use App\MobileEvents\Payload\Trip\RequiredTripIdentity;
use DateTimeZone;
use Exception;

final class MilestoneUpdatePayload implements EventPayload
{
    /** @var Position */
    private $position;

    /** @var RequiredTripIdentity */
    private $tripIdentity;

    /** @var string */
    private $createdSource;

    /** @var string */
    private $userId;

    /** @var DateTimeZone */
    private $currentTimeZone;

    /** @var string */
    private $offset;

    /** @var MilestoneIdentity */
    private $milestoneIdentity;

    /** @var string */
    private $comment;

    /**
     * @param RequiredTripIdentity $tripIdentity
     * @param Position $position
     * @param string $createdSource
     * @param string $userId
     * @param DateTimeZone $currentTimeZone
     * @param string $offset
     * @param MilestoneIdentity $milestoneIdentity
     * @param string $comment
     */
    public function __construct(
        RequiredTripIdentity $tripIdentity,
        Position $position,
        string $createdSource,
        string $userId,
        DateTimeZone $currentTimeZone,
        string $offset,
        MilestoneIdentity $milestoneIdentity,
        string $comment
    ) {
        $this->tripIdentity = $tripIdentity;
        $this->position = $position;
        $this->createdSource = $createdSource;
        $this->userId = $userId;
        $this->currentTimeZone = $currentTimeZone;
        $this->offset = $offset;
        $this->milestoneIdentity = $milestoneIdentity;
        $this->comment = $comment;

        if ($this->validatePayload() === false) {
            throw new Exception('milestone update payload is invalid');
        }
    }

    public function toArray(): array
    {
        return [
            "shipment_id" => $this->getShipmentId(),
            "trip_id" => $this->tripIdentity->getTripId(),
            "driver_id" => $this->tripIdentity->getDriverId(),
            "stop_id" => $this->getStopId(),
            "order_id" => $this->getMilestoneOrderId(),
            "inc_id" => $this->milestoneIdentity->getIncrementId(),
            "latitude" => $this->position->getLatitude(),
            "longitude" => $this->position->getLongitude(),
            "stop_type" => $this->getStopType(),
            "vehicle_id" => $this->getVehicleId(),
            "curtz" => $this->getCurrentTimeZoneName(),
            "hrs" => $this->offset,
            "ord_id" => $this->getTripOrderId(),
            "comment" => $this->getComment(),
        ];
    }

    private function validatePayload(): bool
    {
        return !empty($this->getCreatedSource()) &&
            !empty($this->getUserId());
    }

    public function getCreatedSource(): string
    {
        return $this->createdSource;
    }

    public function getUserId(): string
    {
        return $this->userId;
    }

    public function getStopId(): string
    {
        return $this->milestoneIdentity->getStopId();
    }

    public function getMilestoneOrderId(): string
    {
        return $this->milestoneIdentity->getOrderId();
    }

    public function getTripOrderId(): string
    {
        return $this->tripIdentity->getOrderId();
    }

    public function getShipmentId(): string
    {
        return $this->tripIdentity->getShiftId();
    }

    public function getStatusId(): string
    {
        return $this->milestoneIdentity->getStatusId();
    }

    public function getStopType(): string
    {
        return $this->milestoneIdentity->getStopType();
    }

    public function getCurrentTimeZoneName(): string
    {
        return $this->currentTimeZone->getName();
    }

    public function getVehicleId(): string
    {
        return $this->tripIdentity->getVehicleId();
    }

    public function getComment(): string
    {
        return $this->comment ?? '';
    }
}
