<?php

namespace App\MobileEvents\Payload;

use App\EventsQueue\Payload\EventPayload;
use App\MobileEvents\Payload\Document\UploadDocument;
use App\MobileEvents\Payload\Milestone\MilestoneIdentity;
use App\MobileEvents\Payload\Position\Position;
use App\MobileEvents\Payload\Trip\RequiredTripIdentity;
use DateTimeImmutable;
use DateTimeZone;
use Exception;

class PodUploadsPayload implements EventPayload
{
    /** @var Position */
    protected $position;

    /** @var RequiredTripIdentity */
    protected $tripIdentity;

    /** @var string */
    protected $createdSource;

    /** @var string */
    protected $userId;

    /** @var DateTimeImmutable */
    protected $currentDateTime;

    /** @var DateTimeZone */
    protected $currentTimeZone;

    /** @var string */
    protected $offset;

    /** @var MilestoneIdentity */
    protected $milestoneIdentity;

    /** @var UploadDocument */
    protected $document;

    /**
     * @param RequiredTripIdentity $tripIdentity
     * @param Position $position
     * @param string $createdSource
     * @param string $userId
     * @param DateTimeImmutable $currentDateTime
     * @param DateTimeZone $currentTimeZone
     * @param string $offset
     * @throws Exception
     */
    public function __construct(RequiredTripIdentity $tripIdentity, Position $position, string $createdSource, string $userId, DateTimeImmutable $currentDateTime, DateTimeZone $currentTimeZone, string $offset, MilestoneIdentity $milestoneIdentity, UploadDocument $document)
    {
        $this->tripIdentity = $tripIdentity;
        $this->position = $position;
        $this->createdSource = $createdSource;
        $this->userId = $userId;
        $this->currentDateTime = $currentDateTime;
        $this->currentTimeZone = $currentTimeZone;
        $this->offset = $offset;
        $this->milestoneIdentity = $milestoneIdentity;
        $this->document = $document;
        if ($this->validatePayload() === false) {
            throw new Exception('POD upload payload is invalid');
        }
    }

    /**
     * Return payload internals into the array in format expected by external systems
     *
     * @return array
     */
    public function toArray(): array
    {
        return [
            "shipment_id" => $this->getTripIdentity()->getShiftId(),
            "trip_id" => $this->getTripIdentity()->getTripId(),
            "driver_id" => $this->getTripIdentity()->getDriverId(),
            "stop_id" => $this->getMilestoneIdentity()->getStopId(),
            "order_id" => $this->getMilestoneIdentity()->getOrderId(),
            "txn_id" => $this->getMilestoneIdentity()->getTransactionCode(),
            "txn_code" => $this->getMilestoneIdentity()->getTransactionCode(),
            "file_name" => $this->getDocument()->getFileName(),
            "file_type" => $this->getDocument()->getFileType(),
            "inc_id" => $this->getMilestoneIdentity()->getTransactionCode(),
            'pod_type' => $this->getMilestoneIdentity()->getStatusId(),
            "latitude" => $this->getPosition()->getLatitude(),
            "longitude" => $this->getPosition()->getLongitude(),
            "stop_type" => $this->getMilestoneIdentity()->getStopType(),
            "curtz" => $this->getCurrentTimeZone()->getName(),
            "hrs" => $this->getOffset(),
            "ord_id" => $this->getTripIdentity()->getOrderId(),
            "vehicle_id" => $this->getTripIdentity()->getVehicleId()
        ];
    }

    private function validatePayload(): bool
    {
        return !empty($this->getCreatedSource()) &&
            !empty($this->getUserId()) &&
            !empty($this->getCurrentTimeZone()) &&
            !empty($this->getOffset());
    }

    /**
     * @return string
     */
    public function getCreatedSource(): string
    {
        return $this->createdSource;
    }

    /**
     * @return string
     */
    public function getUserId(): string
    {
        return $this->userId;
    }

    /**
     * @return DateTimeZone
     */
    public function getCurrentTimeZone(): DateTimeZone
    {
        return $this->currentTimeZone;
    }

    /**
     * @return DateTimeImmutable
     */
    public function getCurrentDateTime(): DateTimeImmutable
    {
        return $this->currentDateTime;
    }

    /**
     * @return string
     */
    public function getOffset(): string
    {
        return $this->offset;
    }

    /**
     * @return RequiredTripIdentity
     */
    public function getTripIdentity(): RequiredTripIdentity
    {
        return $this->tripIdentity;
    }

    /**
     * @return Position
     */
    public function getPosition(): Position
    {
        return $this->position;
    }

    /**
     * @return MilestoneIdentity
     */
    public function getMilestoneIdentity(): MilestoneIdentity
    {
        return $this->milestoneIdentity;
    }

    /**
     * @return UploadDocument
     */
    public function getDocument(): UploadDocument
    {
        return $this->document;
    }
}
