<?php

namespace App\MobileEvents\Payload;

use App\EventsQueue\Payload\EventPayload;
use App\MobileEvents\Payload\Position\Position;
use App\MobileEvents\Payload\Trip\RequiredTripIdentity;
use DateTimeImmutable;
use DateTimeZone;
use Exception;

final class StartTripPayload implements EventPayload
{
    /** @var Position */
    private $position;

    /** @var RequiredTripIdentity */
    private $tripIdentity;

    /** @var string */
    private $createdSource;

    /** @var string */
    private $userId;

    /** @var DateTimeImmutable */
    private $currentDateTime;

    /** @var DateTimeZone */
    private $currentTimeZone;

    /** @var string */
    private $offset;

    /**
     * @param RequiredTripIdentity $tripIdentity
     * @param Position $position
     * @param string $createdSource
     * @param string $userId
     * @param DateTimeImmutable $currentDateTime
     * @param DateTimeZone $currentTimeZone
     * @param string $offset
     *
     * @throws Exception
     */
    public function __construct(
        RequiredTripIdentity      $tripIdentity,
        Position  $position,
        string            $createdSource,
        string            $userId,
        DateTimeImmutable $currentDateTime,
        DateTimeZone      $currentTimeZone,
        string            $offset
    ) {
        $this->tripIdentity = $tripIdentity;
        $this->position = $position;
        $this->createdSource = $createdSource;
        $this->userId = $userId;
        $this->currentDateTime = $currentDateTime;
        $this->currentTimeZone = $currentTimeZone;
        $this->offset = $offset;

        if ($this->validatePayload() === false) {
            throw new Exception('Start Trip is invalid');
        }
    }

    /**
     * Return payload internals into the array in format expected by external systems
     *
     * @return array
     */
    public function toArray(): array
    {
        return [
            "shipment_id" => $this->getTripIdentity()->getShiftId(),
            "trip_id" => $this->getTripIdentity()->getTripId(),
            "driver_id" => $this->getTripIdentity()->getDriverId(),
            "order_id" => $this->getTripIdentity()->getOrderId(),
            "vehicle_id" => $this->getTripIdentity()->getVehicleId(),
            "user_id" => $this->getUserId(),
            "stop_id" => '',
            "latitude" => $this->getPosition()->getLatitude(),
            "longitude" => $this->getPosition()->getLongitude(),
            "curtz" => $this->getCurrentTimeZone()->getName(),
            "hrs" => $this->getOffset(),
        ];
    }

    private function validatePayload(): bool
    {
        return !empty($this->getCreatedSource()) &&
            !empty($this->getUserId()) &&
            !empty($this->getCurrentDateTime()) &&
            !empty($this->getCurrentTimeZone()) &&
            !empty($this->getOffset());
    }

    /**
     * @return string
     */
    public function getCreatedSource(): string
    {
        return $this->createdSource;
    }

    /**
     * @return string
     */
    public function getUserId(): string
    {
        return $this->userId;
    }

    /**
     * @return DateTimeImmutable
     */
    public function getCurrentDateTime(): DateTimeImmutable
    {
        return $this->currentDateTime;
    }

    /**
     * @return DateTimeZone
     */
    public function getCurrentTimeZone(): DateTimeZone
    {
        return $this->currentTimeZone;
    }

    /**
     * @return string
     */
    public function getOffset(): string
    {
        return $this->offset;
    }

    /**
     * @return RequiredTripIdentity
     */
    public function getTripIdentity(): RequiredTripIdentity
    {
        return $this->tripIdentity;
    }

    /**
     * @return Position
     */
    public function getPosition(): Position
    {
        return $this->position;
    }
}
