<?php

namespace App\MobileEvents\Payload;

use App\EventsQueue\Payload\EventPayload;
use App\MobileEvents\Payload\Position\Position;
use App\MobileEvents\Payload\Trip\RequiredTripIdentity;
use DateTimeZone;
use Exception;

final class StopStatusUpdatePayload implements EventPayload
{
    /** @var Position */
    private $position;

    /** @var RequiredTripIdentity */
    private $tripIdentity;

    /** @var string */
    private $userId;

    /** @var DateTimeZone */
    private $currentTimeZone;

    /** @var string */
    private $offset;

    /** @var bool */
    private $allAccepted;

    /** @var string */
    private $stopId;

    /**
     * @param RequiredTripIdentity $tripIdentity
     * @param Position $position
     * @param string $userId
     * @param bool $allAccepted
     * @param string $stopId
     * @param DateTimeZone $currentTimeZone
     * @param string $offset
     * @throws Exception
     */
    public function __construct(
        RequiredTripIdentity     $tripIdentity,
        Position $position,
        string           $userId,
        bool             $allAccepted,
        string           $stopId,
        DateTimeZone     $currentTimeZone,
        string           $offset
    ) {
        $this->tripIdentity = $tripIdentity;
        $this->position = $position;
        $this->userId = $userId;
        $this->currentTimeZone = $currentTimeZone;
        $this->offset = $offset;
        $this->allAccepted = $allAccepted;
        $this->stopId = $stopId;

        if ($this->validatePayload() === false) {
            throw new Exception('Status Update is invalid');
        }
    }

    /**
     * Return payload internals into the array in format expected by external systems
     *
     * @return array
     */
    public function toArray(): array
    {
        return [
            "shipment_id" => $this->tripIdentity->getShiftId(),
            "trip_id" => $this->tripIdentity->getTripId(),
            "driver_id" => $this->tripIdentity->getDriverId(),
            "stop_id" => $this->getStopId(),
            "order_id" => $this->tripIdentity->getOrderId(),
            "latitude" => $this->position->getLatitude(),
            "longitude" => $this->position->getLongitude(),
            "curtz" => $this->currentTimeZone->getName(),
            "hrs" => $this->getOffset()
        ];
    }

    private function validatePayload(): bool
    {
        return !empty($this->getUserId()) &&
            !empty($this->getCurrentTimeZone()) &&
            !empty($this->getOffset());
    }

    /**
     * @return string
     */
    public function getUserId(): string
    {
        return $this->userId;
    }

    /**
     * @return DateTimeZone
     */
    public function getCurrentTimeZone(): DateTimeZone
    {
        return $this->currentTimeZone;
    }

    /**
     * @return string
     */
    public function getOffset(): string
    {
        return $this->offset;
    }

    /**
     * @return RequiredTripIdentity
     */
    public function getTripIdentity(): RequiredTripIdentity
    {
        return $this->tripIdentity;
    }

    /**
     * @return Position
     */
    public function getPosition(): Position
    {
        return $this->position;
    }

    /**
     * @return bool
     */
    public function isAllAccepted(): bool
    {
        return $this->allAccepted;
    }

    /**
     * @return string
     */
    public function getStopId(): string
    {
        return $this->stopId;
    }
}