<?php

namespace App\Pagination;

use CI_DB_query_builder;

class Pagination
{
    /** @var int $currentPage */
    private $currentPage;
    /** @var int $limit */
    private $limit;
    /** @var CI_DB_query_builder $query */
    private $query;
    /** @var array */
    private $urlArguments;

    public function __construct(CI_DB_query_builder $query, int $limit, int $currentPage, array $urlArguments = [])
    {
        $this->query = $query;
        if ($limit > 50) {
            $limit = 50;
        }

        if ($limit < 0) {
            $limit = 10;
        }

        $this->limit = $limit;

        if ($currentPage < 1) {
            $currentPage = 1;
        }

        $this->currentPage = $currentPage;
        $this->urlArguments = $urlArguments;
    }

    public function paginate(): PaginationResponse
    {
        $queryPaginator = new QueryPaginator($this->query, $this->limit, $this->currentOffset(), $this->nextOffset());
        $results = $queryPaginator->paginateResults();

        $hasNextPage = $this->hasNextPage($results, $queryPaginator);

        return new PaginationResponse(
            $results,
            $hasNextPage,
            $this->hasPreviousPage(),
            $this->limit,
            $this->currentPage,
            $this->urlArguments
        );
    }

    private function currentOffset(): int
    {
        if ($this->currentPage < 1) {
            return 0;
        }

        return ($this->currentPage - 1) * $this->limit;
    }

    /**
     * @param array $results
     * @param QueryPaginator $queryPaginator
     * @return bool
     */
    private function hasNextPage(array $results, QueryPaginator $queryPaginator): bool
    {
        if (($this->limit > 0) && count($results) === $this->limit) {
            return $queryPaginator->nextPageContainsResults();
        }

        return false;
    }

    private function nextOffset(): int
    {
        return (($this->currentPage - 1) + 1) * $this->limit;
    }

    private function hasPreviousPage(): bool
    {
        return $this->currentPage > 1;
    }
}
