<?php

namespace App\PodUploads\Builders;

use App\ApiRequest\DataPreprocessing\PostTimezoneFilter;
use App\Models\Signature;
use App\PodUploads\MakePod\Address;
use App\PodUploads\MakePod\Measurements;
use App\PodUploads\MakePod\MakePodFromOrderAndSignature;
use App\PodUploads\MakePod\PodData;
use App\ShiftsData\TbStopStatusFields\StopType;
use CI_Controller;
use DateTime;
use Exception;

class EPodBuilder
{
    protected object $order;
    protected CI_Controller $CI;
    protected string $epodTemplate = 'customer/downloadepod';
    private string $documentDate;
    private string $timezone = 'UTC';
    private array $measurements = [];
    private Signature $signature;
    private string $podFilename;
    private array $deliveryComments = [];
    private string $pickupDateTime = '';
    private string $deliveryDateTime = '';

    /**
     * @param string $order_id
     * @param CI_Controller $CI
     */
    public function __construct(string $order_id, CI_Controller $CI)
    {
        $this->CI = $CI;

        $this->fetchOrderByOrderId($order_id)
            ->setTimezone()
            ->fetchStopStatuses()
            ->setDocumentDate(explode(" ", $this->deliveryDateTime)[0])
            ->fetchMeasurements()
            ->fetchSignature();
    }

    public function setDocumentDate(string $date): self {
        $this->documentDate = $date ?: date("d/M/Y");

        return $this;
    }

    /**
     * @param string $timezone
     * @return EPodBuilder
     */
    public function setTimezone(string $timezone = 'UTC'): self {
        try {
            $this->timezone = PostTimezoneFilter::getCurtz($timezone);
        } catch (Exception $e) {
            $this->timezone = 'UTC';
        }

        return $this;
    }

    /**
     * @return EPodBuilder
     */
    public function useOpenboxesTemplate(): self {
        $this->epodTemplate = 'customer/downloadepod_openboxes';

        return $this;
    }

    /**
     * @return EPodBuilder
     */
    public function createPdf():self {
        $makePod = new MakePodFromOrderAndSignature($this->CI);
        $this->podFilename = $makePod->download($this->getPodData(), $this->epodTemplate);
        $this->logCompleted();

        return $this;
    }

    public function getPodFilename(): string {
        return $this->podFilename;
    }

    // use POD data to generate HTML version of ePOD document
    public function getHtml():string {
        return $this->CI->load->view($this->epodTemplate, $this->getPodData()->toArray(), true);
    }

    protected function logCompleted():void {
        log_message(
            "error",
            "EPodBuilder: epod generation completed for order :".
            " {$this->order->id} / {$this->order->order_id} / {$this->order->external_order_id}"
        );
    }

    protected function getPodData (): PodData
    {
        return new PodData(
            $this->getShipperAddress(),
            $this->getConsigneeAddress(),
            $this->getPickupDateTime(),
            $this->getDeliveryDateTime(),
            $this->order->order_id,
            $this->order->external_order_id ?: '-',
            $this->signature->getFilename(),
            $this->signature->getHash(),
            $this->getDeliveryComments(),
            new Measurements(
                $this->measurements['quantity'] ?? 1,
                $this->measurements['volume'] ?? 0,
                $this->measurements['weight'] ?? 0
            ),
            $this->timezone,
            $this->order->company_code,
            $this->documentDate
        );
    }

    /**
     * @return DateTime|null
     */
    private function getPickupDateTime(): ?DateTime {
        try {
            return $this->pickupDateTime ? new DateTime($this->pickupDateTime) : null;
        } catch (Exception $e) {
            return null;
        }
    }

    /**
     * @return DateTime|null
     */
    private function getDeliveryDateTime(): ?DateTime {
        try {
            return $this->deliveryDateTime ? new DateTime($this->deliveryDateTime) : null;
        } catch (Exception $e) {
            return null;
        }
    }

    private function getShipperAddress(): Address {
        return new Address(
            $this->order->pickup_company,
            empty($this->order->pickup_address2)
                ? $this->order->pickup_address1
                :  $this->order->pickup_address1 ."\n". $this->order->pickup_address2,
            $this->order->pickup_city,
            '',
            $this->order->pickup_country,
            $this->order->pickup_pincode
        );
    }

    private function getConsigneeAddress(): Address {
        return new Address(
            $this->order->delivery_company,
            empty($this->order->delivery_address2)
                ? $this->order->delivery_address1
                :  $this->order->delivery_address1 ."\n". $this->order->delivery_address2,
            $this->order->delivery_city,
            '',
            $this->order->delivery_country,
            $this->order->delivery_pincode
        );
    }

    private function getDeliveryComments():string {
        return implode('<br />',  array_filter($this->deliveryComments));
    }

    private function fetchSignature():self {
        $this->signature = new Signature();
        $this->signature->fetchByOrderId(
            $this->order->id
        );

        return $this;
    }

    private function fetchMeasurements(): self {
        $this->CI->load->model('Order');
        $this->measurements = $this->CI->Order->getTotalMeasurementData($this->order->id);

        return $this;
    }

    private function fetchStopStatuses():self {
        $this->deliveryComments = [];
        $this->deliveryDateTime = '';
        $this->pickupDateTime = '';

        $this->CI->load->model('stop_status');
        $results = $this->CI->stop_status->filterResults(
            $this->CI->stop_status->fetchStopStatusesByOrderId($this->order->id ?? 0),
            $requiredStatus = 1,
            $requiredStatusCodes = ['0500', '2300', '3000']
        );

        foreach ($results as $row) {
            $this->parseStopStatusRow($row);
        }

        return $this;
    }

    protected function parseStopStatusRow(object $row): void {

        if (!$this->deliveryDateTime && $row->stop_type === StopType::DROP) {
            $this->deliveryDateTime = $row->createdon;
        }
        if (!$this->pickupDateTime && $row->stop_type === StopType::PICKUP) {
            $this->pickupDateTime = $row->createdon;
        }
        if ($row->comment ?? "" !== '' && $row->status_code === '2300') {
            $this->deliveryComments [] = trim($row->comment);
        }
    }

    private function fetchOrderByOrderId(string $order_id): self {
        $this->CI->load->model("order");
        $this->order = $this->CI->order->getOrderByOrderId($order_id);

        return $this;
    }
}
