<?php

declare(strict_types=1);

namespace App\Reporting\MisReport;

use App\Reporting\MisReport\Vendor\VendorFactory;
use DateTimeImmutable;

final class AttemptsCalculator
{
    private const DATE_FORMAT = 'Y-m-d';

    private $manifestdated;
    private $pickupdate;
    private $cancelchk = '';
    private $fecalldate;
    private PickupAttempts $pickupAttempts;
    private DeliveryAttempts $deliveryAttempts;

    public function __construct(int $orderSource, array $orderScanData, int $vendorId, string $stsType)
    {
        $vendor = VendorFactory::create($vendorId);

        $this->pickupAttempts = new PickupAttempts($orderSource, $orderScanData, $vendor);
        $this->deliveryAttempts = new DeliveryAttempts($orderScanData, $vendor, $stsType);

        foreach ($orderScanData as $scanData) {
            if ($scanData['scan_val'] == "Manifested" || $scanData['scan_instruction'] == "Shipment details manifested" || $scanData['scan_val'] == "Pickup Pending") {
                $this->manifestdated = new DateTimeImmutable($scanData['scan_status_time']);
            }
            if (($scanData['scan_val'] == "In Transit" || $scanData['scan_val'] == "Open") && ($scanData['scan_instruction'] == "Pickup completed" || $scanData['scan_instruction'] == "Shipment picked up") && $vendor->isDelhivery()) {
                $this->pickupdate = new DateTimeImmutable($scanData['scan_status_time']);
            }
            if ($scanData['scan_val'] == "Shipment Picked Up" || $scanData['scan_status_code'] == '0011') {
                $this->pickupdate = new DateTimeImmutable($scanData['scan_status_time']);
            }

            if ($scanData['scan_val'] == "Canceled" || ($scanData['scan_val'] == "Not Picked" && $scanData['scan_type'] == "UD") || ($scanData['scan_instruction'] == "Seller cancelled the order" || $scanData['scan_instruction'] == "Pickup/KYC request cancelled")) {
                $this->cancelchk = $scanData['scan_instruction'];
            }

            if ($scanData['call_duration'] > 0) {
                $this->fecalldate = new DateTimeImmutable($scanData['scan_status_time']);
            }
        }
    }

    public function firstPickup(): string
    {
        return $this->pickupAttempts->first();
    }

    public function lastPickup(): string
    {
        return $this->pickupAttempts->last();
    }

    public function pickupsInTotal(): int
    {
        return $this->pickupAttempts->inTotal();
    }

    public function returnDispatchInTotal(): int
    {
        return $this->pickupAttempts->inTotalReturnDispatch();
    }

    public function firstReturnDispatch(): string
    {
        return $this->pickupAttempts->firstReturnDispatch();
    }

    public function deliveryDate(): string
    {
        return $this->deliveryAttempts->deliveryDate();
    }

    public function rtodtodate(): string
    {
        return $this->deliveryAttempts->rtodtodate();
    }

    public function rtoreason(): string
    {
        return $this->deliveryAttempts->rtoreason();
    }

    public function txnclosed(): string
    {
        return $this->deliveryAttempts->txnclosed();
    }

    public function pickupDate(): string
    {
        if (!$this->pickupdate instanceof DateTimeImmutable) {
            return '';
        }

        return $this->pickupdate->format(self::DATE_FORMAT);
    }

    public function manifested(): string
    {
        if (!$this->manifestdated instanceof DateTimeImmutable) {
            return '';
        }

        return $this->manifestdated->format(self::DATE_FORMAT);
    }

    public function cancelchk(): string
    {
        return $this->cancelchk;
    }

    public function fecalldate(): string
    {
        if (!$this->fecalldate instanceof DateTimeImmutable) {
            return '';
        }
        return $this->fecalldate->format(self::DATE_FORMAT);
    }

}
