<?php

declare(strict_types=1);

namespace App\Reporting\MisReport;

use App\Time\TimerInterface;

final class DeliveryStatus
{
    private const NOT_AVAILABLE = 'N/A';
    private const STATUS_DELAY = 'Delay';
    private const STATUS_ON_TIME = 'On-Time';
    private const STATUS_EARLY = 'Early';

    private int $delayDays = 0;

    private string $orderTypeCode;
    private string $status;
    private string $deliveryStatus = self::NOT_AVAILABLE;
    private string $deliveryDate;

    /**
     * @param string $deliveryDate Actual delivery date
     * @param string $promiseDate Promised delivery date
     * @param string $orderTypeCode Order type code
     * @param string $status Delivery status
     * @param TimerInterface $currentDate Timer for current date representation
     */
    public function __construct(string $deliveryDate, string $promiseDate, string $orderTypeCode, string $status, TimerInterface $currentDate)
    {
        $this->orderTypeCode = $orderTypeCode;
        $this->status = $status;
        $this->deliveryDate = $deliveryDate;

        if (!$promiseDate) {
            return;
        }

        $promiseDateTimestamp = strtotime(date('Y-m-d', strtotime($promiseDate)));

        if (!$deliveryDate) {
            $this->setDelayDays($promiseDateTimestamp, $currentDate->getTimestampForToday());
            $this->setDeliveryStatus();

            return;
        }

        $deliveryDateTimestamp = strtotime(date('Y-m-d', strtotime($deliveryDate)));

        if (!empty($promiseDate)) {
            // Contact Logistics Sales Order
            if (in_array($orderTypeCode, ['SO', 'Return'])) {
                $this->setDelayDays($promiseDateTimestamp, $deliveryDateTimestamp);
                $this->setDeliveryStatus();
            }

            if (in_array($status, ['In Transit', 'Return In Transit'])) {
                $this->setDelayDays($promiseDateTimestamp, $currentDate->getTimestampForToday());
                $this->setDeliveryStatus();
            }
        }
    }

    public function delayDays(): string
    {
        if ($this->isStatusAvailable()) {
            if ($this->deliveryDate === '' && $this->delayDays === 0 && $this->deliveryStatus === self::STATUS_ON_TIME) {
                return self::NOT_AVAILABLE;
            }

            return (string) $this->delayDays;
        }

        return self::NOT_AVAILABLE;
    }

    public function deliveryStatus(): string
    {
        if ($this->isStatusAvailable()) {
            return $this->deliveryStatus;
        }

        return self::NOT_AVAILABLE;
    }

    private function isStatusAvailable(): bool
    {
        if ($this->orderTypeCode == 'CC' || $this->status == 'Lost' || $this->deliveryStatus === self::NOT_AVAILABLE) {
            return false;
        }

        return true;
    }

    private function setDelayDays(int $promiseDate, int $deliveryDateTimestamp): void
    {
        $this->delayDays =  ($deliveryDateTimestamp - $promiseDate) / (60 * 60 * 24);

        if ($this->deliveryDate === '' && $this->delayDays < 0) {
            $this->delayDays = 0;
        }
    }

    private function setDeliveryStatus(): void
    {
        if ($this->deliveryDate === '') {
            $this->deliveryStatus = $this->delayDays > 0 ? self::STATUS_DELAY : self::STATUS_ON_TIME;

            return;
        }

        $this->deliveryStatus =  $this->delayDays > 0 ? self::STATUS_DELAY : ($this->delayDays < 0 ? self::STATUS_EARLY : self::STATUS_ON_TIME);
    }
}
