<?php

namespace App\Trip\Models;

use App\Order\Models\OrderReference;

/**
 * Updated stop fields: [order_id, shift_id, stop_type, trip_id, latitude, longitude]
 */
class CancelTrip
{
    private const STATUS_ID = 48;
    private const STATUS_CODE = '3050';

    private $db;

    /**
     * @var int
     */
    private $shiftId;

    private int $userId;

    public function __construct(string $shiftId, int $userId)
    {
        $ci = &get_instance();
        $this->db = $ci->db;
        $this->shiftId = $shiftId;
        $this->userId = $userId;
    }

    /**
     * @return bool
     */
    public function cancel() : bool
    {
        $this->db->trans_start();

        $this->updateStopStatus();
        $this->updateStops();
        $this->updateAssignedVehicle();
        $this->updateAssignedDriver();
        $this->updateTrip();
        $this->updateShift();
        $this->updateOrder();

        $refs = new OrderReference();
        $refs->deleteForShift($this->shiftId, ['DQ']);

        if ($this->db->trans_status() === FALSE) {
            $this->db->trans_rollback();

            return false;
        }

        $this->db->trans_complete();

        return true;
    }

    /**
     * @return int
     */
    public function updateStopStatus() : int
    {
        $sql = "SELECT id, order_id, shipment_id, stop_type, trip_id, latitude, longitude
                FROM tb_stop_status
                WHERE shipment_id = ? AND status_id = ? AND status_code = ?";

        $result = $this->db->query($sql, [$this->shiftId, 9, '0100'])->result();

        foreach ($result as $item) {
            $sql = "
                INSERT INTO tb_stop_status (
                    order_id,
                    shipment_id,
                    stop_id,
                    stop_detail_id,
                    stop_type,
                    trip_id,
                    status_id,
                    latitude,
                    longitude,
                    `status` ,
                    reason,
                    vehicle_id,
                    driver_id,
                    status_code,
                    createdon,
                    updatedon
                ) VALUES (" .
                    $item->order_id . "," .
                    $item->shipment_id . ",
                    0,
                    0,'" .
                    $item->stop_type . "'," .
                    $item->trip_id . "," .
                    self::STATUS_ID . ",'" .
                    $item->latitude . "','" .
                    $item->longitude . "',
                    1,
                    'Cancelled from Admin',
                    0,
                    0,
                    '" . self::STATUS_CODE . "',
                    '" . date("Y-m-d H:i:s") . "',
                    '" . date("Y-m-d H:i:s") . "'
                )";

                $this->db->query($sql);
        }

        return true;
    }

    /**
     * @return bool
     */
    private function updateStops() : bool
    {
        $sql = "UPDATE tb_shiporder_stops
                SET
                    stopstatus = ?,
                    status = ?,
                    updated_on = ?
                WHERE shipment_id = ?";

        return $this->db->query($sql, [0, 0, date("Y-m-d H:i:s"), $this->shiftId]);
    }

    private function logOrdersBeforeReset(): void
    {

        $query = $this->db->query("SELECT
            id, shipmentid, shift_id, vendor_id, trip_id, trip_sts, status, updatedon
        FROM tb_orders
        WHERE
            shift_id = ? AND user_id = ?", [ $this->shiftId, $this->userId ]);

        if (!$query->num_rows()) {
            return;
        }

        foreach($query->result() as $row) {
            log_message('error', "cancelTrip requested for order :
                id: {$row->id}, shipmentid: {$row->shipmentid}, shift_id: {$row->shift_id},
                vendor_id: {$row->vendor_id}, trip_id: {$row->trip_id}, trip_sts: {$row->trip_sts},
                status: {$row->status}, updatedon: {$row->updatedon} ");
        }
    }

    private function updateOrder() : void
    {
        $this->logOrdersBeforeReset();

        $this->db->query(
            "UPDATE tb_orders
                SET
                    shipmentid = ?,
                    shift_id = ?,
                    vendor_id = ?,
                    trip_sts = ?,
                    trip_id = ?,
                    status = ?,
                    updatedon = ?
                WHERE shift_id = ? AND user_id = ?",
            [
                null,
                0,
                0,
                0,
                0,
                1,
                date("Y-m-d H:i:s"),
                $this->shiftId,
                $this->userId
            ]
        );
    }

    private function updateShift() : void
    {
        $sql = "UPDATE tb_shifts SET status = ?, updated_on = ?
            WHERE id = ? AND user_id = ?";

        $this->db->query($sql, [0, date("Y-m-d H:i:s"), $this->shiftId, $this->userId]);
    }

    private function updateTrip() : void
    {
        $sql = "UPDATE tb_trips t
                JOIN tb_orders o ON o.trip_id = t.id
                SET t.status = ?, t.updated_on = ?
                WHERE o.shift_id = ?";

        $this->db->query($sql, [0, date("Y-m-d H:i:s"), $this->shiftId]);
    }

    private function updateAssignedVehicle() : void
    {
        $sql = "UPDATE tb_shft_veh sv
                JOIN tb_orders o ON o.shift_id = sv.shft_id
                SET sv.status = ?, sv.updated_on = ?
                WHERE o.shift_id = ? AND sv.user_id = ?";

        $this->db->query($sql, [0, date("Y-m-d H:i:s"), $this->shiftId, $this->userId]);
    }

    private function updateAssignedDriver() : void
    {
        $sql = "UPDATE tbl_assigned_drivers sv
                JOIN tb_trips t ON t.driver_id = sv.driver_id AND t.vehicle_id = sv.vehicle_id
                JOIN tb_orders o ON o.shift_id = t.shift_id
                SET sv.status = ?, sv.updated_on = ?
                WHERE o.shift_id = ? AND sv.user_id = ?";

        $this->db->query($sql, [0, date("Y-m-d H:i:s"), $this->shiftId, $this->userId]);
    }
}
