<?php

namespace App\TripStop;

class TripStopDetailsModel extends \CI_Model
{
    private array $cacheShiporderStops = [];
    private array $cachePickupDates = [];
    private array $cacheEmployeeStopOrDrop = [];
    private array $cacheReferenceValues = [];
    private array $cacheStopStatuses = [];
    private string $curtz;

    public function __construct()
    {
        parent::__construct();
        $this->curtz = $this->session->userdata("usr_tzone")['timezone'];
    }

    // cache read

    public function aggregateData(array $shipmentIds): void
    {
        $this->fetchShiporderStops($shipmentIds);
        $this->fetchEmployeeStopOrDrop();
        $this->fetchPickupDates();
        $this->fetchReferenceValues();
        $this->fetchStopStatuses();
    }

    public function getShiporderStops($id)
    {
        return $this->cacheShiporderStops[$id] ?? [];
    }

    public function getPickupDates(string $order_id)
    {
        return $this->cachePickupDates[$order_id] ?? [];
    }

    public function getEmployeeStopOrDrop(array $ids): array
    {
        $result = ['drop' => [], 'stop' => []];

        foreach (['drop', 'stop'] as $type) {
            foreach ($this->cacheEmployeeStopOrDrop[$type] ?? [] as $id => $row) {
                if (in_array($id, $ids, true)) {
                    $result [$type][$id] = $row;
                }
            }
        }
        return $result;
    }

    public function getReferenceValues(string $orderRowId): array {
        return $this->cacheReferenceValues[$orderRowId] ?? [];
    }

    public function getStopStatus(array $emps, string $stop_id, string $shipment_id): array
    {
        $key = $stop_id .'/'. $shipment_id;
        $statusOne = 0;
        $statusOthers = 0;

        if (!isset($this->cacheStopStatuses[$key])) {
            return [0,0];
        }

        foreach ($this->cacheStopStatuses[$key] as $detailId => $statuses) {
            if (!in_array($detailId, $emps, false)) {
                continue;
            }
            $statusOne += $statuses[1] ?? 0;
            $statusOthers += array_sum($statuses);
        }

        return [ $statusOne, $statusOthers ];
    }

    private function fetchShiporderStops(array $ids): void
    {
        $this->cacheShiporderStops = [];

        if (empty($ids)) {
            return;
        }

        $query = $this->db->query(
            "-- tripstopdetails model, fetchShiporderStops (1)
            --
            SELECT
                id, shipment_id,
                stopname,address,plat,plng,stoptype,stopstatus,ordernumber,
                convertToClientTZ(startdate,'" . $this->getCurtz() . "') as startdate,
                convertToClientTZ(enddate,'" . $this->getCurtz() . "') as enddate,
                weight,volume,ship_units
             FROM tb_shiporder_stops
             WHERE status = 1 AND shipment_id IN ?
             ORDER BY ordernumber ASC", [$ids]
        );
        if (!$query) {
            return;
        }
        foreach ($query->result_array() as $row) {
            $this->cacheShiporderStops [$row['shipment_id']] [$row['id']] = $row;
        }
    }

    private function getStopIdsFromShiporderStops(): array {
        return array_unique(
            array_reduce( $this->cacheShiporderStops,
                static function(array $carry, array $items) {
                    return [...$carry, ...array_keys($items)];
                },
                [])
        );
    }

    private function fetchEmployeeStopOrDrop(): void
    {
        // ids should be taken from cacheShiporderStops
        $ids = $this->getStopIdsFromShiporderStops();

        $this->cacheEmployeeStopOrDrop = [];

        if (empty($ids)) {
            return;
        }

        $query = $this->db->query(
            "-- tripstopdetails model, fetchEmployeeStopOrDrop (sub 1)
        SELECT
            id, order_id,
            stop_id, drop_stopid,
            shipment_volume, no_of_pkgs, accepted
        FROM tb_employee
        WHERE stop_id IN ? AND status= ?",
            [$ids, 1]
        );

        if ($query) {
            foreach ($query->result() as $row) {
                $this->cacheEmployeeStopOrDrop ['stop'][$row->stop_id] = $row;
            }
        }

        $query = $this->db->query(
            "-- tripstopdetails model, fetchEmployeeStopOrDrop (sub 2)
        SELECT
            id, order_id,
            stop_id, drop_stopid,
            shipment_volume, no_of_pkgs, accepted
        FROM tb_employee
        WHERE drop_stopid IN ? AND status= ?",
            [$ids, 1]
        );

        if ($query) {
            foreach ($query->result() as $row) {
                $this->cacheEmployeeStopOrDrop ['drop'][$row->drop_stopid] = $row;
            }
        }
    }

    private function fetchPickupDates(): void
    {
        // orderIds should be taken from cacheEmployeeStopOrDrop
        $orderIds = array_unique([
            ...$this->getKeyFromArrayOfObjects($this->cacheEmployeeStopOrDrop['drop'] ?? [], 'order_id'),
            ...$this->getKeyFromArrayOfObjects($this->cacheEmployeeStopOrDrop['stop'] ?? [], 'order_id'),
        ]);

        $this->cachePickupDates = [];

        if (empty($orderIds)) {
            return;
        }

        $query = $this->db->query(
            "-- tripstopdetails model, fetchPickupDates (2)
            --
            SELECT
                id,
                order_id,
                convertToClientTZ(pickup_datetime,'" . $this->getCurtz() . "') as early_pickup,
                convertToClientTZ(pickup_endtime,'" . $this->getCurtz() . "') as late_pickup,
                convertToClientTZ(delivery_datetime,'" . $this->getCurtz() . "') as early_delivery,
                convertToClientTZ(drop_endtime,'" . $this->getCurtz() . "') as late_delivery
            FROM tb_orders
            WHERE order_id IN ?", [$orderIds]
        );
        if (!$query) {
            return;
        }
        foreach ($query->result() as $row) {
            $this->cachePickupDates [$row->order_id] = $row;
        }
    }

    private function fetchReferenceValues():void
    {
        // orderRowIds should be taken from cachePickupDates .id
        $this->cacheReferenceValues = [];
        $orderRowIds = array_unique(
            array_reduce($this->cachePickupDates,
                static function($carry, $row) {
                    return [...$carry, $row->id];
                },
                [])
        );

        if (empty($orderRowIds)) {
            return;
        }

        $query = $this->db->query(
            "-- tripstopdetails model, fetchReferenceValues (3)
            --
            SELECT
                order_id,
                ref_value, reference_id
            FROM tb_order_references
            WHERE
                order_id IN ?
                AND reference_id IN ('CTR', 'SCAC')",
            [$orderRowIds]
        );

        if (!$query) {
            return;
        }

        foreach($query->result() as $row) {
            $this->cacheReferenceValues [$row->order_id][] = $row;
        }
    }

    private function fetchStopStatuses():void
    {
        $this->cacheStopStatuses = [];

        // emps are possible from cacheEmployeeStopOrDrop ids
        $emps = array_unique([
            ...$this->getKeyFromArrayOfObjects($this->cacheEmployeeStopOrDrop['drop'] ?? [], 'id'),
            ...$this->getKeyFromArrayOfObjects($this->cacheEmployeeStopOrDrop['stop'] ?? [], 'id'),
        ]);
        $stopIds = $this->getStopIdsFromShiporderStops();
        $shipmentIds = array_unique(array_keys($this->cacheShiporderStops));

        if (empty($emps) || empty($stopIds) || empty($shipmentIds)) {
            return;
        }

        $query = $this->db->query(
            "-- tripstopdetails model, fetchStopStatuses (4+5)
            --
            SELECT
                id,
                stop_detail_id, stop_id, shipment_id,
                status_id
            from tb_stop_status
            where stop_detail_id IN ?
            AND stop_id IN ?
            AND shipment_id IN ?
            AND status=1",
            [$emps, $stopIds, $shipmentIds]
        );
        if (!$query) {
            return;
        }
        foreach($query->result() as $row) {

            $key = $row->stop_id . '/' .$row->shipment_id;
            $detail = $row->stop_detail_id ?? '-';

            if (!isset($this->cacheStopStatuses[$key][$detail][$row->status_id])) {
                // inital values for status_id
                $this->cacheStopStatuses [$key][$detail][$row->status_id] = 0;
            }
            // only number of matching rows matters
            $this->cacheStopStatuses [$key][$detail][$row->status_id] ++;
        }
    }

    private function getKeyFromArrayOfObjects(array $objects, string $key):array
    {
        return array_values(
            array_reduce($objects, static function($carry, $item) use ($key) {
                $carry [$item->$key] = $item->$key;
                return $carry;
            }, [])
        );
    }

    private function getCurtz(): string
    {
        return $this->curtz ?: DFLT_TZ;
    }

}
