<?php

defined('BASEPATH') or exit('No direct script access allowed');

use App\ApiRequest\DataPreprocessing\BadRequestException;
use App\ApiRequest\DataPreprocessing\NoDataFoundException;
use App\ApiRequest\Printable;
use App\PalletLabel\Zpl\ZplBarCode;
use App\PalletLabel\Zpl\ZplLabelGenerator;
use Zebra\Client;

/**
 * @Description Only works for AUSTRALIA warehouse implementation, for UK we still use Etnxdocapiv1
 */
class Etnxdocapiv2 extends CI_Controller
{
    public function __construct()
    {
        header('Content-Type: application/json');
        parent::__construct();
        if (isset($_SERVER['HTTP_ORIGIN'])) {
            header("Access-Control-Allow-Origin: {$_SERVER['HTTP_ORIGIN']}");
            header('Access-Control-Allow-Credentials: true');
            header('Access-Control-Max-Age: 86400');
        }
        if ($_SERVER['REQUEST_METHOD'] == 'OPTIONS') {
            if (isset($_SERVER['HTTP_ACCESS_CONTROL_REQUEST_METHOD'])) {
                header("Access-Control-Allow-Methods: GET, POST, PUT, DELETE, OPTIONS");
            }
            if (isset($_SERVER['HTTP_ACCESS_CONTROL_REQUEST_HEADERS'])) {
                header("Access-Control-Allow-Headers: {$_SERVER['HTTP_ACCESS_CONTROL_REQUEST_HEADERS']}");
            }
        }
        $this->load->model('common');
        $this->load->helper('log_helper');
        $this->load->helper('aulabel_helper');
    }

    /**
     * @throws BadRequestException
     * @throws NoDataFoundException
     */
    public function submit_pallet()
    {
        $start = microtime(true);
        if (!isset($_POST)) {
            throw new BadRequestException("POST data should not be empty.");
        }
        log_message("error", "submit_pallet " . json_encode($_POST));
        $action = $_POST['action'] ?? 'print';
        if (!in_array($action, ['print', 'zpl'])) {
            $action = 'print';
        }
        $orderId = $_POST['order_id'] ?? "0";
        $cargoes = $_POST['cargoes'] ?? [];
        $cargoRecords = json_decode($cargoes, true);
        if (empty($cargoRecords)) {
            throw new NoDataFoundException("Cargo data can't be empty.");
        }

        $cargoDetailId = $cargoRecords[0]["id"];
        $labelExists = $this->labelExistsForCargo($orderId, $cargoDetailId);

        if ($labelExists === false) {
           $this->updateCargoScannedQuantity($cargoRecords[0]);
        }

        $labelData = getLabelData($cargoDetailId, $orderId);
        if (empty($labelData)) {
            throw new NoDataFoundException("Cargo doesn't exist.");
        }
        $data = [
            'status' => '1',
            'message' => 'Success',
            'user_id' => $labelData['userId']
        ];
        $zplGenerator = new ZplLabelGenerator();
        $zpl = $zplGenerator->generate($labelData)->label();
        if ($action == 'zpl') {
            $data['zpl'] = $zpl;
        } else {
            $this->printLabel($zpl, $labelData['userId']);
        }
        $data = Printable::printable($data);
        echo json_encode($data);
        log_message(
            "error",
            "[LabelPrinting] Etnxdocapiv2 > submit_pallet > label print response time  :: " . (microtime(true) - $start)
        );
    }


    /**
     * API endpoint to print label
     *
     * @param string $zpl Label to print
     * @param string $userid User to fetch printing data
     *
     * @throws BadRequestException
     */
    private function printLabel(string $zpl, $userid): void
    {
        ini_set('max_execution_time', '300');

        $printerIP = null;
        $printerPort = null;
        $printType = "WebSocket";
        $printerSerialNumber = null;

        if ($userid == '') {
            throw new BadRequestException('User Id cannot be empty.');
        }

        $userPrinters = $this->common->gettblrowdata(array(
            'user_id' => $userid,
            'status' => 1
        ), 'printerip, printerport, print_type, serial_number', "tb_user_printers", 0, 0);

        if (!empty($userPrinters)) {
            $printerIP = $userPrinters['printerip'];
            $printerPort = $userPrinters['printerport'];
            $printType = $userPrinters['print_type'];
            $printerSerialNumber = $userPrinters['serial_number'];
        }

        if ($printType == 'WebSocket' && $printerIP != null && $printerPort != null) {
            try {
                $start = microtime(true);
                $client = new Client($printerIP, $printerPort);
                $client->send($zpl);
                log_message(
                    "error",
                    "[LabelPrinting][zpl_socket_duration]Etnxdocapiv2 > printLabel > The time it took for the ZPL file to be sent to the WebSocket :: " . (microtime(
                            true
                        ) - $start)
                );
            } catch (Exception $e) {
                log_message("error", "Etnxdocapiv2 > printLabel > print job failed :: " . $e->getMessage());
                return;
            }
        } elseif ($printType == "ZebraCloud" && $printerSerialNumber != null) {
            try {
                $start = microtime(true);
                $this->sendZebraCloudPrint($zpl, $printerSerialNumber);
                log_message(
                    "error",
                    "[LabelPrinting][zpl_ZebraCloud_duration]Etnxdocapiv2 > printLabel > The time it took for the ZPL file to be sent to the ZebraCloud :: " . (microtime(
                            true
                        ) - $start)
                );
            } catch (Exception $e) {
                log_message("error", "Etnxdocapiv2 > printLabel > print job failed :: " . $e->getMessage());
                return;
            }
        }
    }

    /**
     * @param string $zpl ZPL label
     * @param string $printerSerialNumber
     *
     * @return void
     */
    private function sendZebraCloudPrint(string $zpl, $printerSerialNumber): void
    {
        $zebraCloudAPIKey = ZEBRA_API_KEY;
        $zebraCloudTenantId = "8372a7f9c09d5084a5cf07e3d0eeac8a";
        $tempFilePath = tempnam(sys_get_temp_dir(), "ZPL");
        $tempFilePointer = fopen($tempFilePath, "w+");
        fwrite($tempFilePointer, $zpl);
        fclose($tempFilePointer);

        $curl = curl_init();
        curl_setopt_array($curl, [
            CURLOPT_URL => 'https://api.zebra.com/v2/devices/printers/send',
            CURLOPT_RETURNTRANSFER => true,
            CURLOPT_ENCODING => '',
            CURLOPT_MAXREDIRS => 10,
            CURLOPT_TIMEOUT => 0,
            CURLOPT_FOLLOWLOCATION => true,
            CURLOPT_HTTP_VERSION => CURL_HTTP_VERSION_1_1,
            CURLOPT_CUSTOMREQUEST => 'POST',
            CURLOPT_POSTFIELDS => [
                'sn' => $printerSerialNumber,
                'zpl_file' => new CURLFILE($tempFilePath)
            ],
            CURLOPT_HTTPHEADER => [
                'apikey: ' . $zebraCloudAPIKey,
                'tenant: ' . $zebraCloudTenantId,
                'Content-Type: multipart/form-data'
            ],
        ]);
        if (ENVIRONMENT !== "development") {
            curl_setopt($curl, CURLOPT_PROXY, PROXY_CONSTANT_HERE);
        }
        $response = curl_exec($curl);
        curl_close($curl);

        $responseArray = json_decode($response, true);
        unlink($tempFilePath);
        if (!isset($responseArray["status"]) || $responseArray["status"] != "SUCCESS") {
            log_message("error", "Etnxdocapiv2 > sendZebraCloudPrint > print job failed :: " . $response);
        }
    }

    /**
     * @param array $cargoDetail Cargo details
     *
     * @return bool Returns false in case label was already generated
     * @throws BadRequestException
     */
    private function updateCargoScannedQuantity(array $cargoDetail): bool
    {
        $cargoDetailId = $cargoDetail['id'];
        $orderId = $cargoDetail['order_id'];
        $scannedQuantity = $cargoDetail['scanned_quantity'];
        $totalQuantity = $cargoDetail['total_quantity'];

        if ($totalQuantity < $scannedQuantity) {
            throw new BadRequestException("Scanned Quantity is greater than total quantity.");
        } else {
            $cargoDetails = $this->db->select(
                "id, cargo_id, quantity"
            )->get_where("tb_order_cargodetails", [
                'id' => $cargoDetailId,
                'order_id' => $orderId,
                'status' => '1'
            ]);

            if ($cargoDetails->num_rows() > 0) {
                $cargoId = $cargoDetails->row()->cargo_id;
                if ($this->labelExistsForCargo($orderId, $cargoId)) {
                    log_message('error', 'submit_pallet ' . sprintf(
                            'Cannot add new barcode for order_id=%s and cargo_id=%s, such barcode already exists.',
                            $orderId, $cargoId
                        )
                    );

                    return false;
                }
                $this->generateBarCode($orderId, $cargoDetail, $cargoId);

                $totalQuantity = $cargoDetails->row()->quantity;
                if ($totalQuantity == $scannedQuantity) {
                    $cargoDetailToUpdate['pallet_close'] = 0;
                }
                $cargoDetailToUpdate['scanned_quantity'] = $scannedQuantity;
                $this->db->where([
                    'id' => $cargoDetailId
                ])->update("tb_order_cargodetails", $cargoDetailToUpdate);
            }
            $this->updateOrderStatus($orderId);

            return true;
        }
    }

    /**
     * Scanned status update if all cartons scanned
     *
     * @param string $orderId Order identifier
     */
    private function updateOrderStatus($orderId): void
    {
        $scaninfo = $this->common->gettblrowdata(
            [
                'order_id' => $orderId
            ],
            'sum(quantity) tot_qty,sum(scanned_quantity) tot_scan_qty',
            'tb_order_cargodetails',
            0,
            0
        );
        if (!empty($scaninfo)) {
            $code = '2491';
            $statusinfo = $this->common->gettblrowdata([
                "status_code" => $code,
                'status' => 1
            ], 'id', 'tb_status_master', 0, 0);
            if (!empty($statusinfo) && round($scaninfo['tot_qty']) == round($scaninfo['tot_scan_qty'])) {
                $dt = date('Y-m-d H:i:s');
                $status_id = $statusinfo['id'];
                $innr_array = [
                    'order_id' => $orderId,
                    'status_id' => $status_id,
                    'status_code' => $code,
                    'status' => 1,
                    'updatedon' => $dt
                ];
                $check_array = [
                    'order_id' => $orderId,
                    'status_id' => $status_id,
                    'status_code' => $code,
                    'status' => 1
                ];
                $chk_qry = $this->db->select("id")->get_where('tb_order_status', $check_array);
                if ($chk_qry->num_rows() == 0) {
                    $innr_array['createdon'] = $dt;
                    $innr_array['status_date'] = $dt;
                    $this->db->insert('tb_order_status', $innr_array);
                }
            }
        }
    }

    /**
     * Generate and store new barcode with container_no sequence
     *
     * @param string $orderId Order identifier
     * @param array $cargoDetail Cargo detail
     * @param string $cargoId Cargo identifier
     */
    private function generateBarCode(string $orderId, array $cargoDetail, string $cargoId): void
    {
        $consignmentData = $this->common->gettblrowdata([
            'order_row_id' => $orderId
        ], 'consignment_note', 'tb_order_details', 0, 0);

        if (empty($consignmentData)) {
            return;
        }
        $consignmentNote = $consignmentData['consignment_note'] ?? '';

        $lastInsertedBarCodes = $this->db->query(
            'SELECT bar_code
                    FROM tb_order_cargo_labels
                    WHERE bar_code LIKE ?
                        AND order_id = ?
                    ORDER BY id DESC
                    LIMIT 1',
            [$consignmentNote . '%', $orderId]
        );

        $barCode = new ZplBarCode();
        if ($lastInsertedBarCodes->num_rows() > 0) {
            $newBarCode = $barCode->getNext($lastInsertedBarCodes->row()->bar_code);
        } else {
            $newBarCode = $barCode->getFirstForConsignment($consignmentNote);
        }

        $cargoLabel = [
            'cargo_id' => $cargoId,
            'order_id' => $orderId,
            'pallet_id' => $cargoDetail['pallet_id'] ?? 0,
            'bar_code' => $newBarCode,
            'status' => 1
        ];

        $this->common->insertTableData("tb_order_cargo_labels", $cargoLabel);
    }

    private function labelExistsForCargo(string $orderId, string $cargoId): bool
    {
        $existingBarcode = $this->db->query(
            'SELECT bar_code
                    FROM tb_order_cargo_labels
                    WHERE order_id = ?
                        AND cargo_id = ?
                    ORDER BY id DESC
                    LIMIT 1',
            [$orderId, $cargoId]
        );

        return $existingBarcode->num_rows() > 0;
    }
}
