<?php

use App\ApiRequest\DataPreprocessing\BadRequestException;
use App\ApiRequest\DataPreprocessing\UnauthorizedException;
use App\Mobile\V1\Authentication\GetShipmentIdsFromAuthorization;
use App\Mobile\V1\Trips\Exception\ShipmentDoesNotBelongToAuthorizationException;
use App\PodUploads\ImageNotFound;

class MobileV1Doc extends \CI_Controller
{
    public function __construct()
    {
        parent::__construct();
        $this->load->driver('cache', array('adapter' => 'redis', 'backup' => 'file'));
    }

    public function doc() {
        $this->checkAuthorization();

        if ($_SERVER['REQUEST_METHOD'] === 'POST') {
            $this->storeImage();
        } else if ($_SERVER['REQUEST_METHOD'] === 'GET') {
            $this->sendImageToUser();
        }
    }

    private function sendImageToUser()
    {
        $fileName = $_GET['name'];
        $filePath = NEW_POD_PATH . $fileName;

        if (!file_exists($filePath)) {
            throw new ImageNotFound("Image not found.");
        }

        $fileContent = file_get_contents($filePath);
        header("Content-Disposition: attachment; filename=\"$fileName\"");

        echo $fileContent;
    }

    private function storeImage() {
        $headers = $this->input->request_headers();
        $fileName = $headers['X-File-Name'] ?? $headers['x-file-name'];

        $fileNameParts = explode('.', $fileName);
        $extension = strtolower(end($fileNameParts));

        if (!in_array($extension, ["jpeg", "jpg", "png", "gif", "pdf"])) {
            throw new BadRequestException("Image Format Not Allowed");
        }

        $filePath = NEW_POD_PATH . $fileName;

        if (file_exists($filePath)) {
            $response = [
                'status' => 'ok',
                'notice' => 'File already uploaded.',
            ];
            echo json_encode($response);
            die;
        }

        $index = array_key_first($_FILES);
        $org = $_FILES[$index]['tmp_name'];
        move_uploaded_file($org, $filePath);
        http_response_code(204);
    }

    public function delete() {
        $this->checkAuthorization();

        if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
            throw new BadRequestException("Please use POST method in here.");
        }

        $fileName = $_POST['name'];
        $filePath = NEW_POD_PATH . $fileName;

        if (!file_exists($filePath)) {
            throw new ImageNotFound('Image for deletion not found');
        }

        unlink($filePath);
        http_response_code(204);
    }

    private function checkAuthorization(): void
    {
        $headers = $this->input->request_headers();
        $authorization = $headers['Authorization'] ?? $headers['authorization'] ?? null;

        if ($authorization === null) {
            throw new UnauthorizedException("Authorization is empty.");
        }

        $shipmentIds = GetShipmentIdsFromAuthorization::getShipmentIdsFromAuthorization($authorization, $this->cache);

        if (empty($shipmentIds)) {
            throw new ShipmentDoesNotBelongToAuthorizationException("Shipment not found in token.");
        }
    }
}
