<?php

use App\ApiRequest\DataPreprocessing\BadRequestException;
use App\ApiRequest\DataPreprocessing\JwtNotValidException;
use App\ApiRequest\DataPreprocessing\PostTimezoneFilter;
use App\ApiRequest\DataPreprocessing\UnauthorizedException;
use App\Mobile\V1\Authentication\AuthorizationCache;
use App\Mobile\V1\Authentication\GetNewAuthorization;
use App\Mobile\V1\Trips\GetTripsFromTripIds;
use App\Mobile\V1\Trips\Exception\ShipmentIdsNotProvidedForAuthorizationException;

class MobileV1Import extends \CI_Controller
{
    public function __construct()
    {
        parent::__construct();
        $this->load->driver('cache', array('adapter' => 'redis', 'backup' => 'file'));
        $this->load->helper(['jwt']);
    }

    /**
     * @throws BadRequestException
     * @throws JwtNotValidException
     * @throws ShipmentIdsNotProvidedForAuthorizationException
     * @throws \App\ApiRequest\DataPreprocessing\TimezoneNotSpecifiedException
     * @throws \App\Mobile\V1\Authentication\ShipmentIdNotFoundException
     * @throws \App\Mobile\V1\Authentication\StrongRandomnessMissingException
     * @throws \App\ShipmentStops\OrderNotFound
     * @throws \App\ShipmentStops\ShipmentOrderStopsNotFoundException
     */
    public function import() {
        $tokenData = $this->getTokenData($_GET['jwt'] ?? null);

        if (isset($tokenData['error']) && $tokenData['error'] === 'expired') {
            throw new JwtNotValidException("token is expired");
        }
        if (isset($tokenData['error_message']) && $tokenData['error_message'] === 'signature_invalid') {
            throw new JwtNotValidException("token signature not valid");
        }

        $headers = getallheaders();
        $curtz = PostTimezoneFilter::getValidatedCurtz($headers['Timezone'] ?? $headers['timezone'] ?? null);
        $shipmentIds = $tokenData['token']['sub'] ?? null;

        if ($shipmentIds === null) {
            throw new ShipmentIdsNotProvidedForAuthorizationException('Shipment IDs are missing');
        }

        if (is_string($shipmentIds)) {
            $shipmentIds = [$shipmentIds];
        }

        $authorization = $headers['Authorization'] ?? null;
        if ($authorization === null) {
            $authorization = GetNewAuthorization::get($this->cache);
        }

        $this->addShipmentIdToAuthorizationToken(
            $authorization,
            $this->cache,
            $shipmentIds
        );

        header('Authorization: ' . $authorization);

        $trips = GetTripsFromTripIds::getTripsFromShipmentIds($shipmentIds, $this->db, $curtz);
        echo json_encode($trips[0]);
    }

    /**
     * @param string $authorization
     * @param CI_Cache $cache
     * @param array $shipmentIds
     * @throws BadRequestException
     */
    private function addShipmentIdToAuthorizationToken(
        string $authorization,
        CI_Cache $cache,
        array $shipmentIds
    ): void {
        $cacheKey = AuthorizationCache::key($authorization);
        $list = $cache->get($cacheKey);
        if ($list === false) {
            throw new UnauthorizedException("Authorization is empty.");
        }

        $cache->save($cacheKey, ["shipmentIds" => $shipmentIds], AuthorizationCache::TIME);
    }

    /**
     * @param $jwt
     * @return array
     * @throws BadRequestException
     */
    private function getTokenData($jwt): array
    {
        return validate_jwt($jwt);
    }
}
