<?php

use App\ApiRequest\DataPreprocessing\UnauthorizedException;
use App\Mobile\V1\Trips\Exception\ShipmentDoesNotBelongToAuthorizationException;
use App\Mobile\V1\Trips\GetNextTripStatus;
use App\Mobile\V1\Trips\GetTripsFromTripIds;
use App\ApiRequest\DataPreprocessing\PostTimezoneFilter;
use App\ApiRequest\DataPreprocessing\BadRequestException;
use App\Mobile\V1\Trips\Model\Trip;
use App\Mobile\V1\Trips\Model\TripStatus;
use App\Mobile\V1\Trips\CheckAuthorization;

class MobileV1Trip extends \CI_Controller
{
    private $response;

    public function __construct()
    {
        parent::__construct();
        $this->load->driver('cache', array('adapter' => 'redis', 'backup' => 'file'));
    }

    public function index()
    {
        $headers = $this->input->request_headers();
        $authorization = $headers['Authorization'] ?? null;

        if ($authorization === null) {
            throw new UnauthorizedException("Authorization is empty.");
        }

        $curtz = PostTimezoneFilter::getValidatedCurtz($headers['Timezone'] ?? null);

        if ($this->input->server('REQUEST_METHOD') === 'POST') {
            $shipmentId = $this->updateTrip();

            $response = [
                'status' => $this->response['status'],
                'notice' => $this->response['notice'],
                'trip' => ($this->response['status'] == 'success' || $this->response['status'] == 'failed') ? GetTripsFromTripIds::getTripFromShiftId($shipmentId, $this->db, $curtz) : []
            ];
        } elseif($this->input->server('REQUEST_METHOD') === 'GET') {
            $shipmentId = $this->retrieveTrip();
            $response = GetTripsFromTripIds::getTripFromShiftId($shipmentId, $this->db, $curtz);
        } else {
            throw new BadRequestException('Invalid Request');
        }

        echo json_encode($response);
    }

    /**
     * @param string $shipmentId
     * @param array $data
     * @throws BadRequestException
     *
     * @return array Response data
     */
    private function statusUpdate(string $shipmentId, array $data): array
    {
        $statusData = Trip::getPreviousStatusData($this->db, $shipmentId);

        if (empty($data) || empty($statusData)) {
            throw new BadRequestException('Invalid Request');
        }

        $response = [
            'status' => 'success',
            'notice' => ''
        ];

        if (isset($data['status'])) {
            $newStatusValue =  $data['status'];
            $currentStatusValue = $statusData['status_value'] ?? '';

            if (Trip::statusExists($this->db, $shipmentId, $newStatusValue)) {
                $response['status'] = 'failed';
                $response['notice'] = 'Status ' . $newStatusValue . ' already exists, nothing was modified.';
            }

            $possibleNextStatuses = GetNextTripStatus::getNextStatuses(TripStatus::make($currentStatusValue));

            if (!in_array($newStatusValue, $possibleNextStatuses)) {
                return [
                    'status' => 'failed',
                    'notice' => 'Invalid trip status requested'
                ];
            }

            foreach ($possibleNextStatuses as $nextStatus) {
                if ($newStatusValue == TripStatus::started()) {
                    Trip::start($this->db, $shipmentId);
                } else {
                    Trip::update($this->db, $shipmentId, $nextStatus->getValue());
                }
            }
        }

        return $response;
    }

    /**
     * @return string
     * @throws BadRequestException
     * @throws ShipmentDoesNotBelongToAuthorizationException
     */
    private function updateTrip()
    {
        if (!$data =  $this->input->post()) {
            if ($this->input->server('REQUEST_METHOD') !== 'POST') {
                throw new BadRequestException("Invalid request");
            }

            $data = json_decode($this->security->xss_clean($this->input->raw_input_stream), true);
        }

        $shipmentId = Trip::getShipmentId($this->db, $data['id'] ?? 0);
        $headers = $this->input->request_headers();
        $authorization = $headers['Authorization'] ?? null;
        CheckAuthorization::isAuthorizedForShipment($authorization, $this->cache, $shipmentId);
        $this->response = $this->statusUpdate($shipmentId, $data);

        return $shipmentId;
    }

    /**
     * @return string
     * @throws BadRequestException
     * @throws ShipmentDoesNotBelongToAuthorizationException
     */
    private function retrieveTrip()
    {
        if (!$data =  $this->input->get()) {
            if ($this->input->server('REQUEST_METHOD') !== 'GET') {
                throw new BadRequestException("Invalid request");
            }

            $data = json_decode($this->security->xss_clean($this->input->raw_input_stream), true);
        }

        $shipmentId = Trip::getShipmentId($this->db, $data['id'] ?? 0);
        $headers = $this->input->request_headers();
        $authorization = $headers['Authorization'] ?? null;
        CheckAuthorization::isAuthorizedForShipment($authorization, $this->cache, $shipmentId);

        return $shipmentId;
    }
}
