<?php

if (!defined('BASEPATH')) {
    exit('No direct script access allowed');
}

class Rateenquiry extends CI_Controller
{
    private const FRT_CHARGE_ID = '39';
    private const WEIGHT_CHARGE_BASIS = ['PER KG', 'KG'];
    private const VOLUME_CHARGE_BASIS = ['PER CBM', 'CBM'];
    private const QUANTITY_CHARGE_BASIS = ['PER PIECE', 'PIECE'];
    private const TRIP_CHARGE_BASIS = ['PER TRIP', 'TRIP'];
    private const KM_CHARGE_BASIS = ['PER KM', 'KM'];
    private const MILES_CHARGE_BASIS = ['PER MILE', 'MILE'];
    private const LDM_CHARGE_BASIS = ['PER LDM', 'LDM'];

    public function __construct()
    {
        parent::__construct();
        if ($this->session->userdata('user_id') === null) {
            redirect('login');
        }
        $this->userId = $this->session->userdata('user_id');
        $this->companyCode = $this->session->userdata('company_code');
        $this->branchCode = $this->session->userdata('branch_code');
        $this->userCurrency = $this->session->userdata("usr_tzone")['currency'];
        $this->load->model("rateenquirymodel");
        $this->crossBorderCountries = $this->rateenquirymodel->checkCrossBorderCountries($this->companyCode);
    }

    public function index(): void
    {
        $data['page_title'] = $data['sub_title'] = $this->lang->line('rate_enquiry');
        $data['vehicleTypes'] = $this->rateenquirymodel->getVehicleTypeMasterData($this->crossBorderCountries);
        $data['cargoTypes'] = ["Pallet", "Box", "Roll", "Bottles", "Peices"];
        $data['companyCode'] = $this->companyCode;
        $data['pickupPartyType'] = $this->companyCode === "SGKN" ? "Pickup" : "Shipper";
        $data['deliveryPartyType'] = $this->companyCode === "SGKN" ? "Delivery" : "Consignee";
        $data['userCurrency'] = $this->userCurrency;
        $data['SGMYCondition'] = checkAccessConditions('RELATED_2_SG_MY', $this->companyCode);
        $postData = $this->input->post(null, true);
        if (!empty($postData)) {
            $data['rateRecords'] = $this->getRatesList($postData);
            $ajaxCall = $postData['ajaxCall'] ?? 0;
            if ($ajaxCall > 0) {
                echo json_encode($data['rateRecords']);
                return;
            }
        }
        $data['postData'] = $postData;
        $this->newtemplate->dashboard('rateenquiry/listpage', $data);
    }

    public function getPartyMasterData(): void
    {
        $data = [];
        $postData = $this->input->post(null, true);
        $partyType = $postData['partyType'] ?? 'Shipper';
        $partyTypeId = in_array($partyType, ['Shipper', 'Pickup']) ? 1 : 0;
        $partyTypes[] = $partyType;
        if ($this->crossBorderCountries['enabled'] > 0) {
            $partyTypes = $this->crossBorderCountries['enabled'] > 0 ? ['Yard', 'Border', 'xDock', 'In Transit', 'Port'] : [];
            if ($partyTypeId > 0) {
                $partyTypes[] = 'Shipper';
                $partyTypes[] = 'Pickup';
            } else {
                $partyTypes[] = 'Delivery';
                $partyTypes[] = 'Consignee';
            }
        }
        $partyMaterData = $this->rateenquirymodel->getPartyMasterData($partyTypes, $this->crossBorderCountries['companyCodes']);
        $masterData = $partyMaterData['masterData'] ?? [];
        $typesData = $partyMaterData['partyTypeData'] ?? [];
        foreach ($masterData as $eachLine) {
            $checkBox = "<input type='radio' name='masterlist' id='masterlist_" . $eachLine['id'] . "' class='masterlist' onchange='selectparty(" . $eachLine['id'] . "," . $partyTypeId . ")' value='" . $eachLine['id'] . "'>";
            $data[] = [
                'checkBox' => $checkBox,
                'rowId' => $eachLine['id'],
                'partyTypeId' => $eachLine['party_type_id'],
                'name' => $eachLine['name'],
                'code' => $eachLine['code'],
                'city' => $eachLine['location_id'],
                'country' => $eachLine['country'],
                'state' => $eachLine['state'],
                'street' => $eachLine['street'],
                'pincode' => $eachLine['pincode'],
                'district' => $eachLine['division_name'],
                'subDistrict' => $eachLine['sub_district'],
                'typeName' => $typesData['party_type_id'] ?? $partyType,
                'companyCode' => $eachLine['company_code'],
                'branchCode' => $eachLine['branch_code']
            ];
        }
        echo json_encode($data);
    }

    private function getRatesList(array $postData): array
    {
        $geoTypeList = ['1' => 'Region', '2' => 'Country', '3' => 'Province', '4' => 'District', '5' => 'City', '6' => 'Postal Code', '7' => 'Sub District'];
        $ajaxCall = $postData['ajaxCall'] ?? 0;
        $conditions = $this->setWhereConditionsByPostData($postData);
        $addressPairs = $this->generateAddressPairs($conditions['pickupData'], $conditions['deliveryData']);
        $query = $this->rateenquirymodel->getRateCharges(['addressPairs' => $addressPairs, 'userId' => $this->userId, 'conditions' => $conditions['conditions'], 'crossBorder' => $this->crossBorderCountries, 'companyCode' => $this->companyCode, 'customerRowId' => $this->session->userdata('cust_id') ?? 0]);
        $pageCount = $query->num_rows();
        $ratesData = $pageCount > 0 ? $query->result_array() : [];
        if (empty($ratesData)) {
            return [];
        }
        $offeringRowIds = $recordRowIds = $ServiceTypes = $offeringTypes = $rates = $ratesResult = $conversionData = $recordRowIdsWithConversion = $offeringRowIdsWithConversion = [];
        foreach ($ratesData as $eachLine) {
            $offeringTypes[] = $eachLine['offering_type'];
            $ServiceTypes[] = $eachLine['service_type'];
            $conversionRowId = $eachLine['uom_conversion_id'];
            if ($conversionRowId > 0 && $ajaxCall > 0) {
                $recordRowIdsWithConversion[] = $eachLine['recordRowId'];
                $offeringRowIdsWithConversion[] = $eachLine['offeringRowId'];
            } else {
                $offeringRowIds[] = $eachLine['offeringRowId'];
                $recordRowIds[] = $eachLine['recordRowId'];
            }
            $rates[] = [
                'serviceRowId' => $eachLine['serviceRowId'],
                'rateServiceId' => $eachLine['service_id'],
                'rateServiceType' => $eachLine['service_type'],
                'rateServiceName' => $eachLine['service_name'],
                'laneRowId' => $eachLine['lane_id'],
                'source' => $eachLine['source'],
                'destination' => $eachLine['destination'],
                'offeringRowId' => $eachLine['offeringRowId'],
                'rateOfferingId' => $eachLine['offering_id'],
                'rateOfferingName' => $eachLine['offering_name'],
                'rateOfferingType' => $eachLine['offering_type'],
                'offeringTariffType' => $eachLine['offeringTariffType'],
                'recordRowId' => $eachLine['recordRowId'],
                'rateRecordId' => $eachLine['recordId'],
                'rateRecordName' => $eachLine['recordName'],
                'customerProfileId' => $eachLine['cust_profile_id'],
                'vendorProfileId' => $eachLine['vendor_profile_id'],
                'sourceGeo' => $geoTypeList[$eachLine['source_geo']] ?? "",
                'destinationGeo' => $geoTypeList[$eachLine['destination_geo']] ?? "",
                'conversionRowId' => $conversionRowId
            ];
        }
        $offeringRowIds = array_filter($offeringRowIds);
        $recordRowIds = array_filter($recordRowIds);
        $serviceModeMaster = $this->rateenquirymodel->getServiceModeMasterData(array_unique($offeringTypes), array_unique($ServiceTypes));
        if (!empty($recordRowIdsWithConversion)) {
            $amountWithConversion = $this->getRatesUsingConversionFactor(['recordRowIdsWithConversion' => $recordRowIdsWithConversion, 'offeringRowIdsWithConversion' => $offeringRowIdsWithConversion, 'postData' => $postData, 'geoPickupData' => $conditions['geoPickupData'], 'geoDeliveryData' => $conditions['geoDeliveryData']]);
        }
        $amountWithoutConversion = $this->getTotalChargesAmount(array_unique($offeringRowIds), array_unique($recordRowIds), $conditions['geoPickupData'], $conditions['geoDeliveryData'], $postData);
        $amountDetails = $this->arrangeConversionAndNonConversionAmountIntoSingleArray($amountWithoutConversion, $amountWithConversion ?? []);
        $offeringData = $amountDetails['offeringData'] ?? [];
        $recordServicesData = $amountDetails['recordServicesData'] ?? [];
        $recordChargesData = $amountDetails['recordChargesData'] ?? [];
        for ($i = 0, $iMax = count($rates); $i < $iMax; $i++) {
            $offeringAmount = $offeringData[$rates[$i]['offeringRowId']] ?? 0;
            $recordServiceAmount = $recordServicesData[$rates[$i]['recordRowId']] ?? 0;
            $recordChargesAmount = $recordChargesData[$rates[$i]['recordRowId']]['amount'] ?? 0;
            $totalAmount = $offeringAmount + $recordServiceAmount + $recordChargesAmount;
            if ($totalAmount > 0) {
                $rates[$i]['totalAmount'] = $totalAmount;
                $rates[$i]['titleNearTotalAmount'] = $recordChargesData[$rates[$i]['recordRowId']]['title'] ?? "";
                $rates[$i]['vehicleTypeTitle'] = $recordChargesData[$rates[$i]['recordRowId']]['vehicleTypeTitle'] ?? "";
            } else {
                unset($rates[$i]);
            }
        }
        $keys = array_column($rates, 'totalAmount');
        array_multisort($keys, SORT_ASC, $rates);
        $optimised = $this->getLeastRateAmountFromResult($rates[0] ?? []);
        if ($ajaxCall > 0) {
            if ($optimised['conversionRowId'] > 0) {
                $optimised = $this->ShowAllChargesListForOptimisedRatesWithConversion($optimised, $amountWithConversion ?? []);
            } else {
                $optimised = $this->showAllChargesListForOptimisedRates($optimised, $postData);
            }
            return $optimised;
        }
        $config = [];
        $config["base_url"] = base_url() . "rateenquiry/index";
        $config["total_rows"] = count($rates);
        $config["per_page"] = 5;
        $config["uri_segment"] = 3;
        $config['display_pages'] = true;
        $this->pagination->initialize($config);
        $page = ($this->uri->segment(3)) ? (int)$this->uri->segment(3) : 0;
        $perPage = $page + $config["per_page"];
        for ($page; $page < $perPage; $page++) {
            $serviceRowId = $rates[$page]['serviceRowId'] ?? 0;
            if ($serviceRowId === 0) {
                break;
            }
            $ratesResult[] = $rates[$page] ?? [];
        }
        return ['rates' => $ratesResult ?? [], 'serviceModeMaster' => $serviceModeMaster ?? [], 'optimised' => $optimised ?? []];
    }

    private function setWhereConditionsByPostData(array $postData): array
    {
        $pickupData = [$postData['pickupCountry'] ?? "", $postData['pickupState'] ?? "", $postData['pickupDistrict'] ?? "", $postData['pickupCity'] ?? "", $postData['pickupPincode'] ?? "", $postData['pickupSubDistrict'] ?? ""];
        $deliveryData = [$postData['consigneeCountry'] ?? "", $postData['consigneeState'] ?? "", $postData['pickupDistrict'] ?? "", $postData['consigneeCity'] ?? "", $postData['consigneePincode'] ?? "", $postData['consigneeSubDistrict'] ?? ""];
        $geoPickupData = [$postData['pickupPincode'] ?? "", $postData['pickupCity'] ?? "", $postData['pickupState'] ?? "", $postData['pickupCountry']];
        $geoDeliveryData = [$postData['consigneePincode'] ?? "", $postData['consigneeCity'] ?? "", $postData['consigneeState'] ?? "", $postData['consigneeCountry']];
        $modelType = $postData['modelType'] ?? "";
        $tariffType = $postData['tariffType'] ?? "";
        $data = [];
        $ajaxCall = $postData['ajaxCall'] ?? 0;
        if ($ajaxCall > 0) {
            $data['offeringType'] = $modelType;
        } elseif (in_array($modelType, ['FTL', 'LTL'])) {
            $data['offeringType'] = $modelType;
        }
        if ($ajaxCall > 0) {
            $data['service'] = $postData['service'];
        }
        if (in_array($tariffType, ['SELL', 'BUY'])) {
            $data['tarrifType'] = $tariffType;
        }
        $startDate = $postData['searchFromDate'] ?? "";
        $endDate = $postData['searchToDate'] ?? "";
        if (!in_array('', [$startDate, $endDate], true)) {
            $date1 = new DateTime($startDate);
            $date2 = new DateTime($endDate);
            $interval = date_diff($date1, $date2);
            $daysDifference = $interval->format('%a');
            if ($daysDifference > 0) {
                $data['daysCount'] = $daysDifference;
            }
        }
        $vehicleType = $postData['vehicleType'] ?? 0;
        if ($vehicleType > 0) {
            $vehicleProfileIds = $this->rateenquirymodel->getVehicleTypeProfileIds(['companyCodes' => $this->crossBorderCountries['companyCodes'], 'profileId' => $vehicleType]);
            if (!empty($vehicleProfileIds)) {
                $data['vehicleProfileIds'] = $vehicleProfileIds;
            }
            $data['vehicleType'] = $vehicleType;
        }
        $customerRowId = $postData['customerRowId'] ?? 0;
        if ($customerRowId > 0) {
            $customerProfileIds = $this->rateenquirymodel->getCustomerProfilesByCustomerId(['companyCodes' => $this->crossBorderCountries['companyCodes'], 'profileId' => $customerRowId]);
            if (!empty($customerProfileIds)) {
                $data['customerProfileIds'] = $customerProfileIds;
            }
        }
        $vehicleCapacity = $postData['vehicleCapacity'] ?? 0;
        if ($vehicleCapacity > 0) {
            $data['vehicleCapacity'] = $vehicleCapacity;
        }
        return ['pickupData' => $pickupData, 'deliveryData' => $deliveryData, 'geoPickupData' => $geoPickupData, 'geoDeliveryData' => $geoDeliveryData, 'conditions' => $data];
    }

    private function getTiersTotalAmountFromRateRecord(array $data, array $info): array
    {
        $rateTiers = $geoTiers = [];
        $postData = $info['postData'];
        $weight = $postData['searchWeight'] ?? 0;
        $vehicleType = $postData['vehicleType'] ?? 0;
        $ajaxCall = $postData['ajaxCall'] ?? 0;
        $orderTotalWeight = $postData['totalWeight'] ?? 0;
        $orderTotalVolume = $postData['totalVolume'] ?? 0;
        $totalKiloMeters = $postData['totalKiloMeters'] ?? 0;
        $totalMiles = $postData['totalMiles'] ?? 0;
        $pickupData = [$postData['pickupPincode'] ?? "", $postData['pickupCity'] ?? "", $postData['pickupState'] ?? "", $postData['pickupCountry']];
        $deliveryData = [$postData['consigneePincode'] ?? "", $postData['consigneeCity'] ?? "", $postData['consigneeState'] ?? "", $postData['consigneeCountry']];
        $geoPairs = $this->generateAddressPairsForGeoTier($info['pickupData'], $info['deliveryData']);
        foreach ($data as $eachLine) {
            if ($eachLine['tierId'] > 0) {
                $rateTiers[] = ['recordId' => $eachLine['recordId'], 'tierId' => $eachLine['tierId'], 'chargeId' => $eachLine['chargeId']];
            }
            if ($eachLine['geoTierId'] > 0) {
                $geoTiers[] = ['recordId' => $eachLine['recordId'], 'geoTierId' => $eachLine['geoTierId'], 'chargeId' => $eachLine['chargeId']];
            }
        }
        if (!empty($geoTiers)) {
            if ($ajaxCall > 0) {
                $geoTierData = $this->getGeoTiersDataWithCargos(['orderTotalWeight' => $orderTotalWeight, 'orderTotalVolume' => $orderTotalVolume, 'totalKiloMeters' => $totalKiloMeters, 'totalMiles' => $totalMiles, 'pickupData' => $pickupData, 'deliveryData' => $deliveryData], $geoTiers);
            } else {
                $geoTierData = $this->rateenquirymodel->getGeoTierDataFromSheet($geoPairs, ['data' => $geoTiers, 'weight' => $weight, 'userCurrency' => $this->userCurrency, 'userId' => $this->userId, 'vehicleType' => $vehicleType]);
            }
            if (!empty($geoTierData)) {
                foreach ($geoTierData as $eachRow) {
                    $message = $eachRow['message'] ?? "";
                    $billingUom = $eachRow['billingUom'] ?? "";
                    $title = $eachRow['geoTierId'] . " - " . $message . " " . $billingUom;
                    $geoTiersAmount[$eachRow['geoTierRowId']] = ['cost' => $eachRow['cost'], 'title' => $title, 'displayVehicleName' => $eachRow['displayVehicleName'] ?? "", 'vehicleTypeName' => $eachRow['vehicleTypeName'] ?? "", 'geoTierId' => $eachRow['geoTierId'], 'chargeId' => $eachLine['chargeId'] ?? ""];
                }
            }
        }
        if (!empty($rateTiers)) {
            if ($ajaxCall > 0) {
                $rateTierData = $this->getRateTiersDataWithCargos(['orderTotalWeight' => $orderTotalWeight, 'orderTotalVolume' => $orderTotalVolume, 'totalKiloMeters' => $totalKiloMeters, 'totalMiles' => $totalMiles], $rateTiers);
            } else {
                $rateTierData = $this->rateenquirymodel->getRateTierDataFromSheet(['data' => $rateTiers, 'weight' => $weight, 'userCurrency' => $this->userCurrency, 'userId' => $this->userId]);
            }
            if (!empty($rateTierData)) {
                foreach ($rateTierData as $eachRow) {
                    $message = $eachRow['message'] ?? "";
                    $billingUom = $eachRow['billingUom'] ?? "";
                    $title = $eachRow['tierId'] . " - " . $message . " " . $billingUom;
                    $rateTiersAmount[$eachRow['tierRowId']] = ['cost' => $eachRow['cost'], 'title' => $title, 'chargeId' => $eachRow['chargeId'] ?? ""];
                }
            }
        }
        return ['geoTiersAmount' => $geoTiersAmount ?? [], 'rateTiersAmount' => $rateTiersAmount ?? []];
    }

    private function getLeastRateAmountFromResult(array $data): array
    {
        if (empty($data)) {
            return [];
        }
        $optimised['rateServiceType'] = $data['rateServiceType'] ?? 0;
        $optimised['rateOfferingType'] = $data['rateOfferingType'] ?? "FTL";
        $optimised['totalAmount'] = $data['totalAmount'] ?? "";
        $optimised['offeringTariffType'] = $data['offeringTariffType'] ?? "";
        $optimised['customerProfileId'] = $data['customerProfileId'] ?? 0;
        $optimised['vendorProfileId'] = $data['vendorProfileId'] ?? 0;
        $optimised['serviceRowId'] = $data['serviceRowId'] ?? 0;
        $optimised['laneRowId'] = $data['laneRowId'] ?? 0;
        $optimised['offeringRowId'] = $data['offeringRowId'] ?? 0;
        $optimised['recordRowId'] = $data['recordRowId'] ?? 0;
        $optimised['rateRecordId'] = $data['rateRecordId'] ?? "";
        $optimised['rateOfferingId'] = $data['rateOfferingId'] ?? "";
        $optimised['conversionRowId'] = $data['conversionRowId'] ?? 0;
        if ($optimised['vendorProfileId'] > 0) {
            $optimised['vendor'] = $this->rateenquirymodel->getVendorProfileData($optimised['vendorProfileId']);
        }
        if ($optimised['customerProfileId'] > 0) {
            $optimised['customer'] = $this->rateenquirymodel->getCustomerProfileData($optimised['customerProfileId']);
        }
        return $optimised;
    }

    private function getTotalChargesAmount(array $offeringIds, array $recordIds, array $pickupData, array $deliveryData, array $postData): array
    {
        $ajaxCall = $postData['ajaxCall'] ?? 0;
        if ($ajaxCall > 0) {
            $amountDetails = $this->rateenquirymodel->getTotalChargesAmountWithCargos($offeringIds, $recordIds, $postData, $this->userCurrency, $this->userId);
        } else {
            $amountDetails = $this->rateenquirymodel->getTotalChargesAmount($offeringIds, $recordIds, $this->userCurrency, $this->userId);
        }
        $offering = $amountDetails['offering'] ?? [];
        $recordServices = $amountDetails['recordServices'] ?? [];
        $recordCharges = $amountDetails['recordCharges'] ?? [];
        $otherCurrencyCharges = $amountDetails['otherCurrencyCharges'] ?? [];
        $tiersData = $amountDetails['tiersData'] ?? [];
        $fafWithRateId = $amountDetails['fafWithRateId'] ?? [];
        $frtRecordChargesAmount = $amountDetails['frtRecordChargesAmount'] ?? [];
        $recordChargesData = $recordChargesInfo = $recordRowIds = [];
        foreach ($offering as $eachLine) {
            $offeringData[$eachLine['rateoffering_id']] = $eachLine['totalOfferingAmount'];
        }
        foreach ($recordServices as $eachLine) {
            $recordServicesData[$eachLine['raterecord_id']] = $eachLine['totalRecordServices'];
        }
        foreach ($recordCharges as $eachLine) {
            $othercharges = $otherCurrencyCharges[$eachLine['raterecord_id']] ?? 0;
            $recordChargesInfo[$eachLine['raterecord_id']] = ($othercharges + $eachLine['totalRecordCharges']);
            $recordRowIds[] = $eachLine['raterecord_id'];
        }
        if (!empty($tiersData)) {
            $tiersAmount = $this->getTiersTotalAmountFromRateRecord($tiersData, ['pickupData' => $pickupData, 'deliveryData' => $deliveryData, 'postData' => $postData]);
            $geoTiersAmount = $tiersAmount['geoTiersAmount'] ?? [];
            $rateTiersAmount = $tiersAmount['rateTiersAmount'] ?? [];
            foreach ($tiersData as $data) {
                $chargesId2[$data['recordId']][$data['geoTierId']] = $data['chargeId'];
            }
        }
        if ($ajaxCall > 0) {
            if (empty($recordChargesInfo) && !empty($tiersData)) {
                foreach ($tiersData as $eachTier) {
                    $recordChargesInfo[$eachTier['recordId']] = 0;
                }
            }
        } else {
            foreach ($tiersData as $eachRecord) {
                $recordId = $eachRecord['recordId'];
                if (in_array($recordId, $recordRowIds, true)) {
                    continue;
                }
                $recordChargesInfo[$recordId] = 0;
            }
        }
        $dummyGeoTiers = $dummyRateTiers = [];
        foreach ($recordChargesInfo as $key => $value) {
            $frtAmount = $frtRecordChargesAmount[$key] ?? 0;
            $title = $vehicleTypeTitle = "";
            foreach ($tiersData as $eachTier) {
                if ($key == $eachTier['recordId']) {
                    if ($eachTier['geoTierId'] > 0) {
                        $charge = $geoTiersAmount[$eachTier['geoTierId']]['chargeId'] ?? 0;
                        $chargeId = $chargesId2[$key][$eachTier['geoTierId']] ?? $charge;
                        if ($chargeId == self::FRT_CHARGE_ID) {
                            $frtAmount += $geoTiersAmount[$eachTier['geoTierId']]['cost'] ?? 0;
                        }
                        $value += $geoTiersAmount[$eachTier['geoTierId']]['cost'] ?? 0;
                        $title .= isset($geoTiersAmount[$eachTier['geoTierId']]['title']) ? $geoTiersAmount[$eachTier['geoTierId']]['title'] . ",&#10;" : "";
                        $displayVehicleName = $geoTiersAmount[$eachTier['geoTierId']]['displayVehicleName'] ?? 0;
                        $vehicleTypeName = $geoTiersAmount[$eachTier['geoTierId']]['vehicleTypeName'] ?? "";
                        $geoTierId = $geoTiersAmount[$eachTier['geoTierId']]['geoTierId'] ?? "";
                        if ($displayVehicleName > 0 && $vehicleTypeName != "") {
                            $vehicleTypeTitle .= "Vehicle Type is empty for " . $geoTierId . ",&#10;";
                        }
                        $dummyGeoTiers[] = $eachTier['geoTierId'];
                    }
                    if ($eachTier['tierId'] > 0) {
                        $value += $rateTiersAmount[$eachTier['tierId']]['cost'] ?? 0;
                        $title .= isset($rateTiersAmount[$eachTier['tierId']]['title']) ? $rateTiersAmount[$eachTier['tierId']]['title'] . ",&#10;" : "";
                        $charge2 = $rateTiersAmount[$eachTier['tierId']]['chargeId'] ?? 0;
                        $chargeId2 = $chargesId2[$key][$eachTier['tierId']] ?? $charge2;
                        if ($chargeId2 == self::FRT_CHARGE_ID) {
                            $frtAmount += $rateTiersAmount[$eachTier['tierId']]['cost'] ?? 0;
                        }
                    }
                    $dummyRateTiers[] = $eachTier['tierId'];
                }
            }
            $fafIds = $fafWithRateId[$key] ?? [];
            $surchargeData = 0;
            if ($frtAmount > 0 && !empty($fafIds)) {
                $fuelSurchargeData = $this->rateenquirymodel->calculateFuelSurchargeData(['ids' => $fafIds, 'amount' => $frtAmount, 'currency' => $this->userCurrency]);
                foreach ($fafIds as $eachId) {
                    $eachIdData = $fuelSurchargeData[$eachId] ?? [];
                    foreach ($eachIdData as $eachSurcharge) {
                        $value += $eachSurcharge['amount'];
                        $surchargeData += $eachSurcharge['amount'];
                    }
                }
            }
            $recordChargesData[$key]['amount'] = $value;
            $recordChargesData[$key]['title'] = $title;
            $recordChargesData[$key]['vehicleTypeTitle'] = $vehicleTypeTitle;
            $recordChargesData[$key]['fuelSurcharge'] = $surchargeData;
        }
        foreach ($tiersData as $singleTier) {
            $tierTitle = $tierVehicleTitle = "";
            $tierValue = 0;
            $checkStatus = 0;
            if ($singleTier['recordId'] > 0 && in_array($singleTier['recordId'], $recordIds, true)) {
                if ($singleTier['geoTierId'] > 0) {
                    if (!in_array($singleTier['geoTierId'], $dummyGeoTiers, true)) {
                        $tierValue += $geoTiersAmount[$singleTier['geoTierId']]['cost'] ?? 0;
                        $tierTitle .= isset($geoTiersAmount[$singleTier['geoTierId']]['title']) ? $geoTiersAmount[$singleTier['geoTierId']]['title'] . ",&#10;" : "";
                        $displayVehicleName = $geoTiersAmount[$singleTier['geoTierId']]['displayVehicleName'] ?? 0;
                        $vehicleTypeName = $geoTiersAmount[$singleTier['geoTierId']]['vehicleTypeName'] ?? "";
                        $geoTierId = $geoTiersAmount[$singleTier['geoTierId']]['geoTierId'] ?? "";
                        if ($displayVehicleName > 0 && $vehicleTypeName != "") {
                            $tierVehicleTitle .= "Vehicle Type is empty for " . $geoTierId . ",&#10;";
                        }
                        $dummyGeoTiers[] = $singleTier['geoTierId'];
                        $checkStatus = 1;
                    }
                }
                if ($singleTier['tierId'] > 0) {
                    if (!in_array($singleTier['tierId'], $dummyRateTiers, true)) {
                        $tierValue += $rateTiersAmount[$singleTier['tierId']]['cost'] ?? 0;
                        $tierTitle .= isset($rateTiersAmount[$eachTier['tierId']]['title']) ? $rateTiersAmount[$singleTier['tierId']]['title'] . ",&#10;" : "";
                        $dummyRateTiers[] = $singleTier['tierId'];
                        $checkStatus = 1;
                    }
                }
            }
            if ($checkStatus > 0) {
                $recordChargesData[$singleTier['recordId']]['amount'] = $tierValue;
                $recordChargesData[$singleTier['recordId']]['title'] = $tierTitle;
                $recordChargesData[$singleTier['recordId']]['vehicleTypeTitle'] = $tierVehicleTitle;
            }
        }
        return ["offeringData" => $offeringData ?? [], "recordServicesData" => $recordServicesData ?? [], "recordChargesData" => $recordChargesData ?? [], "tiersData" => $tiersData ?? []];
    }

    private function arrangeChargesOfAllRecords(array $charges, array $info): array
    {
        $vasIds = $chargeIds = $geoTiers = $rateTiers = $fuelSurchargeIds = $fuelSurchargeData = [];
        $titleNearTotalAmount = "";
        $weight = $info['weight'];
        $vehicleType = $info['vehicleType'];
        $offeringServices = $charges['offeringServices'] ?? [];
        $recordServices = $charges['recordServices'] ?? [];
        $recordCharges = $charges['recordCharges'] ?? [];
        $ajaxCall = $info['ajaxCall'] ?? 0;
        $totalAmount = $totalFrtAmount = $amountWithBasis = 0;
        $geoTierChargeId = $rateTierChargeId = [];
        foreach ($offeringServices as $eachRate) {
            $vasIds[] = $eachRate['ro_vas_id'];
            $chargeIds[] = $eachRate['ro_charge_id'];
            $chargeAmount = $eachRate['amount'];
            $chargeBasis = $eachRate['charge_basis'];
            $minimumAmount = $eachRate['min_amount'];
            if ($ajaxCall > 0) {
                $info['chargeAmount'] = $chargeAmount;
                if ($chargeBasis != "") {
                    $amountWithBasis = $this->multiplyAmountWithChargeBasis($chargeBasis, $info)['amount'];
                    if ($minimumAmount > $amountWithBasis) {
                        $amountWithBasis = $minimumAmount;
                    }
                }
            }
            $totalAmount += $ajaxCall > 0 ? $amountWithBasis : $chargeAmount;
            $chargesData[] = [
                'rowId' => $eachRate['id'],
                'vasId' => $eachRate['ro_vas_id'],
                'chargeId' => $eachRate['ro_charge_id'],
                'chargeBasis' => $chargeBasis,
                'minimumAmount' => $minimumAmount,
                'amount' => $chargeAmount,
                'currency' => $eachRate['currency'],
                'offeringId' => $eachRate['rateoffering_id'],
                'recordId' => 0,
                'geoTierId' => 0,
                'rateTierId' => 0,
                'fuelSurchargeId' => 0,
                'amountWithBasis' => $amountWithBasis
            ];
        }
        foreach ($recordServices as $eachRate) {
            $vasIds[] = $eachRate['raterecord_vas_id'];
            $chargeIds[] = $eachRate['raterecord_charge_id'];
            $chargeAmount = $eachRate['amount'];
            $chargeBasis = $eachRate['charge_basis'];
            $minimumAmount = $eachRate['min_amount'];
            if ($ajaxCall > 0) {
                $info['chargeAmount'] = $chargeAmount;
                if ($chargeBasis != "") {
                    $amountWithBasis = $this->multiplyAmountWithChargeBasis($chargeBasis, $info)['amount'];
                    if ($minimumAmount > $amountWithBasis) {
                        $amountWithBasis = $minimumAmount;
                    }
                }
            }
            $totalAmount += $ajaxCall > 0 ? $amountWithBasis : $chargeAmount;
            $chargesData[] = [
                'rowId' => $eachRate['id'],
                'vasId' => $eachRate['raterecord_vas_id'],
                'chargeId' => $eachRate['raterecord_charge_id'],
                'chargeBasis' => $chargeBasis,
                'minimumAmount' => $minimumAmount,
                'amount' => $chargeAmount,
                'currency' => $eachRate['currency'],
                'recordId' => $eachRate['raterecord_id'],
                'offeringId' => 0,
                'geoTierId' => 0,
                'rateTierId' => 0,
                'fuelSurchargeId' => 0,
                'amountWithBasis' => $amountWithBasis
            ];
        }
        foreach ($recordCharges as $eachRate) {
            $chargeAmount = $eachRate['amount'];
            $chargeBasis = $eachRate['charge_basis'];
            $minimumAmount = $eachRate['min_amount'];
            if ($ajaxCall > 0) {
                $info['chargeAmount'] = $chargeAmount;
                if ($chargeBasis != "") {
                    $amountWithBasis = $this->multiplyAmountWithChargeBasis($chargeBasis, $info)['amount'];
                    if ($minimumAmount > $amountWithBasis) {
                        $amountWithBasis = $minimumAmount;
                    }
                }
            }
            if (($eachRate['rr_charge_id'] == self::FRT_CHARGE_ID) && $eachRate['geo_tier_id'] == 0 && $eachRate['rr_tier_id'] == 0) {
                $totalFrtAmount += $ajaxCall > 0 ? $amountWithBasis : $chargeAmount;
            }
            $chargeIds[] = $eachRate['rr_charge_id'];
            if ($eachRate['geo_tier_id'] > 0) {
                $geoTiers[] = ['recordId' => $eachRate['raterecord_id'], 'geoTierId' => $eachRate['geo_tier_id'], 'chargeId' => $eachRate['rr_charge_id']];
                $geoTierChargeId[$eachRate['geo_tier_id']] = $eachRate['rr_charge_id'];
            } elseif ($eachRate['rr_tier_id'] > 0) {
                $rateTiers[] = ['recordId' => $eachRate['raterecord_id'], 'tierId' => $eachRate['rr_tier_id'], 'chargeId' => $eachRate['rr_charge_id']];
                $rateTierChargeId[$eachRate['rr_tier_id']] = $eachRate['rr_charge_id'];
            } else {
                $totalAmount += $ajaxCall > 0 ? $amountWithBasis : $chargeAmount;
            }
            if ($eachRate['fuel_surcharge_id'] > 0) {
                $fuelSurchargeIds[] = $eachRate['fuel_surcharge_id'];
            }
            $chargesData[] = [
                'rowId' => $eachRate['id'],
                'vasId' => 0,
                'chargeId' => $eachRate['rr_charge_id'],
                'chargeBasis' => $chargeBasis,
                'minimumAmount' => $minimumAmount,
                'amount' => $chargeAmount,
                'currency' => $eachRate['currency'],
                'recordId' => $eachRate['raterecord_id'],
                'offeringId' => 0,
                'geoTierId' => $eachRate['geo_tier_id'],
                'rateTierId' => $eachRate['rr_tier_id'],
                'fuelSurchargeId' => $eachRate['fuel_surcharge_id'],
                'amountWithBasis' => $amountWithBasis
            ];
            $exchangeAmount = $eachRate['exchangeAmount'] ?? 0;
        }
        if (!empty($geoTiers)) {
            if ($ajaxCall > 0) {
                $getGeoTierData = $this->getGeoTiersDataWithCargos($info, $geoTiers);
            } else {
                $geoPairs = $this->generateAddressPairsForGeoTier($info['pickupData'], $info['deliveryData']);
                $getGeoTierData = $this->rateenquirymodel->getGeoTierDataFromSheet($geoPairs, ['data' => $geoTiers, 'weight' => $weight, 'userCurrency' => $this->userCurrency, 'userId' => $this->userId, 'vehicleType' => $vehicleType]);
            }
            foreach ($getGeoTierData as $eachRow) {
                $totalAmount += $eachRow['cost'];
                $message = $eachRow['message'] ?? "";
                $billingUom = $eachRow['billingUom'] ?? "";
                $titleMessage = $message . " " . $billingUom;
                $chargeId = $geoTierChargeId[$eachRow['geoTierRowId']] ?? "";
                if ($chargeId == self::FRT_CHARGE_ID) {
                    $totalFrtAmount += $eachRow['cost'];
                }
                $geoTierData[$eachRow['geoTierRowId']] = ['geoTierId' => $eachRow['geoTierId'], 'amount' => $eachRow['cost'], 'currency' => $eachRow['currency'], 'billingUom' => $billingUom, 'message' => $titleMessage, 'displayVehicleName' => $eachRow['displayVehicleName'] ?? "", 'vehicleTypeName' => $eachRow['vehicleTypeName'] ?? "", 'minimumAmount' => $eachRow['minimumAmount'] ?? 0];
                $titleNearTotalAmount .= $eachRow['geoTierId'] . " - " . $message . " " . $billingUom . ",&#10;";
            }
        }
        if (!empty($rateTiers)) {
            if ($ajaxCall > 0) {
                $getRateTierData = $this->getRateTiersDataWithCargos($info, $rateTiers);
            } else {
                $getRateTierData = $this->rateenquirymodel->getRateTierDataFromSheet(['data' => $rateTiers, 'weight' => $weight, 'userCurrency' => $this->userCurrency, 'userId' => $this->userId]);
            }
            foreach ($getRateTierData as $eachRow) {
                $totalAmount += $eachRow['cost'];
                $message = $eachRow['message'] ?? "";
                $billingUom = $eachRow['billingUom'] ?? "";
                $chargeId = $rateTierChargeId[$eachRow['tierRowId']] ?? "";
                if ($chargeId == self::FRT_CHARGE_ID) {
                    $totalFrtAmount += $eachRow['cost'];
                }
                $rateTierData[$eachRow['tierRowId']] = ['tierId' => $eachRow['tierId'] ?? "", 'amount' => $eachRow['cost'] ?? 0, 'currency' => $eachRow['currency'] ?? "", 'billingUom' => $billingUom ?? "", 'message' => $message . " " . $billingUom, 'minimumAmount' => $eachRow['minimumAmount'] ?? 0];
                $titleNearTotalAmount .= $eachRow['tierId'] . " - " . $message . " " . $billingUom . ",&#10;";
            }
        }
        if (!empty($vasIds)) {
            $vasIds = array_filter($vasIds);
            $vasMasterData = $this->rateenquirymodel->getVasMasterData(array_unique($vasIds));
        }
        if (!empty($chargeIds)) {
            $chargeIds = array_filter($chargeIds);
            $chargeMasterData = $this->rateenquirymodel->getChargeMasterData(array_unique($chargeIds));
        }
        if (!empty($fuelSurchargeIds) && $totalFrtAmount > 0) {
            $fuelSurchargeIds = array_filter($fuelSurchargeIds);
            $fuelSurchargeData = $this->rateenquirymodel->calculateFuelSurchargeData(['ids' => array_unique($fuelSurchargeIds), 'amount' => $totalFrtAmount, 'currency' => $this->userCurrency]);
        }
        foreach ($fuelSurchargeIds as $eachId) {
            $eachIdData = $fuelSurchargeData[$eachId] ?? [];
            foreach ($eachIdData as $eachSurcharge) {
                $totalAmount += $eachSurcharge['amount'];
            }
        }

        return ['vasMasterData' => $vasMasterData ?? [], 'chargeMasterData' => $chargeMasterData ?? [], 'chargesData' => $chargesData ?? [], 'geoTierData' => $geoTierData ?? [], 'rateTierData' => $rateTierData ?? [], 'totalAmount' => $totalAmount, 'titleNearTotalAmount' => $titleNearTotalAmount, 'fuelSurchargeData' => $fuelSurchargeData, 'exchangeAmount' => $exchangeAmount ?? 0];
    }

    private function generateAddressPairs(array $pickupDetails, array $deliveryDetails): string
    {
        $geoHierarchy = [2, 3, 4, 5, 6, 7];
        $lanes = [];
        foreach ($pickupDetails as $pickupKey => $pickupvalue) {
            foreach ($deliveryDetails as $deliveryKey => $deliveryValue) {
                if (!in_array("", [$pickupvalue, $deliveryValue])) {
                    $lanes[] = '( lm.source_geo = "' . $geoHierarchy[$pickupKey] . '" AND ' . 'lm.source' . ' = "' . $pickupvalue . '" AND lm.destination_geo="' . $geoHierarchy[$deliveryKey] . '" AND ' . 'lm.destination' . ' = "' . $deliveryValue . '")';
                }
            }
        }
        return implode(' OR ', $lanes);
    }


    public function viewAllratesDetails(): void
    {
        $postData = $this->input->post(null, true);
        $data['rateServiceId'] = $postData['rateServiceId'];
        $data['rateOfferingId'] = $postData['rateOfferingId'];
        $data['rateRecordId'] = $postData['rateRecordId'];
        $data['rateServiceName'] = $postData['rateServiceName'];
        $data['rateOfferingName'] = $postData['rateOfferingName'];
        $data['rateRecordName'] = $postData['rateRecordName'];
        $data['userCurrency'] = $this->userCurrency;
        $pickupData = [$postData['ratePickupPincode'] ?? "", $postData['ratePickupCity'] ?? "", $postData['ratePickupState'] ?? "", $postData['ratePickupCountry']];
        $deliveryData = [$postData['rateConsigneePincode'] ?? "", $postData['rateConsigneeCity'] ?? "", $postData['rateConsigneeState'] ?? "", $postData['rateConsigneeCountry']];
        $data['rateDetails'] = $this->arrangeChargesOfAllRecords($this->rateenquirymodel->getAllServicesAndCharges([$postData['rateOfferingRowId']], [$postData['rateRecordRowId']], $this->userCurrency, $this->userId), ['pickupData' => $pickupData, 'deliveryData' => $deliveryData, 'weight' => $postData['rateSearchWeight'] ?? 0, 'vehicleType' => $postData['rateSearchVehicleType'] ?? 0]);
        $data = $this->rateenquirymodel->getAdditionalRatesData($postData, $data);
        echo $this->load->view('rateenquiry/advanceview', $data, true);
    }

    public function getProfileDataById(): void
    {
        $postData = $this->input->post(null, true);
        $data['tariffType'] = $postData['tariffType'] ?? 0;
        $data['row'] = $postData['row'] ?? 0;
        if ($data['tariffType'] > 0) {
            $customerProfileData = $this->rateenquirymodel->getCustomerDataFromProfileId($postData['profileId'] ?? 0);
        } else {
            $vendorProfileData = $this->rateenquirymodel->getVendorDataFromProfileId($postData['profileId'] ?? 0);
        }
        $data['vendorProfileData'] = $vendorProfileData ?? [];
        $data['customerProfileData'] = $customerProfileData ?? [];
        echo $this->load->view("rateenquiry/profilePopupView", $data, true);
    }

    private function showAllChargesListForOptimisedRates(array $optimised, array $postData): array
    {
        $pickupData = [$postData['pickupPincode'] ?? "", $postData['pickupCity'] ?? "", $postData['pickupState'] ?? "", $postData['pickupCountry']];
        $deliveryData = [$postData['consigneePincode'] ?? "", $postData['consigneeCity'] ?? "", $postData['consigneeState'] ?? "", $postData['consigneeCountry']];
        $rateDetails = $this->arrangeChargesOfAllRecords(
            $this->rateenquirymodel->getAllServicesAndCharges([$optimised['offeringRowId']], [$optimised['recordRowId']], $this->userCurrency, $this->userId, $postData['vasData'] ?? [], $postData['ajaxCall'] ?? 0),
            ['pickupData' => $pickupData, 'deliveryData' => $deliveryData, 'weight' => $postData['rateSearchWeight'] ?? 0, 'vehicleType' => $postData['rateSearchVehicleType'] ?? 0, 'orderTotalWeight' => $postData['totalWeight'] ?? 0, 'orderTotalVolume' => $postData['totalVolume'] ?? 0, 'orderTotalQuantity' => $postData['totalQuantity'] ?? 0, 'totalMiles' => $postData['totalMiles'] ?? 0, 'totalKiloMeters' => $postData['totalKiloMeters'] ?? 0, 'ajaxCall' => $postData['ajaxCall'] ?? 0]
        );
        $ajaxCall = $postData['ajaxCall'] ?? 0;
        if (!empty($rateDetails)) {
            $vasMasterData = $rateDetails['vasMasterData'] ?? [];
            $chargeMasterData = $rateDetails['chargeMasterData'] ?? [];
            $chargesData = $rateDetails['chargesData'] ?? [];
            $geoTierData = $rateDetails['geoTierData'] ?? [];
            $rateTierData = $rateDetails['rateTierData'] ?? [];
            $fuelSurchargeData = $rateDetails['fuelSurchargeData'] ?? [];
            $recordId = $optimised['rateRecordId'];
            $rateOfferingId = $optimised['rateOfferingId'];
            foreach ($chargesData as $eachLine) {
                $chargeBasis = $eachLine['chargeBasis'];
                if ($chargeBasis == "FRT") {
                    $fuelSurchargeId = $eachLine['fuelSurchargeId'] ?? 0;
                    if ($fuelSurchargeId > 0) {
                        $fuelSurchargeIdData = $fuelSurchargeData[$fuelSurchargeId] ?? [];
                        foreach ($fuelSurchargeIdData as $eachSurcharge) {
                            $individualCharges[] = ['rateId' => $recordId, 'vasId' => '', 'chargeId' => 'FAF', 'amount' => $eachSurcharge['amount'] . " " . $eachSurcharge['currency'], 'amountWithBasis' => round($eachSurcharge['amount'], 2), 'currency' => $eachSurcharge['currency'], 'minimumAmount' => 0];
                        }
                    }
                } else {
                    $geoTierId = $eachLine['geoTierId'];
                    $rateTierId = $eachLine['rateTierId'];
                    $amountWithBasis = $eachLine['amountWithBasis'] ?? 0;
                    if ($ajaxCall > 0) {
                        if ($amountWithBasis > 0) {
                            $amount = $amountWithBasis . " " . $eachLine['currency'];
                        }
                    } else {
                        $amount = $eachLine['amount'] . " " . $eachLine['currency'];
                    }
                    $rateRecordId = $recordId;
                    if ($geoTierId > 0) {
                        if (empty($geoTierData[$geoTierId])) {
                            continue;
                        }
                        $rateRecordId = $geoTierData[$geoTierId]['geoTierId'] ?? $rateRecordId;
                        $amount = $geoTierData[$geoTierId]['amount'] . " " . $geoTierData[$geoTierId]['currency'];
                        $amountWithBasis = $geoTierData[$geoTierId]['amount'];
                    }
                    if ($rateTierId > 0) {
                        if (empty($rateTierData[$rateTierId])) {
                            continue;
                        }
                        $rateRecordId = $rateTierData[$rateTierId]['tierId'] ?? $rateRecordId;
                        $amount = $rateTierData[$rateTierId]['amount'] . " " . $rateTierData[$rateTierId]['currency'];
                        $amountWithBasis = $rateTierData[$rateTierId]['amount'];
                    }
                    $individualCharges[] = ['rateId' => $eachLine['offeringId'] > 0 ? $rateOfferingId : $rateRecordId, 'vasId' => $eachLine['vasId'] > 0 ? $vasMasterData[$eachLine['vasId']]['vasId'] : "", 'chargeId' => $eachLine['chargeId'] > 0 ? $chargeMasterData[$eachLine['chargeId']]['chargeCode'] : "", 'amount' => $amount ?? 0, 'minimumAmount' => $eachLine['minimumAmount'] ?? 0, 'amountWithBasis' => round($amountWithBasis, 2), 'currency' => $eachLine['currency']];
                }
            }
        }
        $optimised['individualCharges'] = $individualCharges ?? [];
        return $optimised;
    }

    private function multiplyAmountWithChargeBasis(string $chargeBasis, array $info): array
    {
        $orderTotalWeight = $info['orderTotalWeight'] ?? 0;
        $orderTotalVolume = $info['orderTotalVolume'] ?? 0;
        $orderTotalQuantity = $info['orderTotalQuantity'] ?? 0;
        $chargeAmount = $info['chargeAmount'] ?? 0;
        $totalMiles = $info['totalMiles'] ?? 0;
        $totalKiloMeters = $info['totalKiloMeters'] ?? 0;
        if (in_array(strtoupper($chargeBasis), self::WEIGHT_CHARGE_BASIS)) {
            $amountWithBasis = ($chargeAmount * $orderTotalWeight);
        }
        if (in_array(strtoupper($chargeBasis), self::VOLUME_CHARGE_BASIS)) {
            $amountWithBasis = ($chargeAmount * $orderTotalVolume);
        }
        if (in_array(strtoupper($chargeBasis), self::QUANTITY_CHARGE_BASIS)) {
            $amountWithBasis = ($chargeAmount * $orderTotalQuantity);
        }
        if (in_array(strtoupper($chargeBasis), self::KM_CHARGE_BASIS)) {
            $amountWithBasis = ($chargeAmount * $totalKiloMeters);
        }
        if (in_array(strtoupper($chargeBasis), self::MILES_CHARGE_BASIS)) {
            $amountWithBasis = ($chargeAmount * $totalMiles);
        }
        if (in_array(strtoupper($chargeBasis), self::TRIP_CHARGE_BASIS)) {
            $amountWithBasis = $chargeAmount;
        }
        if (in_array(strtoupper($chargeBasis), self::LDM_CHARGE_BASIS)) {
            $amountWithBasis = 0;
        }
        return ['amount' => $amountWithBasis ?? $chargeAmount];
    }

    private function getGeoTiersDataWithCargos(array $info, array $geoTiers): array
    {
        $pickupData = $info['pickupData'] ?? [];
        $deliveryData = $info['deliveryData'] ?? [];
        $orderTotalWeight = $info['orderTotalWeight'] ?? 0;
        $orderTotalVolume = $info['orderTotalVolume'] ?? 0;
        $totalMiles = $info['totalMiles'] ?? 0;
        $totalKiloMeters = $info['totalKiloMeters'] ?? 0;
        $geoPairs = $this->generateAddressPairsForGeoTier($pickupData, $deliveryData);
        $getGeoTierData = $this->rateenquirymodel->getGeoTierSheetsDataForCargo($geoPairs, ['data' => $geoTiers, 'orderTotalWeight' => $orderTotalWeight, 'orderTotalVolume' => $orderTotalVolume, 'totalMiles' => $totalMiles, 'totalKiloMeters' => $totalKiloMeters, 'userCurrency' => $this->userCurrency, 'userId' => $this->userId]);
        foreach ($getGeoTierData as $eachRow) {
            $geoTierData[$eachRow['geoTierRowId']] = ['geoTierId' => $eachRow['geoTierId'], 'cost' => $eachRow['cost'], 'currency' => $eachRow['currency'], 'billingUom' => $eachRow['billingUom'], 'minimumAmount' => $eachRow['minimumAmount'] ?? 0, 'geoTierRowId' => $eachRow['geoTierRowId'], 'chargeId' => $eachRow['chargeId']];
        }
        return $geoTierData ?? [];
    }

    private function getRateTiersDataWithCargos(array $info, array $rateTiers): array
    {
        $orderTotalWeight = $info['orderTotalWeight'] ?? 0;
        $orderTotalVolume = $info['orderTotalVolume'] ?? 0;
        $totalMiles = $info['totalMiles'] ?? 0;
        $totalKiloMeters = $info['totalKiloMeters'] ?? 0;
        $getRateTierData = $this->rateenquirymodel->getRateTierDataFromSheetForCargo(['data' => $rateTiers, 'orderTotalWeight' => $orderTotalWeight, 'orderTotalVolume' => $orderTotalVolume, 'totalKiloMeters' => $totalKiloMeters, 'totalMiles' => $totalMiles, 'userCurrency' => $this->userCurrency, 'userId' => $this->userId]);
        foreach ($getRateTierData as $eachRow) {
            $rateTierData[$eachRow['tierRowId']] = ['tierId' => $eachRow['tierId'], 'cost' => $eachRow['cost'], 'currency' => $eachRow['currency'], 'billingUom' => $eachRow['billingUom'], 'minimumAmount' => $eachRow['minimumAmount'] ?? 0, 'tierRowId' => $eachRow['tierRowId']];
        }
        return $rateTierData ?? [];
    }

    private function generateAddressPairsForGeoTier(array $pickupData, array $deliveryData): array
    {
        $geoHierarchy = ['POSTAL CODE', 'CITY', 'PROVINCE', 'COUNTRY'];
        foreach ($pickupData as $pickupKey => $pickupValue) {
            foreach ($deliveryData as $deliveryKey => $deliveryValue) {
                if ($pickupKey == $deliveryKey) {
                    if (!in_array("", [$pickupValue, $deliveryValue])) {
                        $addressPair1[$geoHierarchy[$deliveryKey]] = '(' . 's.geo_from' . ' = "' . $pickupValue . '" AND ' . 's.geo_to' . ' = "' . $deliveryValue . '")';
                    }
                }
            }
        }
        foreach ($pickupData as $pickupValue) {
            foreach ($deliveryData as $deliveryValue) {
                if (!in_array("", [$pickupValue, $deliveryValue])) {
                    $addressPair2[] = '(' . 's.geo_from' . ' = "' . $pickupValue . '" AND ' . 's.geo_to' . ' = "' . $deliveryValue . '")';
                }
            }
        }
        return ['pair1' => $addressPair1 ?? [], 'pair2' => $addressPair2 ?? []];
    }

    private function getRatesUsingConversionFactor(array $data): array
    {
        $amountDetails = $this->rateenquirymodel->getAllRatesAmountWithConversion($data);
        $tiersData = $amountDetails['tiersData'];
        $fafDetails = $amountDetails['fafDetails'];
        $postData = $data['postData'];
        $rateTiers = $geoTiers = $rateTiersData = [];
        foreach ($tiersData as $eachTier) {
            if ($eachTier['tierId'] > 0) {
                $rateTiers[] = ['TierId' => $eachTier['tierId'], 'offeringId' => $eachTier['offeringId'], 'recordId' => $eachTier['recordId'], 'chargeId' => $eachTier['chargeId']];
            } else {
                $geoTiers[] = ['geoTierId' => $eachTier['geoTierId'], 'offeringId' => $eachTier['offeringId'], 'recordId' => $eachTier['recordId'], 'chargeId' => $eachTier['chargeId']];
            }
        }
        if (!empty($geoTiers)) {
            $geoPairs = $this->generateAddressPairsForGeoTier($data['geoPickupData'], $data['geoDeliveryData']);
            $finalGeoTierData = $this->rateenquirymodel->getGeoTiersAmountWithConversion(['geoPairs' => $geoPairs, 'geoTiers' => $geoTiers, 'amountDetails' => $amountDetails, 'totalVolume' => $postData['totalVolume'], 'totalWeight' => $postData['totalWeight']]);
            $amountDetails = $finalGeoTierData['amountDetails'];
            $geoTiersData = $finalGeoTierData['geoTiers'];
        }
        $fafIds = $fafDetails['fafIds'];
        $fafWithRateId = $fafDetails['fafWithRateId'];
        if (!empty($fafIds)) {
            $amountDetails['offeringData'] = $this->getFAFRatesAmountWithConversion(['records' => $amountDetails['offeringData'], 'fafWithRateId' => $fafWithRateId]);
            $amountDetails['recordServicesData'] = $this->getFAFRatesAmountWithConversion(['records' => $amountDetails['recordServicesData'], 'fafWithRateId' => $fafWithRateId]);
            $amountDetails['recordChargesData'] = $this->getFAFRatesAmountWithConversion(['records' => $amountDetails['recordChargesData'], 'fafWithRateId' => $fafWithRateId, 'geoTiersData' => $geoTiersData]);
        }
        return ['amountDetails' => $amountDetails, 'geoTiersData' => $geoTiersData ?? [], 'rateTiersData' => $rateTiersData ?? [], 'tiersData' => $tiersData];
    }

    private function getFAFRatesAmountWithConversion(array $data): array
    {
        $records = $data['records'];
        $geoTiersData = $data['geoTiersData'] ?? [];
        $fafWithRateId = $data['fafWithRateId'];
        if (empty($records)) {
            return [];
        }
        foreach ($records as $eachLine) {
            $chargesWithoutFaf = $eachLine['chargesWithoutFaf'];
            $rowId = $eachLine['rowId'];
            $iMax = count($chargesWithoutFaf);
            $frtTotalAmount = $totalAmount = 0;
            $charges = [];
            $tiersData = $geoTiersData[$rowId] ?? [];
            for ($i = 0; $i < $iMax; $i++) {
                $charges[] = $chargesWithoutFaf[$i];
                $totalAmount += $chargesWithoutFaf[$i]['amount'];
                if ($chargesWithoutFaf[$i]['chargeId'] == self::FRT_CHARGE_ID) {
                    $frtTotalAmount += $chargesWithoutFaf[$i]['amount'];
                }
                if ($i == ($iMax - 1)) {
                    foreach ($tiersData as $eachTierData) {
                        if ($eachTierData['chargeId'] == self::FRT_CHARGE_ID) {
                            $frtTotalAmount += $eachTierData['amount'];
                        }
                        $totalAmount += $eachTierData['amount'];
                    }
                    if ($frtTotalAmount > 0 && !empty($fafWithRateId[$rowId] ?? [])) {
                        $fuelSurchargeData = $this->rateenquirymodel->calculateFuelSurchargeData(['ids' => $fafWithRateId[$rowId], 'amount' => $frtTotalAmount, 'currency' => $this->userCurrency]);
                        foreach ($fafWithRateId[$rowId] as $eachId) {
                            $eachIdData = $fuelSurchargeData[$eachId] ?? [];
                            foreach ($eachIdData as $eachSurcharge) {
                                $totalAmount += $eachSurcharge['amount'];
                                $charges[] = ['minimumAmount' => 0, 'amount' => $eachSurcharge['amount'], 'currency' => $eachSurcharge['currency'], 'vasId' => 0, 'chargeId' => 'FAF', 'rowId' => $rowId, 'fafId' => $eachSurcharge['fafId']];
                            }
                        }
                    }
                }
            }
            $records[$rowId]['charges'] = $charges;

            $records[$rowId]['amount'] = $totalAmount;
        }
        return $records;
    }


    private function arrangeConversionAndNonConversionAmountIntoSingleArray(array $amountDetails, array $amountWithConversion): array
    {
        if (empty($amountWithConversion)) {
            return $amountDetails;
        }
        $amountDetailsWithConversion = $amountWithConversion['amountDetails'];
        $offeringData = $amountDetailsWithConversion['offeringData'];
        foreach ($offeringData as $key => $value) {
            $amountDetails['offeringData'][$key] = $value['amount'];
        }
        $recordServicesData = $amountDetailsWithConversion['recordServicesData'];
        foreach ($recordServicesData as $key => $value) {
            $amountDetails['recordServicesData'][$key] = $value['amount'];
        }
        $recordChargesData = $amountDetailsWithConversion['recordChargesData'];
        foreach ($recordChargesData as $key => $value) {
            $amountDetails['recordChargesData'][$key] = ['amount' => $value['amount'], 'title' => '', 'vehicleTypeTitle' => '', 'fuelSurcharge' => '0'];
        }
        return $amountDetails;
    }

    private function ShowAllChargesListForOptimisedRatesWithConversion(array $optimised, array $amountWithConversion): array
    {
        $vasMasterData = $chargeMasterData = [];
        $chargeIds[] = $vasIds = $individualCharges = [];
        $data = $amountWithConversion['amountDetails'];
        $tiersData = $amountWithConversion['tiersData'];
        $geoTiersData = $amountWithConversion['geoTiersData'];
        $offeringData = $data['offeringData'];
        $offeringRowId = $optimised['offeringRowId'];
        $recordRowId = $optimised['recordRowId'];
        foreach ($offeringData[$offeringRowId]['charges'] ?? [] as $eachCharge) {
            $chargeIds[] = $eachCharge['chargeId'];
            $vasIds[] = $eachCharge['vasId'];
            $individualCharges[] = ['rateId' => $eachCharge['fafId'] ?? ($optimised['rateOfferingId'] ?? ""), 'vasRowId' => $eachCharge['vasId'], 'chargeRowId' => $eachCharge['chargeId'], 'amount' => $eachCharge['amount'], 'minimumAmount' => $eachCharge['minimumAmount'] ?? 0, 'amountWithBasis' => round($eachCharge['amount'], 2), 'currency' => $eachCharge['currency']];
        }
        $recordServicesData = $data['recordServicesData'];
        foreach ($recordServicesData[$recordRowId]['charges'] ?? [] as $eachCharge) {
            $chargeIds[] = $eachCharge['chargeId'];
            $vasIds[] = $eachCharge['vasId'];
            $individualCharges[] = ['rateId' => $eachCharge['fafId'] ?? ($optimised['rateRecordId'] ?? ""), 'vasRowId' => $eachCharge['vasId'], 'chargeRowId' => $eachCharge['chargeId'], 'amount' => $eachCharge['amount'], 'minimumAmount' => $eachCharge['minimumAmount'] ?? 0, 'amountWithBasis' => round($eachCharge['amount'], 2), 'currency' => $eachCharge['currency']];
        }
        $recordChargesData = $data['recordChargesData'];
        foreach ($recordChargesData[$recordRowId]['charges'] ?? [] as $eachCharge) {
            $chargeIds[] = $eachCharge['chargeId'];
            $individualCharges[] = ['rateId' => $eachCharge['fafId'] ?? ($optimised['rateRecordId'] ?? ""), 'vasRowId' => 0, 'chargeRowId' => $eachCharge['chargeId'], 'amount' => $eachCharge['amount'], 'minimumAmount' => $eachCharge['minimumAmount'] ?? 0, 'amountWithBasis' => round($eachCharge['amount'], 2), 'currency' => $eachCharge['currency']];
        }
        foreach ($geoTiersData[$recordRowId] ?? [] as $eachCharge) {
            $chargeIds[] = $eachCharge['chargeId'];
            $individualCharges[] = ['rateId' => $eachCharge['geoTierId'] ?? "", 'vasRowId' => 0, 'chargeRowId' => $eachCharge['chargeId'], 'amount' => $eachCharge['amount'], 'minimumAmount' => $eachCharge['minimumAmount'] ?? 0, 'amountWithBasis' => round($eachCharge['amount'], 2), 'currency' => $eachCharge['currency']];
        }
        if (!empty($vasIds)) {
            $vasIds = array_filter($vasIds);
            $vasMasterData = $this->rateenquirymodel->getVasMasterData(array_unique($vasIds));
        }
        if (!empty($chargeIds)) {
            $chargeIds = array_filter($chargeIds);
            $chargeMasterData = $this->rateenquirymodel->getChargeMasterData(array_unique($chargeIds));
        }
        foreach ($individualCharges as $eachLine) {
            $eachLine['chargeId'] = $chargeMasterData[$eachLine['chargeRowId']]['chargeCode'] ?? "";
            $eachLine['vasId'] = $vasMasterData[$eachLine['vasRowId']]['vasId'] ?? "";
            $optimised['individualCharges'][] = $eachLine;
        }
        return $optimised;
    }

}
