<?php

defined('BASEPATH') or exit('No direct script access allowed');

use App\Integrations\Mft\MftClient;
use App\Libraries\infrastructure\ScheduledJobLoggerService;

class UniqloMftSwiftlogScheduledJob extends CI_Controller
{
    private const FLAG_MOVE_TO_BACKUP = true;
    private const LOG_ENTITYNAME = 'UniqloMftSwiftlogScheduledJob';

    private const MFT_UNIQLO_SOURCE_FOLDER = '/pub/outbound/uniqlo/etnshipmentready/';
    private const MFT_UNIQLO_BACKUP_FOLDER = '/pub/outbound/uniqlo/backups/etnshipmentready/';
    private const ETN_UNIQLO_DESTINATION_FOLDER = './assets/swiftlog/etnshipmentready/';

    private ScheduledJobLoggerService $scheduledJobLoggerService;
    private MftClient $mftClient;

    public function __construct()
    {
        parent::__construct();
        set_error_handler(['UniqloMftSwiftlogScheduledJob', 'error'], E_ALL);

        $this->load->model('common');
        $this->load->helper('log_helper');
        $this->scheduledJobLoggerService = ScheduledJobLoggerService::instance(get_class($this));
        $this->mftClient = new MftClient($this->scheduledJobLoggerService, self::LOG_ENTITYNAME);

        if (!file_exists(self::ETN_UNIQLO_DESTINATION_FOLDER)) {
            mkdir(self::ETN_UNIQLO_DESTINATION_FOLDER, 0700, true);
        }
    }

    public function copy(int $copyLimit): void
    {
        $this->scheduledJobLoggerService->startJob();
        $this->mftClient->connect();
        $fileList = $this->mftClient->getFileList(self::MFT_UNIQLO_SOURCE_FOLDER, ['xml']);
        $filesInTotal = count($fileList);
        $copiedFiles = 0;

        if ($filesInTotal == 0) {
            $this->scheduledJobLoggerService->log(self::LOG_ENTITYNAME, 'No files found on MFT.');

            return;
        }

        $this->scheduledJobLoggerService->log(self::LOG_ENTITYNAME, sprintf('Number of files to get: %d', $filesInTotal));

        if ($filesInTotal > $copyLimit) {
            $this->scheduledJobLoggerService->log(
                self::LOG_ENTITYNAME,
                sprintf('Number of files (%d) exceeds the limit (%d). Only %d will be imported.', $filesInTotal, $copyLimit, $copyLimit)
            );

            $fileList = array_slice($fileList, 0, $copyLimit);
        }

        foreach ($fileList as $fileName) {
            $etnDestinationFilePath = $this->getFileDestinationPath($fileName);
            $mftSourceFilePath = $this->getFileMftSourcePath($fileName);

            if (file_exists($etnDestinationFilePath)) {
                $this->scheduledJobLoggerService->log(
                    self::LOG_ENTITYNAME,
                    sprintf('%s - file already exists in assets, will be overridden', $fileName)
                );
                unlink($etnDestinationFilePath);
            }

            $originalFileSize = $this->mftClient->filesize($mftSourceFilePath);

            try {
                $this->mftClient->get($mftSourceFilePath, $etnDestinationFilePath);
            } catch (\Error $error) {
                $this->scheduledJobLoggerService->log(self::LOG_ENTITYNAME, $fileName . ' - SFTP Error ' . $error->getMessage());
                unlink($etnDestinationFilePath);

                $this->summary(false, $copiedFiles);
                return;
            }

            if (false === $this->validateImport($fileName, $etnDestinationFilePath, $originalFileSize)) {
                continue;
            }

            if (self::FLAG_MOVE_TO_BACKUP) {
                $mftBackupFilePath = $this->getFileMftBackupPath($fileName);

                $this->mftClient->put($mftBackupFilePath, $this->mftClient->get($mftSourceFilePath));
                $this->mftClient->delete($mftSourceFilePath);
            }

            $copiedFiles++;
        }

        $this->summary(true, $copiedFiles);
        unset($this->mftClient);
    }

    private function summary(bool $isSuccess, int $copiedFiles): void
    {
        $this->scheduledJobLoggerService->log(self::LOG_ENTITYNAME, sprintf('Successfully imported %d files', $copiedFiles));
        $this->scheduledJobLoggerService->finishJob($isSuccess);

        if ($isSuccess) {
            echo sprintf('%s - Success' . PHP_EOL, self::LOG_ENTITYNAME);
        } else {
            echo sprintf('%s - Failed due to SFTP errors' . PHP_EOL, self::LOG_ENTITYNAME);
        }
    }

    private function validateImport(string $fileName, string $etnDestinationFilePath, int $originalFileSize): bool
    {
        $etnFileSize = filesize($etnDestinationFilePath);

        if ($etnFileSize < $originalFileSize) {
            $this->scheduledJobLoggerService->log(
                self::LOG_ENTITYNAME,
                sprintf('%s has lower size than the original file on MFT. Will be removed.', $fileName)
            );
            unlink($etnDestinationFilePath);
            return false;
        }

        if (false === simplexml_load_file($etnDestinationFilePath)) {
            $this->scheduledJobLoggerService->log(
                self::LOG_ENTITYNAME,
                sprintf('%s is not a valid XML. Will be removed.', $fileName)
            );
            unlink($etnDestinationFilePath);
            return false;
        }

        return true;
    }

    private function getFileDestinationPath(string $filename): string
    {
        return self::ETN_UNIQLO_DESTINATION_FOLDER . $filename;
    }

    private function getFileMftSourcePath(string $filename): string
    {
        return self::MFT_UNIQLO_SOURCE_FOLDER . $filename;
    }

    private function getFileMftBackupPath(string $filename): string
    {
        return self::MFT_UNIQLO_BACKUP_FOLDER . $filename;
    }

    /**
     * SFTP implementations throws user_error('Expected SSH_FXP_HANDLE or SSH_FXP_STATUS'); which is being handled by this method
     */
    public static function error($code, $string, $file, $line, $context)
    {
        throw new \Error(sprintf('%s - %s', $code, $string));
    }
}
