<?php
if (!defined('BASEPATH')) {
    exit('No direct script access allowed');
}
header('Content-Type: application/json');

use App\ApiRequest\Printable;

class Shipment extends CI_Controller
{
    private const SUCCESS_CODE = 200;
    private const PARAMS_ERROR_CODE = 400;
    private const VALIDATION_ERROR_CODE = 401;
    private const SERVER_ERROR_CODE = 500;

    function __construct()
    {
        parent::__construct();
    }

    private function validateRequest(): bool
    {
        $headers = $this->input->request_headers();
        if (null === $headers['X-Kn-Api-Key']) {
            return false;
        } else {
            if ($headers['X-Kn-Api-Key'] !== WECHAT_AUTHORIZATION_PASSWORD) {
                return false;
            }
        }
        return true;
    }

    public function getShipmentStatus(): void
    {
        ob_start();
        $data = "";
        $code = SELF::SUCCESS_CODE;
        $message = "ok";
        if (!$this->validateRequest()) {
            $code = SELF::VALIDATION_ERROR_CODE;
            $message = "User Account does not exist / Disabled, please reach out to KN Team.";
        } elseif (null === $this->input->get('search_key') || (!(null === $this->input->get('search_key')) && empty($this->input->get('search_key')))) {
            $code = SELF::PARAMS_ERROR_CODE;
            $message = "No Shipment(s) found with given ID.";
        } else {
            $searchKey = $this->input->get('search_key', true);
            $result = $this->getShipmentsData($searchKey);
            if (!empty($result)) {
                $data = $result;
            } else {
                $code = SELF::PARAMS_ERROR_CODE;
                $message = "No Shipment(s) found with given ID.";
            }
        }
        $response = ["data" => $data, "code" => $code, "msg" => $message];
        ob_end_clean();
        echo json_encode(Printable::printable($response));
    }

    private function getShipmentsData(string $searchKey): array
    {
        $response = [];
        $this->load->model('wechatmodel', 'chat');
        $shipmentsData = $this->chat->collectShipments($searchKey);
        if ($shipmentsData) {
            foreach ($shipmentsData as $data) {
                $pickUpDate = $departureDate = $arrivedDate = "";
                if ((int)$data['is_picked'] === 1) {
                    $pickUpDate = $this->chat->getShipmentStopStatus($data['id'], '0500');
                    $pickUpDate = $this->dateCustomFormat($pickUpDate);
                }
                if ((int)$data['is_departure'] === 1) {
                    $departureDate = $this->chat->getShipmentStopStatus($data['id'], '0191');
                    $departureDate = $this->dateCustomFormat($departureDate);
                }

                if ((int)$data['is_arrived'] === 1) {
                    $arrivedDate = $this->chat->getShipmentStopStatus($data['id'], '0192');
                    $arrivedDate = $this->dateCustomFormat($arrivedDate);
                }
                $estimatedArrival = !empty($data['drop_endtime']) ? $this->dateCustomFormat($data['drop_endtime']) : "";
                $response[] = [
                    "bookingCode" => $data['order_id'],
                    "deliveryNote" => $data['ref_value'],
                    "estimatedArrival" => $estimatedArrival,
                    "currentLocation" => $this->findOrderCurrentLocation($data['id']),
                    "from" => $data['from_location'],
                    "to" => $data['to_location'],
                    "pickUpDate" => $pickUpDate,
                    "departureDate" => $departureDate,
                    "arrivedDate" => $arrivedDate,
                    "isReceived" => $data['is_delivered']
                ];
            }
        }
        return $response;
    }

    private function dateCustomFormat(string $date): string
    {
        if (empty($date)) {
            return "";
        }
        return (new DateTime($date))->format('Y-m-d\TH:i:s.v\Z');
    }

    private function findOrderCurrentLocation(int $id): string
    {
        $this->load->model('wechatmodel', 'chat');
        $getLastStatusInfo = $this->chat->orderLocation($id);
        if (empty($getLastStatusInfo)) {
            return "";
        }
        if (!empty($getLastStatusInfo['loc_name'])) {
            return trim(preg_replace('/\s\s+/', ' ', str_replace("\n", " ", $getLastStatusInfo['loc_name'])));
        }
        if (!empty($getLastStatusInfo['latitude']) && !empty($getLastStatusInfo['longitude'])) {
            $locationName = getLocationName($getLastStatusInfo['latitude'], $getLastStatusInfo['longitude']);
            return trim(preg_replace('/\s\s+/', ' ', str_replace("\n", " ", $locationName)));
        }
        return "";
    }

    public function sendShipment(): void
    {
        ob_start();
        $code = $message = "";
        $orderId = $this->input->post('order_id');
        $this->load->library('Wechat');
        $this->wechat->orderId = $orderId;
        $response = $this->wechat->sendNotificationsToWeChat();
        ob_end_clean();
        echo json_encode(Printable::printable($response));
    }

    public function getShipmentStatusbytrackingNumber() : void {
        $response = [];
        $trackingNumber = $this->input->get('searchKey',true);
        $this->load->model('wechatmodel', 'chat');
        $orderInfo = $this->chat->validateTrackingNumber($trackingNumber);
        if (!empty($orderInfo)) {
            $response = $this->processOrderData($orderInfo['order_id'], $trackingNumber);
        }
        echo json_encode($response);
    }

    private function processOrderData(int $orderId, int $trackingNumber): array {
        $result = $data = [];
        $this->chat->orderId = $orderId;
        $result['basic'] = $this->chat->getBasicShipmentInfo();
        $response = $this->processOrderDetails($result);
        $data["salogTracking"]          = $trackingNumber;
        $data["vehicleTypeCode"]        = $response["vehicleTypeCode"];
        $data["vehicleModelCode"]        = $response["vehicleTypeMode"];
        $data["driverDetails"]          = $response["driverDetails"];
        $data["source"]                 = $response["source"];
        $data["destination"]            = $response["destination"];
        $data["estimatedPickupTime"]    = $response["estimatedPickupTime"];
        $data["estimatedDeliveryTime"]  = $response["estimatedDeliveryTime"];
        $data["orderStatus"]            = $response["orderStatus"];
        return $data;
    }

    private function processStatusLists(array $orderStatusResults = [], int $orderId = 0, int $shiftId = 0): array {
        $response = [];
        foreach ($orderStatusResults as $status) {
            $getETADateTime = getdatetimebytimezone(DFLT_TZ, $status['next_stop_eta'], $status['cntry_timezone']);
            $ETADateTime    = $getETADateTime['date'];
            $getCreatedOn   = getdatetimebytimezone(DFLT_TZ, $status['createdon'], $status['cntry_timezone']);
            $createdOn      = $getCreatedOn['date'];
            $gpsInfo = [];
            $statusCode = $status['status_code'];
            if ($statusCode === '1550') {
                $gpsInfo = $this->chat->getGPSPings($orderId, $shiftId);
            }
            $response[] = [
                'statusCode'            =>  (string)$statusCode,
                'statusValue'           =>  (string)$status['status_name'],
                'statusType'            =>  (string)$status['status_name'],
                'dateTime'              =>  (new DateTime($createdOn))->format('Y-m-d\TH:i:s.v\Z'),
                'timeZone'              =>  "+08:00/CST",
                'lat'                   =>  (string) $status['latitude'],
                'lng'                   =>  (string) $status['longitude'],
                'location'              =>  (string) $status['loc_name'],
                'nextStopETA'           =>  $status['next_stop_eta'] ? (new DateTime($ETADateTime))->format('Y-m-d\TH:i:s.v\Z') : "",
                'nextStopDuration'      =>  (string) $status['next_stop_duration'],
                'additionalInfo'        =>  [
                    "reasonCode"        => "",
                    "reasonDescription" => "",
                    "remarkCode"        => "",
                    "remarkDescription" => ""
                ],
                'gps_info'=>$gpsInfo
            ];
        }
        return $response;
    }

    private function processOrderDetails(array $orderData): array {
        $data = $orderData['basic'];
        $driverInfo = [];
        if ((int) $data["driver_id"] > 0) {
            $driverInfo = $this->chat->getAssignedDriverInfo($data["driver_id"]);
        }
        $orderStatus = [];
        if ($data["shift_id"] > 0) {
            $orderStatusResults = $this->chat->getTripStatusInfo($data["id"],$data["shift_id"]);
            if ($orderStatusResults) {
                $orderStatus = $this->processStatusLists($orderStatusResults, $data["id"], $data["shift_id"]);
            }
        }
        $response = [
            "estimatedPickupTime"   => (new DateTime($data["pickup_datetime"]))->format('Y-m-d\TH:i:s.v\Z'),
            "estimatedDeliveryTime"   => (new DateTime($data["drop_endtime"]))->format('Y-m-d\TH:i:s.v\Z'),
            "vehicleTypeCode" =>$data["vehicle_type"],
            "vehicleTypeMode" =>$data["transport_mode"],
            "driverDetails" =>$driverInfo,
            "source" => [
                "companyName" => $data["companyname"],
                "address1"    => $data["address1"],
                "address2"    => $data["address2"],
                "street"      => $data["street"],
                "city"        => $data["city"],
                "state"       => $data["state"],
                "postal"      => $data["postal"],
                "country"     => !empty($data["origin_country"]) ? $data["origin_country"] : $data["country"],
                "lat"         => $data["plat"],
                "log"         => $data["plng"],
            ],
            "destination" => [
                "companyName" => $data["dcompanyname"],
                "address1"    => $data["daddress1"],
                "address2"    => $data["daddress2"],
                "street"      => $data["dstreet"],
                "city"        => $data["dcity"],
                "state"       => $data["dstate"],
                "postal"      => $data["dpostal"],
                "country"     => !empty($data["destination_country"]) ? $data["destination_country"] : $data["dcountry"],
                "lat"         => $data["dlat"],
                "log"         => $data["dlng"],
            ],
            "orderStatus" =>$orderStatus
        ];
        return $response;
    }
}