<?php

namespace {
    use kuehnenagel\svkonekt\log;

    if (! function_exists('log_backtrace')) {
        function log_backtrace(string $type, string $message, int $depth = 1) :void {
            log\backtrace($type, $message, $depth);
        }
    }

    if (! function_exists('log_debug')) {
        # Must be used to log debug informations, that is to say messages that assist in debugging. For example, if a class has been initialized, you could log this as debugging info.
        function log_debug(string $message, int $depth = 1) :void {
            log\debug($message, $depth);
        }
    }

    if (! function_exists('log_info')) {
        # Must be used to log informational informations, that is to say messages simply giving information regarding some process.
        function log_info(string $message, int $depth = 1) :void {
            log\info($message, $depth);
        }
    }

    if (! function_exists('log_error')) {
        # Must be used to log informational related to errors, that is to say PHP errors or user errors (wrong data in HTML form, network errors, file system errors, and so on.
        function log_error(string $message, int $depth = 1) :void {
            log\error($message, $depth);
        }
    }
}

namespace kuehnenagel\svkonekt\log {

    use App\Libraries\infrastructure\LogService;

    function backtrace(string $type, string $message, int $depth = 1): void
    {
        $backtrace = debug_backtrace();

        foreach ($backtrace as $key => $trace) {
            if (!isset($trace['class'])) {
                switch ($trace['function']) {
                    case 'log_debug':
                    case 'log_info':
                    case 'log_error':
                        break 2;
                }
            }
        }

        $fileAndLineKey = $key + ($depth - 1);

        $message = sprintf('file %s at line %s: %s', $backtrace[$fileAndLineKey]['file'], $backtrace[$fileAndLineKey]['line'], $message);

        $functionKey = $key + $depth;

        if (isset($backtrace[$functionKey]['function'])) {
            $message = sprintf('%s(): %s', $backtrace[$functionKey]['function'], $message);
        }

        if (isset($backtrace[$functionKey]['class'])) {
            $message = sprintf('%s::%s', $backtrace[$functionKey]['class'], $message);
        }

        $message = sprintf('[%s][%s] %s', strtoupper($type), (new \DateTime('now', new \DateTimeZone('UTC')))->format('Y-m-d H:i:s e'), $message);

        if (getenv('CRONJOB') === 'true') {
            $logService = new LogService(
                AWS_LOGS_GROUP,
                AWS_LOGS_STREAM
            );

            $logService->getLogger()->log($type, $message);
        } else {
            log_message($type, $message);
        }
    }

    function debug(string $message, int $depth = 1) :void {
        backtrace('debug', $message, $depth);
    }

    function info(string $message, int $depth = 1) :void {
        backtrace('info', $message, $depth);
    }

    function error(string $message, int $depth = 1) :void {
        backtrace('error', $message, $depth);
    }

    function errors(string... $errors) :void {
        foreach ($errors as $error) {
            error($error);
        }
    }

    function exception(\exception $exception) :void {
        $backtrace = $exception->getTrace();

        error(sprintf('file %s at line %s: %s', $backtrace[0]['file'], $backtrace[0]['line'], $exception->getMessage()));
    }
}
