<?php

class AuAwsOrderCreate
{
    private $ci;
    private $routingStatus;
    private $orderTypeRowId = 0;
    private $userId = 0;
    private $customerName = "";
    private $serviceId = 0;
    private $consigneeId = 0;
    private $shipperId = 0;
    private $customerId = 0;
    private $consigneeTypeId = 0;
    private $shipperTypeId = 0;
    private $customerTypeId = 0;

    public function __construct()
    {
        $this->ci = &get_instance();
        $this->ci->load->model(['common', 'AuAwsModel']);
    }

    /**
     * Process order data
     *
     * @param $orderData
     * @return void
     */
    public function processXmlOrderData($orderData): void
    {
        $response = $xmlOrderData = [];
        $status = 0;
        try {
            if (validateXml($orderData)) {
                $xmlOrderData = json_decode(json_encode(simplexml_load_string($orderData)), true);
                $validationStatus = $this->validateData($xmlOrderData);
                if (!empty($validationStatus)) {
                    $ordersStatus = $this->statusCodes("406");
                    $response = array_merge($ordersStatus, $validationStatus);
                    /*this Method calling EDI purposes */
                    $this->getPartyIds($xmlOrderData);
                } else {
                    $response = $this->saveOrderData($xmlOrderData);
                    if ($response['code'] == "200") {
                        $status = 1;
                    }
                }
                echo json_encode($response);
            } else {
                echo json_encode($this->statusCodes("406"));
                log_error("Au_aws_order_create : XML not Valid");
            }
        } catch (Exception $exc) {
            $response['error'] = "Exception Occur when pulling AU_AWS xml to eTN:" . $exc->getMessage();
            log_error(json_encode($response));
        } finally {
            $this->saveEdiLogs($xmlOrderData, $response, $status);
        }
    }

    /**
     * Order xml validation
     *
     * @param $orderData
     * @return array
     */
    public function validateData($orderData): array
    {
        $getValidationMessage = [];
        try {
            $validationData = [];
            $validationData["companyCode"] = $this->validateCompanyCode($orderData);
            $validationData["departmentCode"] = $this->validateDepartmentCode($orderData);
            $validationData["orderType"] = $this->validateOrderType($orderData);
            $validationData["productType"] = $this->validateProductType($orderData);
            $validationData["serviceType"] = $this->validateServiceType($orderData);
            $validationData["deliveryTerms"] = $this->validateDeliveryTerms($orderData);
            $validationData["inTerms"] = $this->validateInTerms($orderData);
            $validationData["modeOfTransport"] = $this->validateModeOfTransport($orderData);
            $validationData["routingDetails"] = $this->validateRoutingDetails($orderData);
            $validationData["ExternalOrderId"] = $this->validateExternalOrderId($orderData);
            $getValidationMessage = $this->validationMessage($validationData, $orderData);
        } catch (Exception $e) {
            log_message('error', 'Exception Occurred when  AU_AWS order validation :' . json_encode($e->getMessage()));
        }
        return $getValidationMessage;
    }

    /**
     *  save order
     *
     * @param $orderData
     * @return array
     */
    public function saveOrderData($orderData): array
    {
        $orderStatusMessage = [];
        try {
            $finalOrderData = $this->prepareOrderArray($orderData);
            if (!empty($finalOrderData)) {
                $orderCreationStatus = $this->ci->AuAwsModel->saveOrder($finalOrderData);
                if ($orderCreationStatus == true) {
                    $orderStatusMessage = $this->statusCodes("200");
                    $getBookingId = $this->ci->AuAwsModel->getOrderBookingId();
                    if (!empty($getBookingId)) {
                        $orderStatusMessage['SvkonektBookingId'] = $getBookingId;
                    }
                } else {
                    $orderStatusMessage = $this->statusCodes("406");
                }
            } else {
                $orderStatusMessage = $this->statusCodes("204");
            }
        } catch (Exception $e) {
            log_message('error', 'Exception Occur when save the AU_AWS order :' . json_encode($e->getMessage()));
        }
        return $orderStatusMessage;
    }

    /**
     * get user table data
     *
     * @param $orderData
     * @return int
     */
    public function getUserData($orderData): int
    {
        $where = [
            "company_code" => $orderData['KNOrgDetails']['CompanyCode'],
            "branch_code" => $orderData['KNOrgDetails']['BranchCode']
        ];
        $getUserData = $this->ci->common->gettblrowdata($where, "id,name", "tb_users", 0, 0);
        return $this->userId = (!empty($getUserData)) ? $getUserData['id'] : 0;
    }

    /**
     * get order type data
     *
     * @param $orderData
     * @return int
     */
    public function getOrderTypeData($orderData): int
    {
        $where = [
            "type_name" => $orderData['OrderDetails']['OrderType'],
            "company_code" => $orderData['KNOrgDetails']['CompanyCode'],
            "status" => 1
        ];
        $getOrderTypes = $this->ci->common->gettblrowdata($where, "id,type_name", "tb_order_types", 0, 0);
        return $this->orderTypeRowId = (!empty($getOrderTypes)) ? $getOrderTypes['id'] : 0;
    }

    /**
     * get service  data
     *
     * @param $orderData
     * @return int
     */
    public function getServiceData($orderData): int
    {
        $where = ["name" => $orderData['OrderDetails']['ServiceType'], "status" => 1];
        $serviceTypeData = $this->ci->common->gettblrowdata($where, "id,name", "tb_service_master", 0, 0);
        return $this->serviceId = (!empty($serviceTypeData)) ? $serviceTypeData['id'] : 0;
    }

    /**
     * get routing   data
     *
     * @param $OrderParties
     * @return array
     */
    public function getRoutingData($OrderParties): array
    {
        $where = ["code" => $OrderParties['PartyID']['ID'], "status" => 1];
        $getPartyMasterData = $this->ci->common->gettblrowdata(
            $where,
            "id,name,party_type_id",
            "tbl_party_master",
            0,
            0
        );
        if (empty($getPartyMasterData)) {
            $this->routingStatus = "Invalid " . $OrderParties['Type'] . " -:" . $OrderParties['PartyID']['ID'];
        } else {
            $customerType = $OrderParties['Type'] ?? '0';
            switch ($customerType) {
                case 'CONSIGNEE':
                    $this->consigneeId = $getPartyMasterData['id'] ?? 0;
                    $this->consigneeTypeId = $getPartyMasterData['party_type_id'] ?? 0;
                    break;
                case 'SHIPPER':
                    $this->shipperId = $getPartyMasterData['id'] ?? 0;
                    $this->shipperTypeId = $getPartyMasterData['party_type_id'] ?? 0;
                    break;
                case 'CUSTOMER':
                    $this->customerId = $getPartyMasterData['id'] ?? 0;
                    $this->customerTypeId = $getPartyMasterData['party_type_id'] ?? 0;
                    break;
                default:
                    $this->consigneeId = $this->consigneeTypeId =
                    $this->shipperId = $this->shipperTypeId =
                    $this->customerId = $this->customerTypeId = 0;
            }
        }
        return $getPartyMasterData;
    }


    /**
     * Validate company code and branch code
     *
     * @param $orderData
     * @return bool
     */
    public function validateCompanyCode($orderData): bool
    {
        return (!empty($this->getUserData($orderData)));
    }

    /**
     * Validate department code
     *
     * @param $orderData
     * @return bool
     */
    public function validateDepartmentCode($orderData): bool
    {
        $where = [
            "company_code" => $orderData['KNOrgDetails']['CompanyCode'],
            "department_code" => $orderData['OrderDetails']['DepartmentCode']
        ];
        return (!empty(
        $this->ci->common->gettblrowdata(
            $where,
            "department_name",
            "tb_department_master",
            0,
            0
        )
        ));
    }

    /**
     * Validate Order type
     *
     * @param $orderData
     * @return bool
     */
    public function validateOrderType($orderData): bool
    {
        return (!empty($this->getOrderTypeData($orderData)));
    }

    /**
     * Validate Product type
     *
     * @param $orderData
     * @return bool
     */
    public function validateProductType($orderData): bool
    {
        $where = ["name" => $orderData['OrderDetails']['OrderProductType'], "status" => 1];
        return (!empty($this->ci->common->gettblrowdata($where, "name", "tb_products_master", 0, 0)));
    }

    /**
     *  Validate Service type
     *
     * @param $orderData
     * @return bool
     */
    public function validateServiceType($orderData): bool
    {
        return !empty($this->getServiceData($orderData));
    }

    /**
     *  Validate delivery terms
     *
     * @param $orderData
     * @return bool
     */
    public function validateDeliveryTerms($orderData): bool
    {
        $where = ["name" => $orderData['OrderDetails']['DeliveryTerms'], "status" => 1];
        return (!empty($this->ci->common->gettblrowdata($where, "name", "tb_delivery_terms", 0, 0)));
    }

    /**
     *  Validate In terms
     *
     * @param $orderData
     * @return bool
     */
    public function validateInTerms($orderData): bool
    {
        $where = ["name" => $orderData['OrderDetails']['TermsOfTrade']['Incoterm'], "status" => 1];
        return (!empty($this->ci->common->gettblrowdata($where, "name", "tb_incoterms_master", 0, 0)));
    }

    /**
     *  Validate Mode Of Transport
     *
     * @param $orderData
     * @return bool
     */
    public function validateModeOfTransport($orderData): bool
    {
        return (in_array(
                $orderData['OrderDetails']['ModeOfTransport'],
                ['FTL', 'LTL', 'COURIER', 'AIR-LCL', 'AIR-FCL', 'SEA-LCL', 'SEA-FCL', 'GROUPAGE']
            )) ?? false;
    }

    /**
     *  Validate Routing details
     *
     * @param $orderData
     * @return bool
     */
    public function validateRoutingDetails($orderData): bool
    {
        $partyType = $orderData['OrderDetails']['OrderParties']['PartyType'];
        foreach ($partyType as $OrderParties) {
            if (empty($this->getRoutingData($OrderParties))) {
                return false;
            }
        }
        return true;
    }

    /**
     * validate External Order-Id
     *
     * @param $orderData
     * @return bool
     */
    public function validateExternalOrderId($orderData): bool
    {
        $where = ["external_order_id" => $orderData['OrderDetails']['EXTOrderID'], "status <>" => 0];
        return (empty($this->ci->common->gettblrowdata($where, "order_id", "tb_orders", 0, 0)));
    }

    /**
     * if validation is false, set status message
     *
     * @param $validationData
     * @param $orderData
     * @return array
     */
    public function validationMessage($validationData, $orderData): array
    {
        $status = [];
        if ($validationData["companyCode"] == false) {
            $status["errorMessage"][] = $this->setInvalidMessage(
                "company code",
                $orderData['KNOrgDetails']['CompanyCode']
            );
        }
        if ($validationData["departmentCode"] == false) {
            $status["errorMessage"][] = $this->setInvalidMessage(
                "department code",
                $orderData['OrderDetails']['DepartmentCode']
            );
        }
        if ($validationData["orderType"] == false) {
            $status["errorMessage"][] = $this->setInvalidMessage(
                "order type",
                $orderData['OrderDetails']['OrderType']
            );
        }
        if ($validationData["productType"] == false) {
            $status["errorMessage"][] = $this->setInvalidMessage(
                "product type",
                $orderData['OrderDetails']['OrderProductType']
            );
        }
        if ($validationData["serviceType"] == false) {
            $status["errorMessage"][] = $this->setInvalidMessage(
                "service type",
                $orderData['OrderDetails']['ServiceType']
            );
        }
        if ($validationData["deliveryTerms"] == false) {
            $status["errorMessage"][] = $this->setInvalidMessage(
                "delivery terms",
                $orderData['OrderDetails']['DeliveryTerms']
            );
        }
        if ($validationData["inTerms"] == false) {
            $status["errorMessage"][] = $this->setInvalidMessage(
                "inco terms",
                $orderData['OrderDetails']['TermsOfTrade']['Incoterm']
            );
        }
        if ($validationData["modeOfTransport"] == false) {
            $status["errorMessage"][] = $this->setInvalidMessage(
                "mode of transport",
                $orderData['OrderDetails']['ModeOfTransport']
            );
        }
        if ($validationData["routingDetails"] == false) {
            $status["errorMessage"][] = $this->routingStatus;
        }
        if ($validationData["ExternalOrderId"] == false) {
            $status["errorMessage"][] = "external id already exist -:" . $orderData['OrderDetails']['EXTOrderID'];
        }
        return $status;
    }

    /**
     * @param $message
     * @param null $data
     * @return string
     */
    public function setInvalidMessage($message, $data = null): string
    {
        log_message("error", "AU_AWS_order : Invalid " . $message . "-:" . $data);
        return "Invalid " . $message . " -:" . $data;
    }

    /**
     * status codes
     *
     * @param $code
     * @return array
     */
    public function statusCodes($code): array
    {
        $statusCodes = [];
        switch ($code) {
            case "200":
                $statusCodes['code'] = "200";
                $statusCodes['message'] = "Order crated successfully";
                break;
            case "401":
                $statusCodes['code'] = "401";
                $statusCodes['message'] = "Customer authentication fail";
                break;
            case "404":
                $statusCodes['code'] = "404";
                $statusCodes['message'] = "Unable to save order";
                break;
            case "406":
                $statusCodes['code'] = "406";
                $statusCodes['message'] = "Data not acceptable";
                break;
            case "204":
                $statusCodes['code'] = "204";
                $statusCodes['message'] = "no data found";
                break;
            default:
                $statusCodes['code'] = "null";
                $statusCodes['message'] = "no status code found ";
                break;
        }
        return $statusCodes;
    }

    /**
     * get party ids (shipper,consignee,customer )
     *
     * @param $orderData
     * @return array
     */
    public function getPartyIds($orderData): array
    {
        $result = [];
        foreach ($orderData['OrderDetails']['OrderParties']['PartyType'] as $values) {
            $customerType = $values['Type'] ?? '0';
            switch ($customerType) {
                case 'CONSIGNEE':
                    $result['orderConsigneeId'] = $values['PartyID']['ID'] ?? '0';
                    $result['orderConsigneeName'] = $values['PartyID']['Name'] ?? '';
                    break;
                case 'SHIPPER':
                    $result['orderShipperId'] = $values['PartyID']['ID'] ?? '0';
                    $result['orderShipperName'] = $values['PartyID']['Name'] ?? '';
                    break;
                case 'CUSTOMER':
                    $result['orderCustomerId'] = $values['PartyID']['ID'] ?? '0';
                    $result['orderCustomerName'] = $this->customerName = $values['PartyID']['Name'] ?? '';
                    break;
                default:
                    $result['orderCustomerId'] = "";
                    $result['orderCustomerName'] = "";
            }
        }
        return $result;
    }

    /**
     * get order array
     *
     * @param $xmlData
     * @return array
     */
    public function prepareOrderArray($xmlData): array
    {
        $getPartyData = $this->getPartyIds($xmlData);
        $orderCustomerId = $getPartyData['orderCustomerId'] ?? '0';
        $customerRowId = $this->getCustomerData($orderCustomerId);
        $this->customerName = $getPartyData['orderCustomerName'] ?? '';
        $dates = $this->getDateTime($xmlData);
        $transport = $this->modeOfTransport($xmlData);
        $dgGoods = $xmlData['OrderDetails']['OrderCargoSummary']['DangerousGoodsFlag'] ? 1 : 0;
        $createdDate = date('Y-m-d H:i:s');
        return [
            'external_order_id' => $xmlData['OrderDetails']['EXTOrderID'] ?? '',
            'goods_value' => $xmlData['OrderDetails']['GoodsValue']['Value'] ?? '',
            'currency' => $xmlData['OrderDetails']['GoodsValue']['Currency'] ?? '',
            'serviceType' => $xmlData['OrderDetails']['ServiceType'] ?? '',
            'departmentCode' => $xmlData['OrderDetails']['DepartmentCode'] ?? '',
            'inTerm' => $xmlData['OrderDetails']['TermsOfTrade']['Incoterm'] ?? '',
            'orderTypeRowId' => $this->orderTypeRowId,
            'shipper_id' => $getPartyData['orderShipperId'] ?? '0',
            'orderShipperId' => $getPartyData['orderShipperId'] ?? '0',
            'pickupPartyId' => $getPartyData['orderShipperId'] ?? '0',
            'orderConsigneeId' => $getPartyData['orderConsigneeId'] ?? '0',
            'dropPartyId' => $getPartyData['orderConsigneeId'] ?? '0',
            'customer_id' => $customerRowId,
            'orderCustomerId' => $orderCustomerId,
            'orderCustomerName' => $this->customerName,
            'pickup_country' => $xmlData['OrderDetails']['OrderStartLocation']['Address']['Country'] ?? '',
            'pickup_city' => $xmlData['OrderDetails']['OrderStartLocation']['Address']['City'] ?? '',
            'pickupPinCode' => $xmlData['OrderDetails']['OrderStartLocation']['Address']['Postal'] ?? '',
            'pickup_company' => $getPartyData['orderShipperName'] ?? '',
            'pickup_address1' => $xmlData['OrderDetails']['OrderStartLocation']['Address']['Street'] ?? '',
            'pickup_address2' => $xmlData['OrderDetails']['OrderStartLocation']['Address']['State'] ?? '',
            'delivery_country' => $xmlData['OrderDetails']['OrderEndLocation']['Address']['Country'] ?? '',
            'delivery_city' => $xmlData['OrderDetails']['OrderEndLocation']['Address']['City'] ?? '',
            'deliveryPinCode' => $xmlData['OrderDetails']['OrderEndLocation']['Address']['Postal'] ?? '',
            'delivery_company' => $getPartyData['orderConsigneeName'] ?? '',
            'delivery_address1' => $xmlData['OrderDetails']['OrderEndLocation']['Address']['Street'] ?? '',
            'delivery_address2' => $xmlData['OrderDetails']['OrderEndLocation']['Address']['State'] ?? '',
            'quantity' => $xmlData['OrderDetails']['OrderCargoSummary']['TotalQuantity']['Value'] ?? '',
            'volume' => $xmlData['OrderDetails']['OrderCargoSummary']['TotalVolume']['Value'] ?? '',
            'weight' => $xmlData['OrderDetails']['OrderCargoSummary']['TotalWeight']['Value'] ?? '',
            'pickup_dateTime' => $dates['pickupDate'] ?? '',
            'pickup_endTime' => $dates['pickupDateEnd'] ?? '',
            'delivery_dateTime' => $dates['deliveryDate'] ?? '',
            'drop_endTime' => $dates['deliveryDateEnd'] ?? '',
            'modeOfTransportId' => $transport['id'] ?? 0,
            'dgGoods' => $dgGoods,
            'modeOfTransportName' => $transport['name'] ?? '',
            'user_id' => $this->userId,
            'company_code' => $xmlData['KNOrgDetails']['CompanyCode'] ?? '',
            'branch_code' => $xmlData['KNOrgDetails']['BranchCode'] ?? '',
            'product' => $xmlData['OrderDetails']['OrderProductType'] ?? '',
            'KNOrgDetails' => $xmlData['KNOrgDetails'] ?? '',
            'OrderDetails' => $xmlData['OrderDetails'] ?? '',
            'status' => 1,
            'is_created' => 1,
            'created_on' => $createdDate,
            'serviceId' => $this->serviceId,
            'consigneeRowId' => $this->consigneeId,
            'shipperRowId' => $this->shipperId,
            'customerRowId' => $this->customerId,
            'consigneeTypeId' => $this->consigneeTypeId,
            'shipperTypeId' => $this->shipperTypeId,
            'customerTypeId' => $this->customerTypeId,
        ];
    }

    /**
     * get customer id
     *
     * @param $orderCustomerId
     * @return int
     */
    public function getCustomerData($orderCustomerId): int
    {
        $customer = $this->ci->common->gettblrowdata(
            ['code' => $orderCustomerId],
            'id',
            'tb_customers',
            0,
            0
        );
        return (!empty($customer)) ? $customer['id'] : 0;
    }

    /**
     * get all routing dates
     *
     * @param $orderData
     * @return array
     */
    public function getDateTime($orderData): array
    {
        $dates = [];
        $timeZoneStart = $orderData['OrderDetails']['OrderStartLocation']['Address']['Timezone'] ?? "";
        $timeZoneEnd = $orderData['OrderDetails']['OrderEndLocation']['Address']['Timezone'] ?? "";
        $getStartTime = $this->validateTimeZone($timeZoneStart);
        $getEndTime = $this->validateTimeZone($timeZoneEnd);
        $company_code = $orderData['KNOrgDetails']['CompanyCode'];
        $timeZonePickup = $orderData['OrderDetails']['OrderStartLocation']['PlannedDateTime']['From'];
        $timeZoneDelivery = $orderData['OrderDetails']['OrderEndLocation']['PlannedDateTime']['From'];
        $timeZonePickupEnd = $orderData['OrderDetails']['OrderStartLocation']['PlannedDateTime']['To'];
        $timeZoneDeliveryEnd = $orderData['OrderDetails']['OrderEndLocation']['PlannedDateTime']['To'];
        $dates['pickupDate'] = $this->getRoutingDates($timeZonePickup, $getStartTime, $company_code);
        $dates['deliveryDate'] = $this->getRoutingDates($timeZoneDelivery, $getStartTime, $company_code);
        $dates['pickupDateEnd'] = $this->getRoutingDates($timeZonePickupEnd, $getEndTime, $company_code);
        $dates['deliveryDateEnd'] = $this->getRoutingDates($timeZoneDeliveryEnd, $getEndTime, $company_code);
        return $dates;
    }

    /**
     * Validate company code and branch code
     *
     * @param $timeZone
     * @return String
     */
    public function validateTimeZone($timeZone): string
    {
        $getCountryMasterData = [];
        if (!empty($timeZone)) {
            $where = [
                'cntry_timezone' => $timeZone,
                'status' => 1
            ];
            $getCountryMasterData = $this->ci->common->gettblrowdata($where, "id", "tbl_country_master", 0, 0);
        }
        return (!empty($getCountryMasterData)) ? $timeZone : '';
    }

    /**
     * get date
     *
     * @param $data
     * @param $timeZone
     * @param $company_code
     * @return string
     */
    public function getRoutingDates($data, $timeZone, $company_code): string
    {
        $date = "";
        try {
            if (empty($timeZone)) {
                $timeZone = $this->getTimeZone($company_code);
            }
            $timeZonePickup = new DateTime($data);
            $timeZonePickup->setTimezone(new DateTimeZone($timeZone));
            $date = $timeZonePickup->format('Y-m-d H:i:s');
        } catch (Exception $ex) {
            log_error('AU_aws_order : Timezone conversion error :' . json_encode($ex->getMessage()));
        }
        return $date;
    }

    /**
     * get time zone by using company code
     *
     * @param $company_code
     * @return string
     */
    public function getTimeZone($company_code): string
    {
        return ($company_code == "KNAU") ? "Australia/Sydney" : "";
    }

    /**
     * get Mode of transport
     *
     * @param $orderData
     * @return array
     */
    public function modeOfTransport($orderData): array
    {
        $where = ['code' => $orderData['OrderDetails']['ModeOfTransport'], 'status' => 1];
        $getTransportModeData = $this->ci->common->gettblrowdata($where, "id,name", "tb_transportmode", 0, 0);
        return (!empty($getTransportModeData)) ? $getTransportModeData : [];
    }

    /**
     * save EDI logs
     *
     * @param $orderData
     * @param $response
     * @param $status
     */
    public function saveEdiLogs($orderData, $response, $status)
    {
        log_debug('AU AWS Order XMLs storing into EDI logs');
        $this->ci->load->library('Edi_logger');
        $this->ci->edi_logger->setEdi_type(1);
        $this->ci->edi_logger->setTransaction_id(time());
        $this->ci->edi_logger->setEdi_name($this->customerName);
        $this->ci->edi_logger->setBounded_type(1);
        $this->ci->edi_logger->setEdi_format_type('XML');
        $this->ci->edi_logger->setStatus($status);
        $this->ci->edi_logger->setObj_type_name('external_order-id');
        $this->ci->edi_logger->setTxn_obj_id(($orderData['OrderDetails']['EXTOrderID'] ?? 0));
        $this->ci->edi_logger->setEdi_request(json_encode($orderData));
        $this->ci->edi_logger->setEdi_response(json_encode($response));
        $this->ci->edi_logger->setCompany_code(($orderData['KNOrgDetails']['CompanyCode'] ?? "KNAU"));
        $this->ci->edi_logger->setBranch_code(($orderData['KNOrgDetails']['BranchCode'] ?? "KNAU"));
        $this->ci->edi_logger->setUser_id(($this->userId ?? 0));
        $this->ci->edi_logger->saveToEdiLogs();
    }

}
