<?php

require_once './vendor/autoload.php';

use App\MobileEvents\Handler\MilestoneUpdateXmlChecker;
use PhpAmqpLib\Message\AMQPMessage;
use PhpAmqpLib\Connection\AMQPSSLConnection;
use PhpAmqpLib\Wire;

class Roadlog_status
{
    private $ci;

    public function __construct()
    {
        $this->ci = &get_instance();
        $this->ci->load->library('Edi_logger');
        $this->ci->load->model('common');
    }

    public function status_handler(array $data): void
    {
        if (!MilestoneUpdateXmlChecker::isTest()) {
            log_message("error", "Data Received at roadlog status_handler /" . json_encode($data));
        }
        $response = [];
        try {
            self::validateStatusData($data);
        } catch (Throwable $th) {
            $response['_error'] = $th->getMessage();
        }
        $request_xml = '';
        if (!isset($response ['_error'])) {
            $request_xml = $this->roadlogXMLGenerator($data);
            if (MilestoneUpdateXmlChecker::isTest()) {
                MilestoneUpdateXmlChecker::rememberXml($request_xml);
                return;
            }
            if (empty($request_xml)) {
                self::throwException('Roadlog xml generation is failed, with following information/' . json_encode($data));
            } else {
                $this->publishxml($request_xml, $data);
            }
        }
        $this->saveToEdiLogs($data, $request_xml, $response, 'ROADLOG');
    }

    public function publishxml(string $request, array $data): void
    {
        $ci = &get_instance();
        $ci->load->library('uuid');
        $uuid = $ci->uuid->v4();
        $ssl_options = [
            'verify_peer' => false,
            'verify_peer_name' => false,
        ];
        $headers = new Wire\AMQPTable([
            'uuid' => $uuid,
            'KNESB_Routing_SenderId' => KNESB_SENDERID,
            'KNESB_Routing_ReceiverId' => KNESB_RECEIVERID_ROADLOG,
            'KN_MS_PhysRecvID' => $data['physicalsender'],
            'KN_MS_LogRecvID' => $data['logicalreceiver'],
            'KN_MS_PhysSenderID' => $data['physicalreceiver'],
        ]);
        $headers->set('short', -1024, Wire\AMQPTable::T_INT_SHORT);
        $connection = new AMQPSSLConnection(SALOG_URL, 5671, SALOG_USRNAME, SALOG_PWD, SALOG_ESB, $ssl_options);
        $channel = $connection->channel();
        $Xmldata = new AMQPMessage(
            $request,
            ['content_type' => 'Application/xml', 'delivery_mode' => AMQPMessage::DELIVERY_MODE_PERSISTENT]
        );
        $Xmldata->set('application_headers', $headers);
        try {
            $channel->basic_publish($Xmldata, SALOG_EROADSTATUS, null);
        } catch (Exception $e) {
            log_message("error", "Error happen while pushing xml to roadlog " . $e->getMessage() . '/' . $e->getCode() . '/ Xml:' . $request);
        } finally {
            $channel->close();
            $connection->close();
        }
    }

    /*
     * remark of type any
     */
    private function validateRemarks($remark): string
    {
        return strlen((!is_null($remark)) ? trim($remark) : '') ? $remark : '1';
    }

    private static function validateStatusData($data): void
    {
        switch (true) {
            case !isset($data['physicalsender']):
                self::throwException('Physical sender is undefined for order /' . $data['order_id']);

            case empty($data['physicalsender']):
                self::throwException('Physical sender must required a value for order /' . $data['order_id']);

            case !isset($data['logicalsender']):
                self::throwException('Logical Sender is undefined for order /' . $data['order_id']);

            case empty($data['logicalsender']):
                self::throwException('Logical sender must required a value for order /' . $data['order_id']);

            case !isset($data['physicalreceiver']):
                self::throwException('Physical receiver is undefined for order /' . $data['order_id']);

            case empty($data['physicalreceiver']):
                self::throwException('Physical receiver must required a value for order /' . $data['order_id']);

            case !isset($data['logicalreceiver']):
                self::throwException('Logical receiver is undefined for order /' . $data['order_id']);

            case empty($data['logicalreceiver']):
                self::throwException('Logical receiver must required a value for order /' . $data['order_id']);

            case !isset($data['current_date_time']):
                self::throwException('Current date time is undefined for order /' . $data['order_id']);

            case empty($data['current_date_time']):
                self::throwException('Current date must required a value for order /' . $data['order_id']);

            case !isset($data['order_id']):
                self::throwException('Tracking number/Order ID is undefined for order /' . $data['order_id']);

            case empty($data['order_id']):
                self::throwException('Tracking number/Order ID must required a value for order /' . $data['order_id']);

            case !isset($data['assoc_id']):
                self::throwException('Assoc ID /Voyage Number/ Shipment Id is undefined for order /' . $data['order_id']);

            case empty($data['assoc_id']):
                self::throwException('Assoc ID /Voyage Number/ Shipment Id must required a value for order /' . $data['order_id']);

            case !isset($data['ref_value_kn_office']):
                self::throwException('KN office reference is undefined for order /' . $data['order_id']);

            case empty($data['ref_value_kn_office']):
                self::throwException('KN office/ INN reference must required a value for order /' . $data['order_id']);

            case !isset($data['status_code']):
                self::throwException('Status code is undefined for order /' . $data['order_id']);

            case empty($data['status_code']):
                self::throwException('Status code must required a value for order /' . $data['order_id']);
        }
    }

    private static function throwException(string $message): void
    {
        $ci = &get_instance();
        $ci->load->helper('log');
        log_error($message, 2);
        throw new \RuntimeException($message);
    }

    private static function generateXmlElement(string $tagValue, string $tagName): string
    {
        if (empty(trim($tagValue)) || empty(trim($tagName))) {
            return '';
        }
        return '<' . $tagName . '>' . $tagValue . '</' . $tagName . '>';
    }

    private function roadlogXMLGenerator(array $data): string
    {
        $ci = &get_instance();
        $status_code = $data['status_code'] ?? '0';
        $fromSecureLink = $data['from_secure_link'] ?? false;
        if (!$fromSecureLink) {
            $getRoadlogstatuscodes = $ci->db->select("roadlog_status_code")->get_where("tb_statuscodes_reference", ["svk_status_code" => $status_code], 1, 0);
            if ($getRoadlogstatuscodes->num_rows() > 0) {
                $status_code = $getRoadlogstatuscodes->row()->roadlog_status_code;
            }
        }
        log_message("error", "roadlog status code for the order ".$data['order_id']." is ".$status_code);
        if ($status_code == "3002") { return '';   }
        $request = '';
        $request .= "<?xml version='1.0' encoding='UTF-8'?>";
        $request .= "<roadstatus:RoadStatus xmlns:roadstatus='http://knie4.int.kn/msg/roadstatus/0100' xmlns:DateUtil='com.surikat.mobility.common.util.DateUtil' xmlns:date='http://exslt.org/dates-and-times' xmlns:ext='http://exslt.org/common' xmlns:lhdr='http://knie4.int.kn/com/lhdr/0200' xmlns:lmlv='http://knie4.int.kn/com/lmlv/0200' xmlns:lpaci='http://knie4.int.kn/com/lpaci/0200' xmlns:lrcc='http://knie4.int.kn/msg/roadcarriercost/0100' xmlns:lref='http://knie4.int.kn/com/lref/0100' xmlns:lsts='http://knie4.int.kn/com/lsts/0200' xmlns:ltru='http://knie4.int.kn/com/ltru/0200' xmlns:ltypes='http://knie4.int.kn/com/ltypes/0200'>";
        $request .= "<lhdr:MessageHeader>";
        $request .= "<ReceiverDetails ltypes:AddressType='FW'>";
        $request .= "<PhysicalReceiver>" . $data['physicalsender'] . "</PhysicalReceiver>";
        $request .= "<LogicalReceiver>" . $data['logicalsender'] . "</LogicalReceiver>";
        $request .= "</ReceiverDetails>";
        $request .= "<PhysicalSender>" . $data['physicalreceiver'] . "</PhysicalSender>";
        $request .= "<LogicalSender>" . $data['logicalreceiver'] . "</LogicalSender>";
        $request .= "<SendingApplication>SVKONEKT</SendingApplication>";
        $request .= "<MessageType>RoadStatus</MessageType>";
        $request .= "<VersionNumber>0100</VersionNumber>";
        $request .= "<CreationDateTime>" . $data['current_date_time'] . "</CreationDateTime>";
        $request .= "</lhdr:MessageHeader>";
        $request .= "<Message>";
        $request .= "<lmlv:MessageLevel>";
        $request .= "<TrackingNumber>" . $data['order_id'] . "</TrackingNumber>";
        $request .= "<SenderInformation>";
        $request .= "<CompanyCode>" . $data['company_code'] . "</CompanyCode>";
        $request .= "<BranchCode>" . $data['branch_code'] . "</BranchCode>";
        $request .= "<ModeOfTransport>1</ModeOfTransport>";
        $request .= "<ExportImportFlag>1</ExportImportFlag>";
        $request .= "</SenderInformation>";
        $request .= "<MessageNumber>" . $data['assoc_id'] . "</MessageNumber>";
        $request .= "<MessageReferences ltypes:AddressType='FF'>";
        $request .= "<Code>INN</Code>";
        $request .= "<Value>" . $data['ref_value_kn_office'] . "</Value>";
        $request .= "</MessageReferences>";
        $request .= "</lmlv:MessageLevel>";
        $request .= "<lsts:StatusInformation ltypes:StatusCode='" . $status_code . "'>";
        $request .= "<StatusDate>" . $data['current_date'] . "</StatusDate>";
        $request .= "<StatusTime>" . $data['current_time'] . "</StatusTime>";
        $request .= "<StatusLocation ltypes:LocationIssuer='ZZZ'>" . htmlspecialchars($data['city']) . "</StatusLocation>";
        $request .= self::generateXmlElement($data['exception_code'] ?? '', 'ExceptionCode');
        $request .= "<AdditionalStatusInformation>";
        $request .= "<CreationIdentifier>X</CreationIdentifier>";
        $request .= "<StatusEntryDateTime>" . $data['current_date_time'] . "</StatusEntryDateTime>";
        $request .= "<UserIdentifier>9210ec43-4f33-4f6f-8afa-2db7e974044c</UserIdentifier>";
        $request .= "<Remarks>" . $this->validateRemarks($data['remark'] ?? '') . "</Remarks>";
        $request .= "</AdditionalStatusInformation>";
        $request .= "<AdditionalRoadStatusInformation>";
        $request .= self::generateXmlElement($data['pod_signee'] ?? '', 'ContactName');
        $request .= self::generateXmlElement($data['latitude'] ?? '', 'Latitude');
        $request .= self::generateXmlElement($data['longitude'] ?? '', 'Longitude');
        $request .= "</AdditionalRoadStatusInformation>";
        $request .= "</lsts:StatusInformation>";
        $request .= "<Shipment/>";
        $request .= "</Message>";
        $request .= "</roadstatus:RoadStatus>";
        return $request;
    }

    private function saveToEdiLogs(array $data, string $request, array $response, string $ediName): void
    {
        $ediLoggerInstance = $this->ci->edi_logger;
        $ediLoggerInstance->setTransaction_id(time());
        $ediLoggerInstance->setEdi_name($ediName);
        $ediLoggerInstance->setBounded_type(2);
        $ediLoggerInstance->setEdi_format_type('XML');
        $ediLoggerInstance->setStatus(isset($response['_error']) ? 0 : 1);
        $ediLoggerInstance->setObj_type_name('order-id');
        $ediLoggerInstance->setTxn_obj_id($data['order_id']);
        $ediLoggerInstance->setEdi_request($request);
        $ediLoggerInstance->setEdi_response(json_encode($response));
        $ediLoggerInstance->setCompany_code($data['company_code']);
        $ediLoggerInstance->setBranch_code($data['branch_code']);
        $result = $this->ci->common->gettblrowdata(["order_id" => $data['order_id']], ['user_id'], 'tb_orders', 1, 0);
        $ediLoggerInstance->setUser_id($result['user_id'] ?? 0);
        $ediLoggerInstance->saveToEdiLogs();
        unset($ediLoggerInstance);
    }
}
