<?php

class Shorturl_l {
	/**
	 * @var string the characters used in building the short URL
	 *
	 * YOU MUST NOT CHANGE THESE ONCE YOU START CREATING SHORTENED URLs!
	 */
	protected static $chars = "123456789bcdfghjkmnpqrstvwxyzBCDFGHJKLMNPQRSTVWXYZ";

	/**
	 * @var string holds the name of the database table to use
	 */
	protected static $table = "short_urls";

	/**
	 * @var boolean whether to connect to a URL to check if it exists
	 */
	protected static $checkUrlExists = TRUE;

	protected $timestamp;

	private $ci;

	public function __construct() {
		$this->ci = &get_instance();
		$this->ci->load->model( 'common' );
		$this->timestamp = $_SERVER["REQUEST_TIME"];
	}

	/**
	 * Create a short code from a long URL.
	 *
	 * Delegates validating the URLs format, validating it, optionally
	 * connecting to the URL to make sure it exists, and checking the database
	 * to see if the URL is already there. If so, the cooresponding short code
	 * is returned. Otherwise, createShortCode() is called to handle the tasks.
	 *
	 * @param string $url the long URL to be shortened
	 *
	 * @return string the short code on success
	 * @throws Exception if an error occurs
	 */
	public function urlToShortCode( $url ) {
		if ( empty( $url ) ) {
			throw new \Exception( "No URL was supplied." );
		}

		if ( $this->validateUrlFormat( $url ) == FALSE ) {
			throw new \Exception( "URL does not have a valid format." );
		}

		if ( self::$checkUrlExists ) {
			if ( ! $this->verifyUrlExists( $url ) ) {
				throw new \Exception( "URL does not appear to exist." );
			}
		}

		$shortCode = $this->urlExistsInDb( $url );
		if ( $shortCode == FALSE ) {
			$shortCode = $this->createShortCode( $url );
		}

		return $shortCode;
	}

	/**
	 * Retrieve a long URL from a short code.
	 *
	 * Deligates validating the supplied short code, getting the long URL from
	 * the database, and optionally incrementing the URL's access counter.
	 *
	 * @param string $code the short code associated with a long URL
	 * @param boolean $increment whether to increment the record's counter
	 *
	 * @return string the long URL
	 * @throws Exception if an error occurs
	 */
	public function shortCodeToUrl( $code, $increment = TRUE ) {
		if ( empty( $code ) ) {
			throw new \Exception( "No short code was supplied." );
		}

		if ( $this->validateShortCode( $code ) == FALSE ) {
			throw new \Exception(
				"Short code does not have a valid format." );
		}

		$urlRow = $this->getUrlFromDb( $code );
		if ( empty( $urlRow ) ) {
			throw new \Exception(
				"Short code does not appear to exist." );
		}

		if ( $increment == TRUE ) {
			$this->incrementCounter( $urlRow["id"] );
		}

		return $urlRow["long_url"];
	}

	/**
	 * Check to see if the supplied URL is a valid format
	 *
	 * @param string $url the long URL
	 *
	 * @return boolean whether URL is a valid format
	 */
	protected function validateUrlFormat( $url ) {
		return filter_var( $url, FILTER_VALIDATE_URL );
	}

	/* Check to see if the URL exists
	 *
	 * Uses cURL to access the URL and make sure a 404 error is not returned.
	 *
	 * @param string $url the long URL
	 * @return boolean whether the URL does not return a 404 code
	 */
	public function parse_url_all( $url ) {
		$url = substr( $url, 0, 4 ) == 'http' ? $url : 'http://' . $url;
		$d   = parse_url( $url );
		$tmp = explode( '.', $d['host'] );
		$n   = count( $tmp );
		if ( $n >= 2 ) {
			if ( $n == 4 || ( $n == 3 && strlen( $tmp[ ( $n - 2 ) ] ) <= 3 ) ) {
				$d['domain']  = $tmp[ ( $n - 3 ) ] . "." . $tmp[ ( $n - 2 ) ] . "." . $tmp[ ( $n - 1 ) ];
				$d['domainX'] = $tmp[ ( $n - 3 ) ];
			} else {
				$d['domain']  = $tmp[ ( $n - 2 ) ] . "." . $tmp[ ( $n - 1 ) ];
				$d['domainX'] = $tmp[ ( $n - 2 ) ];
			}
		}

		return $d;
	}

	protected function verifyUrlExists( $url ) {
		$url_host = $this->parse_url_all( $url )['host'];

		return checkdnsrr( $url_host, 'A' );
	}

	/**
	 * Check the database for the long URL.
	 *
	 * If the URL is already in the database then the short code for it is
	 * returned.  If the URL is not, false is returned.  An exception is thrown
	 * if there is a database error.
	 *
	 * @param string $url the long URL
	 *
	 * @return string|boolean the short code if it exists - false if it does not
	 */
	protected function urlExistsInDb( $url ) {
		$result = $this->ci->common->gettblrowdata( array( 'long_url' => $url ), 'short_code', self::$table, 1, 0 );

		return ( empty( $result ) ) ? FALSE : $result["short_code"];
	}

	/**
	 * Delegates creating a short code from a long URL.
	 *
	 * Delegates inserting the URL into the database, converting the integer
	 * of the row's ID column into a short code, and updating the database with
	 * the code. If successful, it returns the short code. If there is an error,
	 * an exception is thrown.
	 *
	 * @param string $url the long URL
	 *
	 * @return string the created short code
	 * @throws Exception if an error occurs
	 */
	protected function createShortCode( $url ) {
		return $this->insertUrlInDb( $url );
	}

	/**
	 * Inserts a new row into the database.
	 *
	 * Inserts a new row into the database with the URL and returns the ID of
	 * the row. If there is a database error an exception is thrown.
	 *
	 * @param string $url the long URL
	 *
	 * @return integer on success the integer of the newly inserted  row
	 * @throws Exception
	 */
	protected function insertUrlInDb( $url ) {
		$this->ci->load->helper( 'string' );
		$uniq_id = random_string( 'alnum', 10 );
		$ins_id  = $this->ci->common->insertTableData( self::$table, array( "long_url" => $url, "short_code" => $uniq_id ) );
		if ( $ins_id ) {
			return $uniq_id;
		} else {
			throw new \Exception( "Something went wrong." );
		}
	}

	/**
	 * Check to see if the supplied short code is a valid format
	 *
	 * @param string $code the short code
	 *
	 * @return boolean whether the short code is a valid format
	 */
	protected function validateShortCode( $code ) {
		return preg_match( "|[" . self::$chars . "]+|", $code );
	}

	/**
	 * Get the long URL from the database.
	 *
	 * Retrieve the URL associated with the short code from the database. If
	 * there is an error, an exception is thrown.
	 *
	 * @param string $code the short code to look for in the database
	 *
	 * @return string|boolean the long URL or false if it does not exist
	 */
	protected function getUrlFromDb( $code ) {
		$result = $this->ci->common->gettblrowdata( [ "short_code" => $code ], 'id, long_url', self::$table, 1, 0 );

		return ( empty( $result ) ) ? FALSE : $result;
	}

	/**
	 * Increment the record's access count.
	 *
	 * Increment the number of times a short code has been looked up to retrieve
	 * its URL.
	 *
	 * @param integer $id the ID of the row to increment
	 *
	 */
	protected function incrementCounter( $id ) {
		$this->ci->db->set( 'counter', 'counter+1', FALSE )->where( [ "id" => $id ] )->update( self::$table );
	}
}
