<?php

require_once APPPATH . '/libraries/RussianIntegrationService.php';

class SkifService extends RussianIntegrationService
{
    protected const STOP_STATUS_MAPPING = [
        12 => [self::PICKUP_GATE_IN],
        1 => [self::PICKUP, self::PICKUP_GATE_OUT],
        4 => [self::IN_TRANSIT],
        10 => [self::DELIVERY_GATE_IN],
        5 => [self::DELIVERY_GATE_OUT],
        13 => [self::DELIVERY]
    ];

    public function __construct()
    {
        parent::__construct();

        $this->logPrefix = 'SKIF';
    }

    /**
     * @throws SoapFault
     */
    private function initSoapClient(): SoapClient
    {
        return new SoapClient(
            SKIF_SERVER_IP,
            [
                'login' => SKIF_LOGIN,
                'proxy_host' => KN_PROXY_DNS,
                'proxy_port' => KN_PROXY_PORT,
                'password' => SKIF_PASSWORD,
                'soap_version' => SOAP_1_2,
                'cache_wsdl' => WSDL_CACHE_NONE, //WSDL_CACHE_MEMORY, //, WSDL_CACHE_NONE, WSDL_CACHE_DISK or WSDL_CACHE_BOTH
                'exceptions' => true,
                'trace' => 1
            ]
        );
    }

    /**
     * @throws SoapFault
     */
    public function getRequest(array $order): bool
    {
        // SkifService is not responding anymore
        // and we don't support Russian integrations
        return false;



        $traceOrder = $this->initSoapClient()->TraceOrder([
            'XMLString' => '<?xml version="1.0" encoding="UTF-8"?>
            <SkifCargo>
              <request>
                <OrderNo>' . $order['ref_value'] . '</OrderNo>
              </request>
            </SkifCargo>'
        ]);

        if (!$skifResult = $this->retrieveXml($traceOrder->return)) {
            return false;
        }

        /**
         * Status list from SKIF:
         * 1        Pickup
         * 4        In transit
         * 5, 13    Delivery
         * 10       GateIN Delivery
         * 12       GateIN Pickup
         */
        $errors = $this->validateXmlFields([
            'SenderTownName' => $skifResult['SenderTownName']
        ]);
        if (!empty($errors) || !$this->validateStatus($skifResult['CargoState'])) {
            log_error("SKIF - CargoState invalid value");
            return false;
        }

        $this->saveOrderStatus($skifResult, $order);

        return true;
    }

    /**
     *
     * Create new order status based on the CargoState of the SKIF result
     *
     * @param array $skifResult
     * @param array $order
     * @return bool
     */
    public function saveOrderStatus(array $skifResult, array $order): bool
    {
        date_default_timezone_set('Europe/Moscow');

        // Retrieve stop ids.
        if (($employee = $this->findEmployeeByBookingId($order['order_id'])) === null) {
            log_error('SKIF - No stops found for order "' . $order['id'] . '"');
            return false;
        }

        // Retrieve vehicle id and drive id.
        $vehicle = $this->findVehicleByShiftId($order['shift_id']);
        $driver = $vehicle !== null ? $this->findTruckDriverByVehicleId($vehicle->id) : null;

        // Create the status based on the mapping and the CargoState retrieved from SKIF.
        $stopStatusMappingsReset = array_values(self::STOP_STATUS_MAPPING);
        $skifCargoState = (int)$skifResult['CargoState'];

        // Retrieve the index from the $stopStatusMappings array since it has predefined keys.
        $index = array_search($skifCargoState, array_keys(self::STOP_STATUS_MAPPING));

        if ($index === false) {
            log_error('SKIF - CargoState (' . $skifCargoState . ') was not found within our system.');
            return false;
        }

        // Data for the stop status that should be inserted in the database.
        $data = [
            "order_id" => $order['id'],
            "shipment_id" => $order['shift_id'],
            "stop_id" => $employee->drop_stopid,
            "stop_detail_id" => $employee->id,
            "trip_id" => $order['trip_id'],
            "loc_name" => $skifResult['SenderTownName'],
            "vehicle_id" => $vehicle->vehicle_id,
            "driver_id" => $driver->id,
            "createdon" => date('Y-m-d H:i:s'),
            'reason' => 'SKIF EDI STATUS'
        ];

        // Loop through every stop status until the current one.
        // If a stop status doesn't exist in the database for the current order. Then we create it.
        for ($i = 0; $i <= $index; $i++) {
            foreach ($stopStatusMappingsReset[$i] as $stopStatus) {
                $currentStopStatus = $this->getStopStatuses()[$stopStatus];
                $stopStatusForOrder = $this->ci->db
                    ->select('id, status_code')
                    ->from('tb_stop_status')
                    ->where([
                        'order_id' => $order['id'],
                        'status_code' => $currentStopStatus['status_code']
                    ])
                    ->get()->result();

                if (empty($stopStatusForOrder)) {
                    $this->ci->db->insert(
                        'tb_stop_status',
                        array_merge($currentStopStatus, $data)
                    );
                }
            }
        }

        // When the CargoState from SKIF is 5 or 13 we close the corresponding order.
        if (in_array($skifCargoState, [5, 13])) {
            $this->ci->db
                ->where('id', $order['id'])
                ->update('tb_orders', ['trip_sts' => 1]);
        }

        return true;
    }
}
