<?php

namespace App\Libraries;

use Exception;
use RuntimeException;

/**
 * @throws Exception
 */
class TripNotificationXmlCacheService
{

    private $ci;
    private $userId;
    private $orderId;
    private $model;

    public function __construct()
    {
        $this->ci = &get_instance();
        $this->userId = $this->ci->session->userdata("user_id") ?? 0;
    }

    public function fetch(): self
    {
        $query = $this->ci->db->get_where(
            "tb_trip_notification_xml_cache",
            [
                'user_id' => $this->userId,
                'order_id' => $this->orderId
            ]
        );

        if (!$query) {
            throw new RuntimeException("Abort. Unable to use service");
        }

        $this->model = $query->row();

        if (!isset($this->model->message)) {
            $this->reCreateXml();
        }

        return $this;
    }

    public function store(string $xml): self
    {
        $dataset = [
            'user_id' => $this->userId ?? 0,
            'order_id' => $this->orderId ?? 0,
            'shift_id' => $this->getShiftIdByOrderId(),
            'message' => $xml
        ];

        $this->ci->db->insert(
            "tb_trip_notification_xml_cache",
            $dataset
        );

        $this->model = (object) $dataset;

        return $this;
    }

    public function forOrderId(int $orderId): self
    {
        $this->orderId = $orderId;

        return $this;
    }

    public function getXml(): string
    {
        if (!$this->model || !$this->model->message) {
            throw new \RuntimeException("XML message has empty body");
        }

        return $this->model->message;
    }

    private function getShiftIdByOrderId(): string
    {
        $order = $this->ci->db->query(
            "SELECT user_id, shift_id
            FROM tb_orders
            WHERE order_id= ?",
            [$this->orderId]
        )->row();

        if (!isset($order) || !checkAccessConditions('CWG_TripNotification_ACL', $order->user_id)) {
            return '';
        }

        return $order->shift_id;
    }

    private function reCreateXml(): self
    {
        $shiftId = $this->getShiftIdByOrderId();
        if (!$shiftId || !$this->orderId) {
            throw new RuntimeException("Abort. Please check Order id : {$this->orderId} and Shift id: {$shiftId}");
        }

        // generate Xml by carrierTripNotification
        $this->ci->load->library(
            'edi/carriernotification/CarrierTripNotification',
            'carriertripnotification'
        );

        $xml = $this->ci->carriertripnotification->generateXml($shiftId, '', '');
        $this->store($xml);

        return $this;
    }
}
