<?php if (!defined('BASEPATH')) exit('No direct script access allowed');

require_once APPPATH . 'libraries/UnitFormatter.php';

/**
 * Helper library used for conversion from metric to imperial units and optionally vice-versa.
 */
class UnitConverter
{
    /**
     * Base unit that conversion in based off of.
     *
     * @var string
     */
    private $base_unit;

    /**
     * Target unit of conversion, automatically selected based on user's region.
     * Can be same as base unit.
     *
     * @var string
     */
    private $target_unit;

    /**
     * Initial value for conversion.
     *
     * @var ?float
     */
    private $value;

    /**
     * Configuration value for conversion back to metric for the rest of the world.
     * Can be useful if provided values are already in imperial units.
     *
     * @var bool
     */
    private $metric_conversion = false;

    /**
     * Do a conversion.
     *
     * @param ?float $value
     * @param ?string $unit
     * @param bool $metric_conversion
     */
    public function __construct(?float $value, ?string $unit = null, bool $metric_conversion = false)
    {
        $this->metric_conversion = $metric_conversion;
        $this->base_unit = UnitFormatter::normalize_unit($unit);
        $this->target_unit = $this->choose_target_unit();
        $this->value = $this->convert_value($value);
    }

    /**
     * Returns converted value, without symbol string.
     *
     * @return ?float
     */
    public function get_value(): ?float
    {
        return $this->value;
    }

    /**
     * Returns just the converted symbol string.
     *
     * @return ?string
     */
    public function get_unit(): ?string
    {
        return $this->target_unit;
    }

    /**
     * Does an actual conversion based on automatically selected formula.
     *
     * @param ?float $value
     * @return ?float
     */
    private function convert_value(?float $value): ?float
    {
        // skip if it's dummy conversion - used when only proper unit symbol is needed
        if (is_null($value)) {
            return null;
        }

        // return initial value if there are no units set
        if (is_null($this->base_unit) or is_null($this->target_unit)) {
            return $value;
        }

        // use conversion formula if it exists for given units pair
        $formula = $this->get_formula();
        return $formula ? $formula($value) : $value;
    }

    /**
     * Selects appropriate formula based on base and target units.
     *
     * @return ?Closure
     */
    private function get_formula(): ?Closure
    {
        $formulas = [
            // length
            'cm' => [
                'in' => function($value) {
                    return $value * 0.393700787;
                },
            ],
            'in' => [
                'cm' => function($value) {
                    return $value * 2.54;
                },
            ],
            'm' => [
                'ft' => function($value) {
                    return $value * 3.280839895;
                },
            ],
            'ft' => [
                'm' => function($value) {
                    return $value * 0.3048;
                },
            ],
            'km' => [
                'mi' => function($value) {
                    return $value * 0.621371192;
                },
            ],
            'mi' => [
                'km' => function($value) {
                    return $value * 1.609344;
                },
            ],

            // velocity
            'kph' => [
                'mph' => function($value) {
                    return $value * 0.621371192;
                },
            ],
            'mph' => [
                'kph' => function($value) {
                    return $value * 1.609344;
                },
            ],

            // area
            'cm^2' => [
                'sqin' => function($value) {
                    return $value * 0.15500031;
                },
            ],
            'sqin' => [
                'cm^2' => function($value) {
                    return $value * 6.4516;
                },
            ],
            'm^2' => [
                'sqft' => function($value) {
                    return $value * 10.7639104;
                },
            ],
            'sqft' => [
                'm^2' => function($value) {
                    return $value * 0.09290304;
                },
            ],

            // volume
            'cm^3' => [
                'cuin' => function($value) {
                    return $value * 0.0610237441;
                },
            ],
            'cuin' => [
                'cm^3' => function($value) {
                    return $value * 16.387064;
                },
            ],
            'm^3' => [
                'cuft' => function($value) {
                    return $value * 35.3146667;
                },
            ],
            'cuft' => [
                'm^3' => function($value) {
                    return $value * 0.0283168466;
                },
            ],

            // weight
            'kg' => [
                'lb' => function($value) {
                    return $value * 2.20462262;
                },
            ],
            'lb' => [
                'kg' => function($value) {
                    return $value * 0.45359237;
                },
            ],

            // ...add more if needed
        ];

        if (isset($formulas[$this->base_unit]) and isset($formulas[$this->base_unit][$this->target_unit])) {
            return $formulas[$this->base_unit][$this->target_unit];
        }
        return null;
    }

    /**
     * Selects target imperial unit for conversion based on base metric unit.
     * Works both ways if metric conversion is enabled.
     *
     * @return ?string
     */
    private function choose_target_unit(): ?string
    {
        // unit pairs for automatic conversion
        $imperial_pairs = [
            'cm' => 'in',
            'm' => 'ft',
            'km' => 'mi',
            'cm^2' => 'sqin',
            'm^2' => 'sqft',
            'cm^3' => 'cuin',
            'm^3' => 'cuft',
            'kph' => 'mph',
            'kg' => 'lb',
            // ...add more if needed
        ];
        if ($this->metric_conversion) {
            $metric_pairs = array_flip($imperial_pairs);
            if (isset($metric_pairs[$this->base_unit])) {
                return $metric_pairs[$this->base_unit];
            }
        } else if ($this->should_use_imperial()) {
            if (isset($imperial_pairs[$this->base_unit])) {
                return $imperial_pairs[$this->base_unit];
            }
        }
        return $this->base_unit;
    }

    /**
     * Decides if logged in user should use imperial units or not.
     * Contains list of regions that should be taken into account.
     *
     * @return bool
     */
    private function should_use_imperial(): bool
    {
        $ci = &get_instance();
        $country_code = $ci->session->userdata('usr_tzone')['country'];
        return in_array($country_code, [
            'USA',
            // ...any other regions?
        ]);
    }
}
