<?php if (!defined('BASEPATH')) exit('No direct script access allowed');

class UnitFormatter
{
    /**
     * Normalized unit symbol.
     *
     * @var ?string
     */
    private $unit;

    /**
     * Processed value.
     *
     * @var ?float
     */
    private $value;

    /**
     * Process value and unit and round by specified decimal points.
     *
     * @param ?float $value
     * @param ?string $unit
     * @param int $decimals
     */
    public function __construct(?float $value, ?string $unit = null, int $decimals = 2)
    {
        $this->unit = self::normalize_unit($unit);
        $this->value = $this->process_value($value, $decimals);
    }

    /**
     * Returns formatted value, without symbol string.
     *
     * @return ?float
     */
    public function get_value(): ?float
    {
        return $this->value;
    }

    /**
     * Returns just the formatted symbol string.
     *
     * @return ?string
     */
    public function get_unit(): ?string
    {
        return $this->unit;
    }

    /**
     * Sometimes input string can be diffrerent for same unit.
     * This method normalizes that string to a common set of values.
     *
     * @param ?string $unit
     * @return ?string
     */
    public static function normalize_unit(?string $unit): ?string
    {
        if (empty($unit)) {
            return null;
        }

        // normalize used unit symbols to common values
        // based on values found in the database
        //$unit = mb_strtolower(trim($unit));
        $unit = trim($unit);
        $normalized_units = [
            'cm' => 'cm',
            'm' => 'm',
            'km' => 'km',
            'kms' => 'km',
            'in' => 'in',
            'inches' => 'in',
            'ft' => 'ft',
            'mi' => 'mi',
            'cm^3' => 'cm^3',
            'cm3' => 'cm^3',
            'm^3' => 'm^3',
            'cbm' => 'm^3',
            'mtq' => 'm^3',
            'cr' => 'm^3',
            'куб.м' => 'm^3',
            'sqin' => 'sqin',
            'sqft' => 'sqft',
            'cuin' => 'cuin',
            'cuft' => 'cuft',
            'cbf' => 'cuft',
            'kph' => 'kph',
            'mph' => 'mph',
            'g' => 'g',
            'gms' => 'g',
            'grm' => 'g',
            'kg' => 'kg',
            'lb' => 'lb',
            'lbs' => 'lb',
            'tons' => 'tons',
            'cmt' => 'cm',
            // ...add more if needed
        ];
        // ignore unknown units
        if (!isset($normalized_units[$unit])) {
            return $unit;
        }
        return $normalized_units[$unit];
    }

    /**
     * Return unit symbol string with proper formatting using HTML tags and entities.
     *
     * @param ?string $unit
     * @return string
     */
    public static function pretty_unit(?string $unit): string
    {
        if (is_null($unit)) {
            return '';
        }

        // pretty formatting for some of the unit symbols
        $pretty_formats = [
            'cm^2' => 'cm<sup>2</sup>',
            'm^2' => 'm<sup>2</sup>',
            'cm^3' => 'cm<sup>3</sup>',
            'm^3' => 'm<sup>3</sup>',
            'cuin' => 'cu&nbsp;in',
            'cuft' => 'cu&nbsp;ft',
            'kph' => 'km/h',
        ];
        return isset($pretty_formats[$unit]) ? $pretty_formats[$unit] : $unit;
    }

    /**
     * Magic function that returns value with formatted unit symbol.
     *
     * @return string
     */
    public function __toString()
    {
        if (is_null($this->value)) {
            return self::pretty_unit($this->unit);
        }
        return sprintf('%s&nbsp;%s', $this->value, self::pretty_unit($this->unit));
    }

    /**
     * Cleans up initial value.
     *
     * @return string
     */
    private function process_value(?float $value, int $decimals): ?float
    {
        if (is_null($value)) {
            return null;
        }
        return round($value, $decimals);
    }

}
