<?php

final class Visibilityservice
{
    public const PER_PAGE = 10;
    private object $CI;
    private array $userdata;

    public function __construct()
    {
        $this->CI = &get_instance();
        $this->CI->load->model([
            'common', 'ShipmentVisibilityModel',
            'vendors', 'vehicle', 'vehicles_drivers',
            'ship_order_stops','knopstripsmodel'
        ]);
    }

    public function getShiftsData(array $userdata): array
    {
        $this->userdata = $userdata;

        // get all IDs available
        $allShiftIds = $this->getShiftIdsAccessibleForUser(
            $this->CI->input->post(null, true)
        );

        // enable pagination, on what page am I ?
        $pageNumber = $this->setupPaginationByUrl(count($allShiftIds));

        // get and return Data of trips on that page
        $shiftsData = $this->getTripsList( $allShiftIds, $pageNumber );

        // filling gaps
        $vehicleUpdates = $this->getVehicleUpdates($shiftsData);
        $carrierUpdates = $this->getCarrierUpdates($shiftsData);
        $driverUpdates = $this->getDriverUpdates($shiftsData);
        $endtimeUpdates = $this->CI->ship_order_stops
            ->getEndtimeUpdates($shiftsData);

        $result = [];
        foreach ($shiftsData as $row) {
            $row['register_number'] = $vehicleUpdates[$row['vehicle_id'] ?? '-'] ?? $row['register_number'];
            $row['carrier_name'] = $carrierUpdates[$row['vendor_id'] ?? '-']['name'] ?? $row['carrier_name'];
            $row['carrier_broker'] = $carrierUpdates[$row['vendor_id'] ?? '-']['broker'] ?? $row['carrier_broker'];
            $row['driver_name'] = $driverUpdates[$row['id'] ?? '-'] ?? $row['driver_name'];
            if (isset($row['startdate'])) {
                $row['startdate'] = $this->updateTimestampDate($row['startdate']);
            }
            if (isset($row['enddate'])) {
                $row['enddate'] = $this->updateTimestampDate(
                    $endtimeUpdates[$row['id'] ?? '-'] ?? $row['enddate']
                );
            }

            $row['stops'] = [];

            $result [$row['id']] = $row;
        }

        return $result;
    }

    // replacement for getOrderIds from home_helper, as cache for multiple stop/drop
    public function updateShiftsDataAndStopstatusWithOrderId(array $shiftsData): array
    {
        $stopIds = [];
        foreach ($shiftsData as $row) {
            $stopId = $row['stops'][0]['id'] ?? false;
            if (!$stopId) {
                continue;
            }
            $stopIds [] = $stopId;
        }

        $cache = $this->CI->ship_order_stops
            ->getOrderIdsByMultipleStopDropStops($stopIds);

        foreach ($shiftsData as &$row) {
            $stopId = $row['stops'][0]['id'] ?? false;
            $row ['order_id'] = $cache[$stopId] ?? "";
        }

        return $shiftsData;
    }

    private function updateTimestampDate(string $timestamp): string
    {
        return getdatetimebytimezone(
            $this->userdata["usr_tzone"]['timezone'], $timestamp, DFLT_TZ
        )['datetime'];
    }

    private function getDriverUpdates(array $shiftsData): array
    {
        $shiftIds = array_unique(array_column($shiftsData, 'id'));
        $drivers = $this->CI->vehicles_drivers->getDriverIdsFromVehicleHistory($shiftIds);
        $names = $this->CI->vehicles_drivers->getDriverNames($drivers);

        foreach ($drivers as $shiftId => $driverId) {
            $drivers [$shiftId] = $names[$driverId] ?? $driverId;
        }

        return $drivers;
    }

    private function getCarrierUpdates(array $shiftsData):array
    {
        $vendorIds = array_unique(array_column(
            array_filter($shiftsData, static function($row) {
                return !trim($row['carrier_name']);
            }), 'vendor_id'));

        return $this->CI->vendors->getVendorNames($vendorIds);
    }

    private function getVehicleUpdates(array $shiftsData): array
    {
        $vehicleIds = array_unique(array_column(
            array_filter($shiftsData, static function($row) {
                return !$row['register_number'] && $row['vehicle_id'];
            }), 'vehicle_id'));

        return $this->CI->vehicle->getVehicleRegistrationNumbers($vehicleIds);
    }

    private function getShiftIdsAccessibleForUser(array $post): array
    {
        $userRoleId = $this->userdata['user_role_id'];
        if ($userRoleId === "5") {
            $shiftIds = $this->getShiftsBasedOnKNOps($post);
        } else {
            [$stopDetailUserIds, $userWhere] = $this->getUsersForShipmentsCondition();

            $shiftIds = $this->CI->ShipmentVisibilityModel->getNormalTripsFroVisibility(
                $post,
                $stopDetailUserIds,
                $userWhere,
                'tb_shifts s',
                $indexColumn = 's.id',
                $groupBy = 's.id',
                $excludeCountQuery = 'Yes'
            );
        }

        return empty($shiftIds) ? [] : $shiftIds;
    }

    private function getTripsList(array $shiftIds, int $pageNumber)
    {
        return $this->CI->ShipmentVisibilityModel->getFinalTripsList(
            $shiftIds,
            $pageNumber,
            self::PER_PAGE,
            'tb_shifts s',
            $indexColumn = 's.id',
            $groupBy = 's.id',
            $excludeCountQuery = 'Yes'
        );
    }

    private function setupPaginationByUrl(int $pageCount): int
    {
        $config = [
            "base_url" => base_url() . "visibility/shipment",
            "total_rows" => $pageCount,
            "per_page" => self::PER_PAGE,
            "uri_segment" => 3,
            'display_pages' => true,
        ];
        $this->CI->pagination->initialize($config);

        return (int) $this->CI->uri->segment(3);
    }

    private function getUsersForShipmentsCondition(): array
    {
        $companyCode = $this->userdata['company_code'];
        $userId = $this->userdata['user_id'];
        $adminIds = $this->userdata['country_user_ids'] ?? [];
        $hasUserAccessToCompanyTrips = in_array("trips", checkuserpermissions());
        $isCountryAdminWithCountryUsers = $this->userdata['business_type'] === 'Country Admin' && !empty($adminIds);

        $stopDetailUserIds = [];

        if ($hasUserAccessToCompanyTrips) {
            $userWhere = " s.company_code = '" . $companyCode . "' ";
            $getAllUsers = $this->CI->common->gettbldata(
                ['company_code' => $companyCode, 'status' => 'Active'],
                "id",
                "tb_users",
                0,
                0
            );
            foreach ($getAllUsers as $singleUser) {
                $stopDetailUserIds[] = $singleUser['id'];
            }
        } elseif ($isCountryAdminWithCountryUsers) {
            $users = implode(",", $adminIds);
            $userWhere = " s.user_id IN ($users) ";
            $stopDetailUserIds[] = $users;
            $stopDetailUserIds[] = $userId;
        } else {
            $userWhere = " s.user_id=$userId ";
            $stopDetailUserIds[] = $userId;
        }

        return [$stopDetailUserIds, $userWhere];
    }

    private function getShiftsBasedOnKNOps(array $post): array
    {
        $this->CI->load->model(['Datatables_model', 'knopstripsmodel']);
        $data = $this->CI->knopstripsmodel->getAllKnopsBasedOrderIds($this->userdata);
        $shiftIds = $data['allshiftIds'];
        if (empty($shiftIds)) {
            return [];
        }
        $normalTripsCondition = $this->CI->knopstripsmodel->getSearchConditionsForKNOpsInVisibilityPage($this->userdata, $post, $shiftIds);
        $joinsArray[] = [
            'table_name' => 'tb_shft_veh as sv',
            'condition' => "sv.shft_id=s.id AND sv.status=1",
            'join_type' => 'left'
        ];
        $joinsArray[] = [
            'table_name' => 'tb_vendors as v',
            'condition' => "v.id = sv.carrier_id",
            'join_type' => 'left'
        ];
        $joinsArray[] = [
            'table_name' => 'tb_orders as o',
            'condition' => "o.shift_id=s.id AND o.status>0",
            'join_type' => 'inner'
        ];

        $getRecordListing = $this->CI->Datatables_model->datatablesQuery(
            ['s.id'],
            [],
            'tb_shifts s',
            $joinsArray,
            implode(' AND ', $normalTripsCondition),
            's.id',
            'TRUE',
            's.id',
            'Yes',
            'POST'
        );
        $recordsData = $getRecordListing['data'] ?? [];
        $finalShiftIds = $this->CI->knopstripsmodel->getAllKNOpsBasedCrossBorderShiftIds($this->userdata, $post);
        foreach ($recordsData as $eachRow) {
            $finalShiftIds[] = $eachRow['id'];
        }
        return array_unique($finalShiftIds);
    }

}
