<?php

namespace SVKAPI\v1;

defined('BASEPATH') OR exit('No direct script access allowed');

use JMS\Serializer\Handler\DateHandler;
use Symfony\Component\Validator\Validation;
use JMS\Serializer\SerializerBuilder;
use JMS\Serializer\Handler\HandlerRegistry;
use SVKAPI\v1\Interfaces\InboundHandlerInterface;
use SVKAPI\v1\Exceptions\ValidationException;

abstract class BaseInboundHandler implements InboundHandlerInterface
{
    protected $CI;
    protected $userId = null;
    protected $messageBody;
    protected $modelClassName;
    protected $model;

    public function __construct()
    {
        $this->log('Initializing');

        $this->CI = &get_instance();
    }

    public function setUserId(int $userId): InboundHandlerInterface
    {
        $this->userId = $userId;
        return $this;
    }

    public function setMessageBody(string $body): InboundHandlerInterface
    {
        $this->messageBody = $body;
        return $this;
    }

    protected function deserializeData(): void
    {
        $this->log('Deserializing data');

        $serializer = SerializerBuilder::create()->configureHandlers(function(HandlerRegistry $registry) {
            $registry->registerSubscribingHandler(new DateHandler(\DateTime::ATOM, 'UTC', false));
        })->build();

        $this->model = $serializer->deserialize($this->messageBody, $this->modelClassName, 'xml');
    }

    protected function validateData(): void
    {
        $this->log('Validating data');

        $validator = Validation::createValidatorBuilder()->enableAnnotationMapping()->getValidator();
        $errors = $validator->validate($this->model);
        if (count($errors) > 0) {
            throw new ValidationException((string) $errors);
        }
    }

    public function log(string $message, string $level = 'debug')
    {
        $className = get_class($this);
        log_message($level, "{$className}: {$message}");
    }
}
