<?php
namespace SVKAPI\v1\Mappings;

use SVKAPI\v1\Exceptions\MissingDataException;
use SVKAPI\v1\Interfaces\MappingInterface;
use SVKAPI\v1\Models\Address;
use SVKAPI\v1\Models\DangerousGoods;
use SVKAPI\v1\Models\DeliveryStop;
use SVKAPI\v1\Models\NumberOfPackages;
use SVKAPI\v1\Models\PickupStop;
use SVKAPI\v1\Models\Reference;
use SVKAPI\v1\Models\Trip;
use SVKAPI\v1\Models\Order;
use SVKAPI\v1\Models\OrderGood;
use SVKAPI\v1\Models\ContactDetails;

class OrdersMapping implements MappingInterface
{
    /**
     * Global Object for CI unstance
     */
    private $CI;

    /**
     */
    public function __construct()
    {
        $this->CI = &get_instance();
        $this->CI->load->model([
            'cargo_details',
            'common',
            'order',
            'order_cargo_details',
            'order_references',
        ]);
    }

    /**
     * @param int $shiftId
     * @param Trip $model
     * @return Trip
     */
    public function map(int $shiftId, Trip &$model): Trip
    {
        $orders = [];
        $entities = $this->CI->order->base_query(['shift_id' => $shiftId], false)
            ->select([
                'tb_orders.id', 'tb_orders.order_id', 'tb_orders.user_id',
                'tb_orders.pickup_datetime', 'tb_orders.pickup_endtime',
                'tb_orders.delivery_datetime', 'tb_orders.drop_endtime',
                'tb_orders.pickup_company', 'tb_orders.pickup_country', 'tb_orders.pickup_city',
                'tb_orders.pickup_pincode', 'tb_orders.pickup_address1', 'tb_orders.pickup_address2',
                'tb_orders.delivery_company', 'tb_orders.delivery_country', 'tb_orders.delivery_city',
                'tb_orders.delivery_pincode', 'tb_orders.delivery_address1', 'tb_orders.delivery_address2',
                'tb_orders.volume', 'tb_orders.weight', 'tb_orders.company_code', 'tb_orders.branch_code',
                'tb_orders.plat', 'tb_orders.plng', 'tb_orders.dlat', 'tb_orders.dlng',
                'tb_orders.transport_mode',
                'tb_service_master.service_id', 'tb_service_master.name AS service_name',
            ])
            ->join('tb_order_details', 'tb_orders.id=tb_order_details.order_row_id', 'LEFT')
            ->join('tb_service_master', 'tb_order_details.service=tb_service_master.id', 'LEFT')
            ->get()
            ->result_array();

        if (empty($entities)) {
            throw new MissingDataException('Trip needs to contain at least one order');
        }

        foreach ($entities as $entity) {
            $orderReferences = $this->CI->order_references->find([
                'order_id' => $entity['id']
            ]);
            $references = array_combine(
                array_column($orderReferences, 'reference_id'),
                array_column($orderReferences, 'ref_value')
            );
            $this->CI->edi_logger->setUser_id($entity['user_id']);
            $this->CI->edi_logger->setTxn_obj_id($entity['id']);
            $this->CI->edi_logger->setCompany_code($entity['company_code']);
            $this->CI->edi_logger->setBranch_code($entity['branch_code']);

            // NOTE: can't use JOIN on main orders query, because of "illegal mix of collations" error
            $transportMode = $this->CI->common->gettblrowdata(['code' => $entity['transport_mode']], 'name', 'tb_transportmode', 0, 0);

            $order = new Order([
                'orderId' => $entity['order_id'], // public order id
                'legId' => $references['INN'] ?? '',
                'serviceLevel' => $entity['service_id'] ? "{$entity['service_id']}-{$entity['service_name']}" : '',
                'serviceType' => $transportMode['name'] ?? '',
                'companyCode' => $entity['company_code'],
                'branch' => $entity['branch_code'],
                'trackingNumber' => $references['TRACKING_NUMBER'] ?? '',
            ]);
            $order->setOrderGoods($this->mapOrderGoods($entity['id']));
            $order->setPickup($this->mapPickup($entity, $references, $orderReferences));
            $order->setDelivery($this->mapDelivery($entity, $references, $orderReferences));

            $orders[] = $order;
        }

        $model->setOrders($orders);

        return $model;
    }

    /**
     * @param int $orderId
     * @return OrderGood
     */
    private function mapOrderGoods(int $orderId) : OrderGood
    {
        $orderCargoDetails = $this->CI->order_cargo_details->find([
            'order_id' => $orderId
        ]);
        $cargoDetails = !empty($orderCargoDetails) ? $this->CI->cargo_details->find([
            'id' => array_column($orderCargoDetails, 'cargo_id')
        ]) : [];

        return new OrderGood([
            'dangerous' => !empty(array_filter($cargoDetails, function ($item){ return $item->dg_goods == 1; })),
            'dangerousGoods' => new DangerousGoods([
                'numberOfPackages' => new NumberOfPackages([
                    'value' => 0,
                    'quantityType' => 'PCs'
                ]),
                'uNNumber' => '',
                'class' => '',
                'shippingName' => '',
                'limitedQuantity' => '',
                'exceptedQuantity' => ''
            ]),
            'stackable' => !empty(array_filter($cargoDetails, function ($item){ return $item->stackable == 1; })),
            'handlingUnit' => 'PCs',
            'quantity' => array_sum(array_column($orderCargoDetails, 'quantity')),
            'volumeCBM' => array_sum(array_column($orderCargoDetails, 'volume')),
            'weightKGS' => array_sum(array_column($orderCargoDetails, 'weight'))
        ]);
    }

    /**
     * @param Order $entity
     * @param array $references
     * @return PickupStop
     */
    private function mapPickup(array $entity, array $references, array $orderReferences) : PickupStop
    {
        $pickupReferences = array_map(function($ref) {
            if ($ref->ref_belongs_to == 'SHIPPER') {
                return new Reference([
                    'code' => $ref->reference_id,
                    'value' => $ref->ref_value
                ]);
            } else {
                return [];
            }
        }, $orderReferences);

        return new PickupStop([
            'sequenceNumber' => '',
            'type' => 'PICKUP',
            'dateTimeFrom' => (new \DateTime($entity['pickup_datetime'])),
            'dateTimeTo' => new \DateTime($entity['pickup_endtime']),
            'contactDetails' => new ContactDetails([
                'name' => $references['SHIPPER_NAME'] ?? '',
                'phone' => $references['TRUCKINGWAYPOINT_PUP_CT_TE'] ?? ''
            ]),
            'address' => new Address([
                'addressName' => $entity['pickup_company'],
                'address1' => $entity['pickup_address1'] ?? '',
                'address2' => '',
                'postal' => $entity['pickup_pincode'] ?? '',
                'city' => $entity['pickup_city'] ?? '',
                'state' => $entity['pickup_address2'] ?? '',
                'country' => $entity['pickup_country'] ?? '',
                'timezone' => '',
                'latitude' => $entity['plat'] ?? '',
                'longitude' => $entity['plng'] ?? ''
            ]),
            'pickupReferences' => array_filter($pickupReferences),
            'carrierInstructions' => $references['ORD_PIKINST'] ?? ''
        ]);
    }

    /**
     * @param array $entity
     * @param array $references
     * @return DeliveryStop
     */
    private function mapDelivery(array $entity, array $references, array $orderReferences) : DeliveryStop
    {
        $pickupReferences = array_map(function($ref) {
            if ($ref->ref_belongs_to == 'CONSIGNEE') {
                return new Reference([
                    'code' => $ref->reference_id,
                    'value' => $ref->ref_value
                ]);
            } else {
                return [];
            }
        }, $orderReferences);
        return new DeliveryStop([
            'sequenceNumber' => '',
            'type' => 'DELIVERY',
            'dateTimeFrom' => new \DateTime($entity['delivery_datetime']),
            'dateTimeTo' => new \DateTime($entity['drop_endtime']),
            'contactDetails' => new ContactDetails([
                'name' => $references['CONSIGNEE_NAME'] ?? '',
                'phone' => $references['TRUCKINGWAYPOINT_DEL_CT_TE'] ?? ''
            ]),
            'address' => new Address([
                'addressName' => $entity['delivery_company'],
                'address1' => $entity['delivery_address1'] ?? '',
                'address2' => '',
                'postal' => $entity['delivery_pincode'] ?? '',
                'city' => $entity['delivery_city'] ?? '',
                'state' => $entity['delivery_address2'] ?? '',
                'country' => $entity['delivery_country'] ?? '',
                'timezone' => '',
                'latitude' => $entity['dlat'] ?? '',
                'longitude' => $entity['dlng'] ?? ''
            ]),
            'pickupReferences' => array_filter($pickupReferences),
            'carrierInstructions' => $references['ORD_DLVINST'] ?? ''
        ]);
    }
}
