<?php
namespace SVKAPI\v1\Mappings;

use SVKAPI\v1\Models\Trip;
use SVKAPI\v1\Exceptions\MissingConfigException;
use SVKAPI\v1\Exceptions\MissingDataException;

class TripMapping extends AbstractMapping
{
    /**
     * Global Object for CI instance
     */
    private $CI;

    /**
     */
    public function __construct()
    {
        $this->CI = &get_instance();
        $this->CI->load->model([
            'order',
            'order_references',
            'trips_carrier_details',
            'trips_model',
            'truck_drivers',
            'trucks_data',
            'vendors',
        ]);
    }

    /**
     * @param int $entityId
     * @param Trip $model
     * @return Trip
     * @throws MissingDataException
     */
    public function map(int $entityId, Trip &$model): Trip
    {
        if (!defined('STATUS_API_ENDPOINT') || !defined('POD_API_ENDPOINT')) {
            throw new MissingConfigException('Missing endpoints config for current environment');
        }

        $carrierDetails = $this->CI->trips_carrier_details->find_by_shift_id($entityId);
        $carrierDetails = reset($carrierDetails);

        $shift = $this->CI->trips_model->find_by_id($entityId);
        $shift = reset($shift);
        if (empty($shift)) {
            throw new MissingDataException('Trip data is missing');
        }

        $vendor = $this->CI->vendors->find_by_id($shift->vendor_id);
        $vendor = reset($vendor);
        if (empty($vendor)) {
            throw new MissingDataException('Vendor data is missing');
        }

        $driver = !empty($carrierDetails) ? $this->CI->truck_drivers->find_by_id($carrierDetails->driver_id) : [];
        $driver = reset($driver);

        $vehicle = !empty($carrierDetails) ? $this->CI->trucks_data->find_by_id($carrierDetails->vehicle_id) : [];
        $vehicle = reset($vehicle);

        $references = $this->getReferences($entityId);

        $model->setTripId($shift->shipmentid); // public trip id
        $model->setSenderId('');
        $model->setReceiverId('');
        //$model->setMessageId(''); // Moved to root function
        $model->setDriverId($driver->name ?? '');
        $model->setVehicleId($vehicle->register_number ?? '');
        $model->setAction($shift->is_carrier_notified ? 'UPDATE': 'CREATE');
        $model->setStatusApiEndpoint(STATUS_API_ENDPOINT);
        $model->setPodApiEndpoint(POD_API_ENDPOINT);
        $model->setCarrierCode($vendor->code);
        $model->setCarrierName($vendor->name);
        $model->setCarrierInstructions(implode(', ', $references));
        $model->setAdditionalConditions('');

        return $model;
    }

    /**
     * @param int $shiftId
     * @return array
     */
    private function getReferences(int $shiftId) : array
    {
        $orders = $this->CI->order->base_query([
            'shift_id' => $shiftId
        ], false)->select(['id'])->get()->result_array();
        if (empty($orders)) {
            return [];
        }

        $orderIds = array_column($orders, 'id');
        $references = $this->CI->order_references->base_query([
            'order_id' => $orderIds,
            'reference_id' => 'TRANS_COND_TRIP_NOTE'
        ])->select([
            'ref_value'
        ])->get()->result_array();

        return array_column($references, 'ref_value');
    }
}
