<?php
defined('BASEPATH') or exit('No direct script access allowed');

class ExpressBeesEDIServices
{
    private $CI;
    //private $carrier      = 'BUSYBEES LOGISTICS SOLUTIONS PVT. LTD';
    private $carrier      = 'EXPRESS BEES';
    private $appKey       = '';
    private $apiKey       = '';
    private $baseUrl      = '';
    private $code         = '';

    public function __construct()
    {
        $this->CI = &get_instance();

        $this->CI->load->model(['Singlestatusmodel', 'Order']);
        $this->CI->load->library(['Edi_logger', 'uuid', 'apis/FlipkartEDIServices']);
        $this->CI->load->helper('user_helper');

        $query = $this->CI->db->get_where(
            'tb_auth_tokens',
            ['token_for' => $this->carrier]
        );

        $tokenRow = $query->row_array();

        $this->appKey  = $tokenRow['auth_token'] ?? '';
        $this->apiKey  = $tokenRow['api_key'] ?? '';
        $this->baseUrl = $tokenRow['base_url'] ?? '';
        $this->code    = $tokenRow['code'] ?? '';
    }

    public function createOrder(array $orderData, string $orderId, int $id): void
    {
        $pickupDate = $orderData['sourceInfo']['EstimatedDateTime']['From']['DateTime'] ?? '';

        if (empty($pickupDate) || $pickupDate === '0000-00-00 00:00:00') {
            $pickupDate = date('d-m-Y H:i:s');
        }

        $dt = new DateTime($pickupDate);

        $pickup = $orderData['sourceInfo'] ?? [];
        $drop   = $orderData['destinationInfo'] ?? [];

        $pickup_name    = $pickup['Company']['Name'] ?? '';
        $pickup_street  = $pickup['Address']['Address2'] ?? '';
        $pickup_city    = $pickup['Address']['City'] ?? '';
        $pickup_state   = $pickup['Address']['State'] ?? '';
        $pickup_pincode = $pickup['Address']['Postal'] ?? '';
        $pickup_contact = $pickup['Address']['ContactInfo']['ContactNo'] ?? '';
        $pickup_email   = $pickup['Address']['ContactInfo']['EmailAddress'] ?? '';

        $drop_name    = $drop['Company']['Name'] ?? '';
        $drop_street  = $drop['Address']['Address2'] ?? '';
        $drop_city    = $drop['Address']['City'] ?? '';
        $drop_state   = $drop['Address']['State'] ?? '';
        $drop_pincode = $drop['Address']['Postal'] ?? '';
        $drop_contact = $drop['Address']['ContactInfo']['ContactNo'] ?? '';
        $drop_email   = $drop['Address']['ContactInfo']['EmailAddress'] ?? '';

        $thuData = $orderData['cargoDetails']['CargoThuDetails'] ?? [];
        $thuData = isset($thuData[0]) ? $thuData : [$thuData];
        
        $refData = $orderData['manageReferences']['References'] ?? [];
        $refData = isset($refData[0]) ? $refData : [$refData];
        
        foreach ($refData as $j => $ref) {
            $refName = $ref['RefType']['Code'];
            if($refName == "AWB"){
                $awbValue = $ref['RefType']['Value'];
            }
        }

        $lbhData     = [];
        $totalLength = $totalWidth = $totalHeight = $totalWeight = $totalQuantity = $totalGoodsValue = 0;
        $count = 0;
        foreach ($thuData as $i => $cargo) {
            $count = $count + 1;
            $length     = (float) ($cargo['Length'] ?? 0);
            $width      = (float) ($cargo['Width'] ?? 0);
            $height     = (float) ($cargo['Height'] ?? 0);
            $weight     = (float) ($cargo['Weight'] ?? 0);
            $quantity   = (int) ($cargo['Quantity'] ?? 0);
            $goodsValue = (int) ($cargo['ValueOfGoods'] ?? 0);

            $totalLength     += $length;
            $totalWidth      += $width;
            $totalHeight     += $height;
            $totalWeight     += $weight;
            $totalQuantity   += $quantity;
            $totalGoodsValue += $goodsValue;
            
            $lbhData[] = [  
                "ShippingId"                 => "S" . $orderId,
                "Length"                     => $this->formatNumber($length),
                "Breadth"                    => $this->formatNumber($width),
                "Height"                     => $this->formatNumber($height),
                "LBHUnit"                    => "CM",
                "MPSValue"                   => "",
                "PhysicalWeight"             => $this->formatNumber($weight),
                "PhysicalWeightUnit"         => "KG"
            ];
        }
        
        $payload = [
            "ParentAWB"                          => $awbValue ?? $orderId,
            "BookingMode"                        => "Credit",
            "RouteMode"                          => "Surface",
            "Addresses" => [
                [
                    "Type"                       => "Shipper address",
                    "CompanyName"                => trim($pickup_name ?: ""),
                    "SPOCPersonName"             => trim($pickup_name ?: ""),
                    "Address"                    => trim($pickup_street ?: ""),
                    "State"                      => trim($pickup_state ?: ""),
                    "City"                       => trim($pickup_city ?: ""),
                    "Pincode"                    => trim($pickup_pincode ?: ""),
                    "MobileNo"                   => trim($pickup_contact ?: ""),
                    "EmailID"                    => trim($pickup_email ?: ""),
                    "TINNo"                      => "" 
                ],
                [
                    "Type"                       => "Consignee address",
                    "CompanyName"                => trim($drop_name ?: ""),
                    "SPOCPersonName"             => trim($drop_name ?: ""),
                    "Address"                    => trim($drop_street ?: ""),
                    "State"                      => trim($drop_state ?: ""),
                    "City"                       => trim($drop_city ?: ""),
                    "Pincode"                    => trim($drop_pincode ?: ""),
                    "MobileNo"                   => trim($drop_contact ?: ""),
                    "EmailID"                    => trim($drop_email ?: ""),
                    "TINNo"                      => ""
                ]
            ],
            "PickUpDateTime"                     => $pickupDate,
            "InvoiceValue"                       => $totalGoodsValue,
            "AWBPhysicalWeight"                  => $this->formatNumber($totalWeight),
            "AWBPhysicalWeightUnit"              => "KG",
            "Category"                           => "",
            "MPSCount"                           => $count,
            "MPSNO"                              => $lbhData,
            "RiskSurcharge"                      => "",
            "GSTMultiSellerInfo" => [
                [
                    "InvoiceNo"                  => "",
                    "InvoiceDate"                => "",
                    "InvoiceValue"               => $totalGoodsValue,
                    "IsSellerRegUnderGST"        =>"", 
                    "SellerGSTRegNumber"         => "",  
                    "BuyerGSTRegNumber"          => "", 
                    "SupplyStatePlace"           => "",
                    "EBN"                        => 0,  
                    "EBNExpiryDate"              => "",
                    "SellerName"                 => "",
                    "SellerAddress"              => "",
                    "SellerPincode"              => "",
                    "SupplySellerStatePlace"     => "",
                    "Unit"                       => "KG",
                    "HSNDetails" => [
                        [
                            "HSNCode"            => "", 
                            "ProductCategory"    => "",
                            "ProductDescription" => "",
                            "TaxableValue"       => 0.00,
                            "SGSTAmount"         => 0.00, 
                            "CGSTAmount"         => 0.00,
                            "IGSTAmount"         => 0.0,
                            "Discount"           => 0.0,
                            "GSTTaxRateCGSTN"    => 0,
                            "GSTTaxRateSGSTN"    => 0,
                            "GSTTAXRateIGSTN"    => 0,
                            "GSTTaxTotal"        => 0
                        ]
                    ]
                ]
            ]
        ];
       
        $vendor      = $this->CI->db->get_where('tb_vendors', ['code' => $this->code])->row_array();
        $companyCode = $vendor['company_code'] ?? '';
        $branchCode  = $vendor['branch_code'] ?? '';
        $userId      = $vendor['user_id'] ?? '';

        $responseData = [];

        try {
            $serverURL = $this->baseUrl;
            log_message("error", "serverURL:".$serverURL);
            log_message("error", "Payload:".json_encode($payload));
            $ch = curl_init($serverURL);
            curl_setopt_array($ch, [
                CURLOPT_HTTPHEADER     => [
                    'Content-Type: application/json',
                    "XBKey: $this->appKey",
                    "Version: v1",
                ],
                CURLOPT_POST           => true,
                CURLOPT_POSTFIELDS     => json_encode($payload),
                CURLOPT_RETURNTRANSFER => true,
            ]);

            $responseRaw = curl_exec($ch);
            
            log_message("error", "ResponseRaw:".json_encode($responseRaw));

            if ($responseRaw === false) {
                throw new Exception(curl_error($ch));
            }

            curl_close($ch);
            $responseData = json_decode($responseRaw, true);
         
            if (
                empty($responseData) ||
                $responseData['ReturnMessage'] == 'Failed'
            ) {
                log_message('error', 'Invalid API response: ' . $responseRaw);
                $errorCode = $responseData['ReturnCode'] ?? '';
                $errorMsg  = $responseData['MPSAWBNumberReturnMessage']['ShippingId'] ?? 'Unknown error';

                log_message(
                    'error',
                    "API Failure | Code: {$errorCode} | Message: {$errorMsg}"
                );
                return;
            }
        } catch (Exception $e) {
            log_message('error', 'Express Bees Exception: ' . $e->getMessage());
        } finally {
            $this->logEdiTransaction(
                json_encode($payload),
                $responseData,
                $companyCode,
                $branchCode,
                $userId,
                'Express Bees Carrier Order Request'
            );
        }
        
        $expressPickupResponseBookingId = $awbValue ?? $orderId;

        log_message("error", "$expressPickupResponseBookingId: " . $expressPickupResponseBookingId);

        if (! empty($expressPickupResponseBookingId) && $expressPickupResponseBookingId !== '') {

            $express_tkn_ref = [
                'order_id'     => $id,
                'reference_id' => 'TKN',
                'ref_value'    => $expressPickupResponseBookingId,
                'createdon'    => $cdate,
            ];

            $insTkn = $this->CI->db->insert('tb_order_references', $express_tkn_ref);
        }
    }

    public function formatNumber($num)
    {
        $rounded = round((float) $num, 2);
        return fmod($rounded, 1) === 0.0 ? (int) $rounded : number_format($rounded, 2, '.', '');
    }

    private function logEdiTransaction(string $request, array $response, string $companyCode, string $branchCode, string $userId, string $ediName): void
    {
        $this->CI->edi_logger->setEdi_type(1);
        $this->CI->edi_logger->setTransaction_id(time());
        $this->CI->edi_logger->setEdi_name($ediName);
        $this->CI->edi_logger->setBounded_type(1);
        $this->CI->edi_logger->setCompany_code($companyCode);
        $this->CI->edi_logger->setBranch_code($branchCode);
        $this->CI->edi_logger->setUser_id($userId);
        $this->CI->edi_logger->setEdi_format_type('JSON');
        $status = isset($responseData['MPSAWBNumberReturnMessage']) && $responseData['MPSAWBNumberReturnMessage'] === 'SUCCESSFUL' ? 1 : 0;
        $this->CI->edi_logger->setStatus($status);
        $this->CI->edi_logger->setEdi_request($request);
        $this->CI->edi_logger->setEdi_response(json_encode($response));
        $this->CI->edi_logger->setObj_type_name($ediName);
        $this->CI->edi_logger->saveToEdiLogs();
    }

    public function getStatus(string $docketNo, string $orderId, int $id): int
    {
        $query = $this->CI->db->get_where(
            'tb_auth_tokens',
            ['token_for' => 'EXPRESS BEES TRACK']
        );

        $tokenRow = $query->row_array();

        $this->appKey  = $tokenRow['auth_token'] ?? '';
        $this->baseUrl = $tokenRow['base_url'] ?? '';
        
        $message       = '';
        $response      = '';
        $apiResponse   = [];
        $companyCode   = $this->CI->session->userdata('company_code');
        $branchCode    = $this->CI->session->userdata('branch_code');
        $userId        = $this->CI->session->userdata('user_id');
        $chk_qry_order = $this->CI->db->get_where('tb_orders', ['id' => $id])->row_array();
        $orderId       = $chk_qry_order['order_id'];
        if ($userId == null || $userId == "") {
            $userId = $chk_qry_order['user_id'];
        }
        if ($companyCode == null || $companyCode == "") {
            $companyCode = $chk_qry_order['company_code'];
        }
        if ($branchCode == null || $branchCode == "") {
            $branchCode = $chk_qry_order['branch_code'];
        }

        $url = $this->baseUrl;

        $payload = [
            'AWBNumber' => $docketNo,
            'Type'      => 'Parent',
        ];
        
        $jsonPayload = json_encode($payload);
        
        log_message('error', 'Raw Request Body: ' . $jsonPayload);
        
        try{
        
            $ch = curl_init($url);
            
            curl_setopt_array($ch, [
                CURLOPT_CUSTOMREQUEST  => 'POST',           // ✅ REQUIRED
                CURLOPT_POSTFIELDS     => $jsonPayload,      // ✅ RAW JSON BODY
                CURLOPT_RETURNTRANSFER => true,
                CURLOPT_FOLLOWLOCATION => true,
                CURLOPT_TIMEOUT        => 30,
                CURLOPT_HTTPHEADER     => [
                    'Content-Type: application/json',
                    'Accept: application/json',
                    'Content-Length: ' . strlen($jsonPayload),
                    "XBKey: {$this->appKey}",
                    'Version: V1',
                ],
            ]);
            
            $response = curl_exec($ch);
            
            if ($response === false) {
                log_message('error', 'Curl Error: ' . curl_error($ch)); // ✅ fixed
                curl_close($ch);
                return 0;
            }
            
            curl_close($ch);
            
            log_message('error', 'Response: ' . $response);
            
            $data = json_decode($response, true);

    
            if ($data === null) {
                log_message('error', 'Express Tracking Invalid JSON: ' . $response);
                return 0;
            }
    
            if (! isset($data['CargoManifestDetails']['ReturnMessage']) || $data['CargoManifestDetails']['ReturnMessage'] !== 'Successful') {
                log_message('error', 'Express Tracking API failed: ' . json_encode($data));
                return 0;
            }
           
    
            $statusList = $data['CargoManifestDetails']['ParentShipmentDetails'][0]['StatusDetails'] ?? null;
            if (isset($statusList)) {
                $statusList = $statusList;
            }
            
            log_message("error", "statusList: " . json_encode($statusList));
            
            $statusInput = $this->generateStatusXML($docketNo, $statusList);
            $xml         = simplexml_load_string($statusInput);
            log_message("error", "xml: " . $xml);
            return $this->getStatusXml($xml, $orderId, $this->carrier);

        } catch (Exception $e) {
            $message = 'Error: ' . $e->getMessage();
        } finally {
            $this->CI->edi_logger->setEdi_type(1);
            $this->CI->edi_logger->setTransaction_id(time());
            $this->CI->edi_logger->setEdi_name('Express Bees Carrier Tracking Status');
            $this->CI->edi_logger->setBounded_type(1);
            $this->CI->edi_logger->setCompany_code($companyCode);
            $this->CI->edi_logger->setBranch_code($branchCode);
            $this->CI->edi_logger->setUser_id($userId);
            $this->CI->edi_logger->setEdi_format_type('JSON');
            $this->CI->edi_logger->setStatus(((isset($data['CargoManifestDetails']['ReturnMessage']) && $data['CargoManifestDetails']['ReturnMessage']) ? 1 : 0));
            $this->CI->edi_logger->setEdi_request($docketNo);
            $this->CI->edi_logger->setEdi_response(json_encode($response));
            $this->CI->edi_logger->setObj_type_name('Express Bees Carrier Tracking Status');
            $this->CI->edi_logger->saveToEdiLogs();
        }
        
        return "";
    }

    
    public function generateStatusXML(string $docketNo, array $statusList): string
    {
        log_message('error', 'statusList: ' . json_encode($statusList));
        $xml  = '<SVKEDIResponseMessage>';
        $xml .= '<SVKEDITransmissionHeader>';
        $xml .= '<EDIVersion>SVK2.0</EDIVersion>';
        $xml .= '<AckSpec><AckOption>SUCCESS</AckOption></AckSpec>';
        $xml .= '<SourceApp>EXPRESS BEES</SourceApp>';
        $xml .= '<DestinationApp>SVKONEKT</DestinationApp>';
        $xml .= '<Action>BookingStatusUpdate</Action>';
        $xml .= '</SVKEDITransmissionHeader>';
        $xml .= '<SVKEDITransmissionBody><Order>';
        $xml .= '<OrderID>' . $docketNo . '</OrderID>';

        foreach ($statusList as $status) {
            $xml .= '<Status>';
            $xml .= '<StatusCode>' . ($status['StatusCode'] ?? '') . '</StatusCode>';
            $xml .= '<StatusValue>' . ($status['StatusCode'] ?? '') . '</StatusValue>';
            $xml .= '<StatusType>' . ($status['StatusCode'] ?? '') . '</StatusType>';
            $xml .= '<DateTime></DateTime>';
            $xml .= '<TimeZone>+00:00</TimeZone>';
            $xml .= '<Location></Location>';
            $xml .= '</Status>';
        }

        $xml .= '</Order></SVKEDITransmissionBody></SVKEDIResponseMessage>';
        log_message('error', 'XML: ' . $xml);
        return $xml;
    }

    public function getStatusXml(object $xmlData, int $orderId, string $carrier): int
    {
        $statusResponse = 0;

        $body = $xmlData->SVKEDITransmissionBody ?? null;
        if (! $body || empty($body->Order)) {
            log_message('error', 'SVKEDITransmissionBody or Order missing');
            return 0;
        }

        $orders = is_array($body->Order) || $body->Order instanceof Traversable
            ? $body->Order
            : [$body->Order];

        foreach ($orders as $orderNode) {

            $externalOrderId = trim((string) ($orderNode->OrderID ?? ''));

            $chkOrder = $this->CI->db
                ->get_where('tb_orders', ['order_id' => $externalOrderId])
                ->row_array();
            
            if (! $chkOrder) {
                log_message('error', 'Order not found in DB using externalOrderId: ' . $externalOrderId);
                continue;
            }
            
            $orderIdDb = $chkOrder['order_id'];
            $ordersData = [];

            if (empty($orderNode->Status)) {
                continue;
            }

            $statuses = is_array($orderNode->Status) || $orderNode->Status instanceof Traversable
                ? $orderNode->Status
                : [$orderNode->Status];

            foreach ($statuses as $status) {

                $statusCode = (string) ($status->StatusCode ?? '');
                if ($statusCode === '') {
                    continue;
                }

                $ordersData[] = [
                    "OrderID"     => $orderIdDb,
                    "statustype"  => (string) ($status->StatusType ?? ''),
                    "statuscode"  => $statusCode,
                    "statusvalue" => (string) ($status->StatusValue ?? ''),
                    "datetime"    => (string) ($status->DateTime ?? ''),
                    "timezone"    => (string) ($status->TimeZone ?? ''),
                    "lat"         => (string) ($status->Lat ?? ''),
                    "lng"         => (string) ($status->Lng ?? ''),
                    "location"    => (string) ($status->Location ?? ''),
                ];
            }

            if (! empty($ordersData)) {
                $statusResponse = $this->CI
                    ->Singlestatusmodel
                    ->updateXPStatus(json_encode($ordersData), $carrier);
            }
        }

        return $statusResponse;
    }
}
