<?php
defined('BASEPATH') or exit('No direct script access allowed');

class IwayExxEDIServices
{
    private $CI;
    private $returnOrderTypeId = '16';
    private $carrier           = 'I-WAY_LOGISTICS';
    private $customerCode      = 'K1012';
    private $appKey            = '';
    private $baseUrl           = '';

    public function __construct()
    {
        $this->CI = &get_instance();

        $this->CI->load->model(['Singlestatusmodel', 'Order']);
        $this->CI->load->library(['Edi_logger', 'uuid']);
        $this->CI->load->helper('user_helper');

        $query = $this->CI->db->get_where(
            'tb_auth_tokens',
            ['token_for' => $this->carrier]
        );

        $tokenRow = $query->row_array();

        $this->appKey  = $tokenRow['auth_token'] ?? '';
        $this->baseUrl = $tokenRow['base_url'] ?? '';
        $this->code    = $tokenRow['code'] ?? '';
    }

    public function createOrder(array $orderData, string $orderId, int $id): void
    {
        $iwayRefNo  = "ORD-" . (string) $orderId;
        $pickupDate = '';

        if (
            isset($orderData['sourceInfo']['EstimatedDateTime']['From']['DateTime']) &&
            is_string($orderData['sourceInfo']['EstimatedDateTime']['From']['DateTime']) &&
            $orderData['sourceInfo']['EstimatedDateTime']['From']['DateTime'] !== ''
        ) {
            $pickupDate = $orderData['sourceInfo']['EstimatedDateTime']['From']['DateTime'];
        }

        $totalQuantityValue = $orderData['cargoSummary']['TotalQuantity']['Value'];
        $totalWeightValue   = $orderData['cargoSummary']['TotalWeight']['Value'];

        $pickup_name    = $orderData['sourceInfo']['Company']['Name'];
        $pickup_address = $orderData['sourceInfo']['Address']['Address1'] . "," . $orderData['sourceInfo']['Address']['Street'];
        $pickup_street  = $orderData['sourceInfo']['Address']['Address2'];
        $pickup_city    = $orderData['sourceInfo']['Address']['City'];
        $pickup_state   = $orderData['sourceInfo']['Address']['State'];
        $pickup_country = $orderData['sourceInfo']['Address']['Country'];
        $pickup_pincode = $orderData['sourceInfo']['Address']['Postal'];
        $drop_name      = $orderData['destinationInfo']['Company']['Name'];
        $drop_address   = $orderData['destinationInfo']['Address']['Address1'] . "," . $orderData['destinationInfo']['Address']['Street'];
        $drop_street    = $orderData['destinationInfo']['Address']['Address2'];
        $drop_city      = $orderData['destinationInfo']['Address']['City'];
        $drop_state     = $orderData['destinationInfo']['Address']['State'];
        $drop_country   = $orderData['destinationInfo']['Address']['Country'];
        $drop_pincode   = $orderData['destinationInfo']['Address']['Postal'];
        $drop_contact   = $orderData['destinationInfo']['Address']['ContactInfo']['ContactNo'];
        //$carrierCode    = $orderData['header']['CarrierDetails']['ID'];

        //log_message("error", "pickup_datetime: " . $pickupDate);

        if ($pickupDate === '' || $pickupDate === '0000-00-00 00:00:00') {
            $pickupDate = date('Y-m-d H:i:s');
        }

        $dt          = new DateTime($pickupDate);
        $pickup_date = $dt->format('Y-m-d');
        $pickup_time = $dt->format('H:i:s');

        $iwayPickupRequestBooking = [
            "AgentCode"          => "24001",
            "CustomerCode"       => "K1012",
            "Password"           => "K1012",
            "BookingDate"        => $pickup_date,
            "BookingTime"        => $pickup_time,
            "Pcs"                => (int) $totalQuantityValue,
            "Weight"             => formatNumber($totalWeightValue < 1 ? 1 : $totalWeightValue),
            "CODAmount"          => 0.00,
            "TopayAmount"        => 0.0,
            "ModeName"           => "ROAD",
            "ProductName"        => "AIR",
            "BookType"           => "D",
            "ReferenceNo"        => (string) $iwayRefNo,
            "ConsignorName"      => $pickup_name,
            "ConsignorAddress1"  => $pickup_address ?: "",
            "ConsignorAddress2"  => $pickup_street ?: "",
            "ConsignorPincode"   => $pickup_pincode,
            "ConsigneeName"      => $drop_name,
            "ConsigneeAddress"   => $drop_address ?? "",
            "ConsigneePhone"     => $drop_contact ?? "",
            "ConsigneePinCode"   => $drop_pincode,
            "Remarks"            => null,
            "DocketNumber"       => "",
            "lstVolumetric_Size" => null,
            "lstBook_ItemList"   => null,
            "IsTesting"          => 0,
        ];

        //log_message("error", "iwayPickupRequestBooking:" . json_encode($iwayPickupRequestBooking));

        $jsonData = json_encode($iwayPickupRequestBooking);
        //log_message("error", "jsonData:" . json_encode($jsonData));
        $vendor = $this->CI->db->get_where("tb_vendors", ['code' => $this->code])->row_array();

        $companyCode = $vendor['company_code'] ?? '';
        $branchCode  = $vendor['branch_code'] ?? '';
        $userId      = $vendor['user_id'] ?? '';

        $iwayPickupRequestBookingResponse = [];
        try {
            $ch = curl_init($this->baseUrl . '/PostBooking');
            curl_setopt_array($ch, [
                CURLOPT_HTTPHEADER => [
                    "Content-Type: application/json",
                    "X-APPKEY: {$this->appKey}",
                ],
                CURLOPT_POST           => true,
                CURLOPT_POSTFIELDS     => $jsonData,
                CURLOPT_RETURNTRANSFER => true,
            ]);

            $responseRaw = curl_exec($ch);

            //log_message("error", "responseRaw:" . json_encode($responseRaw));

            if ($responseRaw === false) {
                $iwayPickupRequestBookingResponse = ['success' => false, 'error' => curl_error($ch)];
            } else {
                $iwayPickupRequestBookingResponse = json_decode($responseRaw, true) ?: [
                    'success' => false,
                    'error'   => 'Invalid JSON from API',
                    'raw'     => $responseRaw,
                ];
            }

            curl_close($ch);
            log_message("error", "IwayExx Order Response: " . json_encode($iwayPickupRequestBookingResponse));

        } catch (Exception $e) {
            log_message("error", "IwayExx Order Exception: " . $e->getMessage());
        } finally {
            $this->logEdiTransaction($jsonData, $iwayPickupRequestBookingResponse, $companyCode, $branchCode, $userId, 'IwayExx Carrier Order Request');
        }

        $iwayPickupResponseBookingId = $iwayPickupRequestBookingResponse['DocketNumber'];

        log_message("error", "iwayPickupResponseBookingId: " . $iwayPickupResponseBookingId);

        if (! empty($iwayPickupResponseBookingId) && $iwayPickupResponseBookingId !== '') {

            $iway_tkn_ref = [
                'order_id'     => $id,
                'reference_id' => 'TKN',
                'ref_value'    => $iwayPickupResponseBookingId,
                'createdon'    => $cdate,
            ];

            $insTkn = $this->CI->db->insert('tb_order_references', $iway_tkn_ref);
        }
    }

    public function getStatus(string $docketNo, int $orderId): int
    {
        if (empty($docketNo) || $orderId <= 0) {
            log_message('error', "Invalid DocketNo");
            return 0;
        }

        $url = $this->baseUrl . '/TrackConsignment?' . http_build_query([
            'Status'       => 'L',
            'lstAWB'       => $docketNo,
            'customerCode' => $this->customerCode,
        ]);

        $curl = curl_init($url);
        curl_setopt_array($curl, [
            CURLOPT_RETURNTRANSFER => true,
            CURLOPT_HTTPGET        => true,
            CURLOPT_HTTPHEADER     => [
                "Content-Type: application/json",
                "X-APPKEY: {$this->appKey}",
            ],
            CURLOPT_TIMEOUT => 30,
        ]);

        $response = curl_exec($curl);
        curl_close($curl);

        if ($response === false) {
            log_message('error', 'Curl Error: ' . curl_error($curl));
            return 0;
        }

        $data = json_decode($response, true);
        //log_message('error', "API Response: " . $response);

        $responseStatus = $data['ResponseStatus'] ?? [];
        $trackingResult = $data['TrackingResult'][0] ?? [];

        if (($responseStatus['Message'] ?? '') !== 'SUCCESS') {
            return 0;
        }

        $statusList = $trackingResult['TrackList'] ?? [];

        $statusInput = $this->generateStatusXML($docketNo, $statusList);

        $xml = simplexml_load_string($statusInput);
        return $this->getStatusXml($xml, $orderId, $this->carrier);
    }

    private function generateStatusXML(string $docketNo, array $statusList): string
    {
        $xml = '<SVKEDIResponseMessage>';
        $xml .= '<SVKEDITransmissionHeader>';
        $xml .= '<EDIVersion>SVK2.0</EDIVersion>';
        $xml .= '<AckSpec><AckOption>SUCCESS</AckOption></AckSpec>';
        $xml .= '<SourceApp>' . $this->carrier . '</SourceApp>';
        $xml .= '<DestinationApp>SVKONEKT</DestinationApp>';
        $xml .= '<Action>BookingStatusUpdate</Action>';
        $xml .= '</SVKEDITransmissionHeader>';
        $xml .= '<SVKEDITransmissionBody><Order>';
        $xml .= '<OrderID>' . $docketNo . '</OrderID>';

        foreach ($statusList as $status) {
            $status['ActivityCode'] = "BOK";
            $status['ActivityDate'] = DateTime::createFromFormat('d/m/Y H:i:s', $status['ActivityDate'])->format('Y-m-d H:i:s');
            $xml .= '<Status>';
            $xml .= '<StatusCode>' . ($status['ActivityCode'] ?? '') . '</StatusCode>';
            $xml .= '<StatusValue>' . ($status['ActivityCode'] ?? '') . '</StatusValue>';
            $xml .= '<StatusType>' . ($status['ActivityCode'] ?? '') . '</StatusType>';
            $xml .= '<DateTime>' . ($status['ActivityDate'] ?? '') . '</DateTime>';
            $xml .= '<TimeZone>+00:00</TimeZone>';
            $xml .= '<Location>' . ($status['ActivityFromLocation'] ?? '') . '</Location>';
            $xml .= '</Status>';
        }

        $xml .= '</Order></SVKEDITransmissionBody></SVKEDIResponseMessage>';

        log_message('error', "Generated Status XML: " . $xml);
        return $xml;
    }

    private function logEdiTransaction(string $request, array $response, string $companyCode, string $branchCode, string $userId, string $ediName): void
    {
        $this->CI->edi_logger->setEdi_type(1);
        $this->CI->edi_logger->setTransaction_id(time());
        $this->CI->edi_logger->setEdi_name($ediName);
        $this->CI->edi_logger->setBounded_type(1);
        $this->CI->edi_logger->setCompany_code($companyCode);
        $this->CI->edi_logger->setBranch_code($branchCode);
        $this->CI->edi_logger->setUser_id($userId);
        $this->CI->edi_logger->setEdi_format_type('JSON');
        $status = isset($response['ResponseStatus']['Message']) && $response['ResponseStatus']['Message'] === 'SUCCESS' ? 1 : 0;
        $this->CI->edi_logger->setStatus($status);
        $this->CI->edi_logger->setEdi_request($request);
        $this->CI->edi_logger->setEdi_response(json_encode($response));
        $this->CI->edi_logger->setObj_type_name($ediName);
        $this->CI->edi_logger->saveToEdiLogs();
    }

    public function getStatusXml(object $xmlData, int $orderId, string $carrier): int
    {
        $statusResponse = 0;
        $body           = $xmlData->xpath('SVKEDITransmissionBody');

        if (empty($body)) {
            return 0;
        }

        foreach ($body[0]->Order as $order) {
            $chkOrder = $this->CI->db->get_where('tb_orders', ['id' => $orderId])->row_array();
            if (! $chkOrder) {
                continue;
            }

            $ordersData = [];
            $orderIdDb  = $chkOrder['order_id'];

            foreach ($order->Status as $status) {
                $statusCode = (string) ($status->StatusCode ?? '');
                if (! $statusCode) {
                    continue;
                }

                $ordersData[] = [
                    "OrderID"     => $orderIdDb,
                    "statustype"  => (string) ($status->StatusType ?? ''),
                    "statuscode"  => $statusCode,
                    "statusvalue" => (string) ($status->StatusValue ?? ''),
                    "datetime"    => (string) ($status->DateTime ?? ''),
                    "timezone"    => (string) ($status->TimeZone ?? ''),
                    "lat"         => (string) ($status->Lat ?? ''),
                    "lng"         => (string) ($status->Lng ?? ''),
                    "location"    => (string) ($status->Location ?? ''),
                ];
            }

            if (! empty($ordersData)) {
                $statusResponse = $this->CI->Singlestatusmodel->updateXPStatus(json_encode($ordersData), $carrier);
            }
        }

        return $statusResponse;
    }
}
