<?php

namespace App\Libraries\infrastructure;

use DateTime;

class ScheduledJobLoggerService implements ScheduledJobLoggerInterface
{
    private static $instance;

    /** @var \Scheduledjob $scheduledJob */
    private $scheduledJob;

    /** @var \Scheduledjobhistory $scheduledJobHistory */
    private $scheduledJobHistory;

    private $ci;

    /**
     * @param string $scheduledJobName
     */
    private function __construct(string $scheduledJobName)
    {
        try {
            $this->loadModels();
            $this->setScheduledJob($scheduledJobName);
            $this->setScheduledJobHistory();
        } catch (\Exception $exception) {
            echo $exception->getMessage();
        }
    }

    /**
     * @param string $scheduleJobName
     * @return ScheduledJobLoggerService
     */
    public static function &instance(string $scheduleJobName): ScheduledJobLoggerService
    {
        if (!self::$instance) {
            self::$instance = new ScheduledJobLoggerService($scheduleJobName);
        }

        return self::$instance;
    }

        private function loadModels(): void
    {
        $this->ci =& get_instance();
        $this->ci->load->model('scheduledjob');
        $this->ci->load->model('scheduledjobhistory');
        $this->ci->load->model('scheduledjobdetailedlog');
    }


    /**
     * @param array $data
     * @return bool
     * @throws \Exception
     */
    private function logLine(array $data): bool
    {
        if (empty($data) || $this->scheduledJobHistory === null) {
            return false;
        }

        $data['scheduled_job_history_id'] = $this->scheduledJobHistory->getId();
        $scheduledJobDetailedLog = $this->ci->scheduledjobdetailedlog->insert($data);

        if (empty($scheduledJobDetailedLog)) {
            throw new \Exception("ERROR: ScheduledJobLoggerService::logline | line not logged");
        }

        return true;
    }

    /**
     * @param string $scheduledJobName
     * @throws \Exception
     */
    private function setScheduledJob(string $scheduledJobName): void
    {
        $scheduledJob = $this->ci->scheduledjob->instance();
        if (empty($scheduledJob)) {
            throw new \Exception("ERROR: ScheduledJobLoggerService::setScheduledJob | job not found");
        }
        $scheduledJob->setScheduledJobName($scheduledJobName);
        $this->scheduledJob = $scheduledJob;
    }

    /**
     * @throws \Exception
     */
    private function setScheduledJobHistory(): void
    {
        if (empty($this->scheduledJob)) {
            throw new \Exception("ERROR: ScheduledJobLoggerService::setScheduledJobHistory | scheduledJob not set");
        }

        $scheduledJobHistory = $this->ci->scheduledjobhistory->createScheduledJobHistory($this->scheduledJob->getId());
        if (empty($scheduledJobHistory)) {
            throw new \Exception("ERROR: ScheduledJobLoggerService::setScheduledJobHistory | scheduledJobHistory not created");
        }
        $this->scheduledJobHistory = $scheduledJobHistory;
    }

    public function startJob(): void
    {
        try {
            $this->scheduledJobHistory->setStartTime(date("Y-m-d H:i:s"));
        } catch (\Exception $exception) {
            echo $exception->getMessage();
        }
    }

    /**
     * @param bool $success
     */
    public function finishJob(bool $success = true): void
    {
        try {
            $status = $success ? $this->scheduledJobHistory::STATUS_SUCCESS : $this->scheduledJobHistory::STATUS_FAILED;
            $this->scheduledJobHistory->setEndTime(date("Y-m-d H:i:s"), $status);
        } catch (\Exception $exception) {
            echo $exception->getMessage();
        }
    }

    /**
     * @param string $entity_name
     * @param string|null $message
     */
    public function log(string $entity_name, string $message = null): void
    {
        try {
            if (!$this->logLine(["entity_name" => $entity_name, "message" => $message])) {
                throw new \Exception("ERROR: ScheduledJobLoggerService::log | log went wrong");
            }
        } catch (\Exception $exception) {
            echo $exception->getMessage();
        }
    }

    /**
     * @param int $days
     * @return bool
     */
    public function purgeOldLogs(int $days): bool
    {
        if ($days < 7) {
            self::log("purgeOldLogs", "Number of days is below 7, must be 7 minimum.");
            $days = 7;
        }
        $highestId = $this->scheduledJobHistory->getMaxIdFromOldJobs(date("Y-m-d H:i:s", strtotime('-' . $days . ' days')))->result_array();
        if (isset($highestId[0]['id'])) {
            try {
                if ($this->ci->scheduledjobdetailedlog->massdelete($highestId[0]['id'])) {
                    if($this->scheduledJobHistory->massdelete($highestId[0]['id'])) {
                        self::log("purgeOldLogs", "Successfully deleted Scheduled Job Histories and its loglines");
                        return true;
                    }
                }
            } catch (\Exception $exception) {
                echo $exception->getMessage();
                return false;
            }
        }
        return true;
    }
}
