<?php

defined('BASEPATH') or exit('No direct script access allowed');

class OpenboxModel extends CI_Model
{
    public function __construct()
    {
        $this->load->helper(['user', 'Log']);
    }

    public function saveOrder(array $data): bool
    {
        $checkOrderExist = $this->OpenboxModel->getCount(
            'tb_orders',
            ['external_order_id' => $data['external_order_id'], 'status <>' => 0]
        );

        if ($checkOrderExist != 0) {
            log_error('OpenBox Order Create : External order id already exist.');
            return false;
        }

        $this->db->trans_start();
        $order = $this->db->query(
            'INSERT INTO tb_orders(
					external_order_id,
					goods_value,
					currency,
					shipper_id,
					pickup_custid,
					pickup_partyid,
					drop_custid,
					drop_partyid,
					customer_id,
					pickup_country,
					pickup_city,
					pickup_pincode,
					pickup_company,
					pickup_address1,
					pickup_address2,
					delivery_country,
					delivery_city,
					delivery_pincode,
					delivery_company,
					delivery_address1,
                    delivery_address2,
					quantity,
					volume,
					weight,
					pickup_datetime,
					pickup_endtime,
					delivery_datetime,
					drop_endtime,
					user_id,
					company_code,
					branch_code,
					product,
					status,
					is_created,
                    physicalreceiver,
                    LogicalReceiver,
                    physicalsender,
                    logicalsender,
                    transport_mode,
					createdon,
                    sequence_number,
                    created_source
                      )
                 VALUES(?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?)',
            [
                $data['external_order_id'],
                $data['goods_value'],
                $data['currency'],
                $data['shipper_id'],
                $data['pickup_custid'],
                $data['pickup_partyid'],
                $data['drop_custid'],
                $data['drop_partyid'],
                $data['customer_id'],
                $data['pickup_country'],
                $data['pickup_city'],
                $data['pickup_pincode'],
                $data['pickup_company'],
                $data['pickup_address1'],
                $data['pickup_address2'],
                $data['delivery_country'],
                $data['delivery_city'],
                $data['delivery_pincode'],
                $data['delivery_company'],
                $data['delivery_address1'],
                $data['dropState'],
                $data['quantity'],
                $data['volume'],
                $data['weight'],
                $data['pickup_datetime'],
                $data['pickup_endtime'],
                $data['delivery_datetime'],
                $data['drop_endtime'],
                $this->openBoxAdminUseId(),
                $data['company_code'],
                $data['branch_code'],
                $data['product'],
                1,
                1,
                $data['PhysicalReceiver'],
                $data['LogicalReceiver'],
                $data['PhysicalSender'],
                $data['LogicalSender'],
                $data['OrderDetails']['ModeOfTransport'],
                $data['createdon'],
                $data['SequenceNumber'],
                $data['created_source']
            ]
        );
        if (!$order) {
            return $this->rollback("OpenBox Order Create : Save order Trans status failed rollback");
        }
        $orderId = $this->db->insert_id();

        $bookingId = generatebookingid(
        			[
        				"user_id" => $this->openBoxAdminUseId(),
        				"order_id" => $orderId,
        				"country_code" => 'UK',
        				"company_code" => 'UKKN-CG',
        			]
        		);

        $updateOrderId = $this->db->query(
            'UPDATE tb_orders SET order_id = ? WHERE id = ?',
            [$bookingId, $orderId]
        );

        if (!$updateOrderId) {
            return $this->rollback("OpenBox Order Create : Booking id updating failed Trans status rollback");
        }

        $data['bookingRowId'] = $orderId;
        $data['bookingId'] = $bookingId;

        if (!$this->saveOrderDetails($data)) {
            return $this->rollback("OpenBox Order Create : Save order Details failed Trans status rollback");
        }

        $shipmentId = $this->saveOrUpdateShipment($data);
        if (!is_int($shipmentId) || $shipmentId <= 0) {
            return $this->rollback("OpenBox Order Create : Save Shipment failed Trans status rollback");
        }

        $this->db->query(
            'UPDATE tb_orders SET shipment_id = ? WHERE id = ?',
            [$shipmentId, $data['bookingRowId']]
        );

        if (!$this->saveCargo($data)) {
            return $this->rollback("OpenBox Order Create : Save Cargo failed Trans status rollback");
        }

        if (!$this->saveReferences($data)) {
            return $this->rollback("OpenBox Order Create : Save References failed Trans status rollback");
        }

        if (!$this->saveOrderPartys($data)) {
            return $this->rollback("OpenBox Order Create : Save Order party types failed Trans status rollback");
        }

        if (!$this->saveInvolvedPartys($data)) {
            return $this->rollback("OpenBox Order Create : Involved parties types failed Trans status rollback");
        }

        $this->saveRemarks($data);

        $this->db->trans_commit();
        log_info("OpenBox Order Create : Trans status success");

        return true;
    }

    private function rollback(string $logMessage): bool
    {
        $this->db->trans_rollback();
        log_error($logMessage);
        return false;
    }

    private function saveOrderDetails(array $data): bool
    {
        if (!isset($data['bookingId']) || '0' === $data['bookingId'] || '' === $data['bookingId']) {
            log_error('OpenBox Order Create : Save order details data unavailable.');
            return false;
        }

        $partyData = $this->getShipperConsineeId($data['shipper_id'], 'SHIPPER');
        $deliveryTerms = $data['OrderDetails']['DeliveryTerms'] ?? null;

        switch ($deliveryTerms) {
            case 'Shipper':
                $deliveryTermsId = 10;
                break;
            case 'Consignee':
                $deliveryTermsId = 12;
                break;
            default:
                $deliveryTermsId = 82;
        }

        if ($data['Action'] ==  OpenboxMessageHandler::ACTION_CREATE) {
            return $this->db->query(
                'INSERT INTO tb_order_details(order_row_id, order_id, order_status, shipper_id, department_code, order_type, service,
                             delivery_term, incoterm, status, createdon) VALUES(?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?)',
                [
                    $data['bookingRowId'],
                    $data['bookingId'],
                    'Pending',
                    $partyData['partyId'],
                    $data['departmentCode'],
                    $data['orderTypeRowId'],
                    $this->getServiceId($data['serviceType']),
                    $deliveryTermsId,
                    $data['incoterm'],
                    1,
                    $data['createdon'],
                ]
            );
        } else {
            return $this->db->query(
                'UPDATE tb_order_details SET
    		order_status = ?,
     		shipper_id = ?,
       		department_code = ?,
       		order_type = ?,
    		service = ?,
			delivery_term = ?,
			incoterm = ?,
			updatedon = ? WHERE order_row_id = ?',
                [
                    'Pending',
                    $partyData['partyId'],
                    $data['departmentCode'],
                    $data['orderTypeRowId'],
                    $this->getServiceId($data['serviceType']),
                    $deliveryTermsId,
                    $data['incoterm'],
                    $data['createdon'],
                    $data['bookingRowId'],
                ]
            );
        }
    }

    private function saveOrUpdateShipment(array $data): int
    {
        if (isset($data['shipment_id']) && !empty($data['shipment_id'])) {
            $this->db->query(
                'UPDATE tb_shipments SET
                         trucktype = ?,
                         transport_mode = ?,
                         product = ?,
                         freight_term = ?,
                         freight_termname = ?,
                         incoterm = ?,
                         modeoftransport = ?,
                         createdon = ? WHERE id = ?',
                [
                    $data['modeOfTrasportName'] ?? '',
                    $data['OrderDetails']['ModeOfTransport'],
                    $data['OrderDetails']['OrderProductType'],
                    $data['OrderDetails']['TermsOfTrade']['FreightName']['Term'],
                    $data['OrderDetails']['TermsOfTrade']['FreightName']['Name'],
                    $data['OrderDetails']['TermsOfTrade']['Incoterm'],
                    $data['modeofTransportId'],
                    $data['createdon'],
                    $data['shipment_id'],
                ]
            );

            return (int)$data['shipment_id'];
        }

        $shipmentId = 'UK' . rand(0, time());
        $this->db->query(
            'INSERT INTO tb_shipments(
                         shipid,
                         txnid,
                         trucktype,
                         pickupcnt,
                         dropcnt,
                         unitspec,
                         insertusr,
                         carrier,
                         insertuserdate,
                         enddate,
                         insdate,
                         upddate,
                         reason,
                         purpose,
                         ship_object,
                         logdate,
                         transport_mode,
                         domainname,
                         company_code,
                         branch_code,
                         product,
                         freight_term,
                         freight_termname,
                         incoterm,
                         modeoftransport,
                         createdon
                         )
                          VALUES(?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?)',
            [
                $shipmentId,
                $shipmentId,
                $data['modeOfTrasportName'] ?? '',
                '1',
                '1',
                1,
                $data['shipperId'] ?? '0',
                '0',
                $data['createdon'],
                date('Y-m-d H:i:s', strtotime('+1 day')),
                $data['createdon'],
                $data['createdon'],
                'SHIPMENT',
                'SEND INTEGRATION',
                'SHIPMENT',
                $data['createdon'],
                $data['OrderDetails']['ModeOfTransport'],
                $data['KNOrgDetails']['BranchCode'],
                $data['KNOrgDetails']['CompanyCode'],
                $data['KNOrgDetails']['BranchCode'],
                $data['OrderDetails']['OrderProductType'],
                $data['OrderDetails']['TermsOfTrade']['FreightName']['Term'],
                $data['OrderDetails']['TermsOfTrade']['FreightName']['Name'],
                $data['OrderDetails']['TermsOfTrade']['Incoterm'],
                $data['modeofTransportId'],
                $data['createdon'],
            ]
        );

        return (int)$this->db->insert_id();
    }

    private function saveCargo(array $data): bool
    {
        $cargos = $data['OrderDetails']['OrderCargoDetails']['Items'];
        $userId = $this->openBoxUserId();

        foreach ($cargos as $cargo) {
            if ((isset($cargos['CargoType']) && empty($cargos['CargoType'])) ||
                (isset($cargo['CargoType']) && empty($cargo['CargoType']))
            ) {
                log_error('OpenBox Order Create : Cargo Type must set a value or default value.');
                return false;
            }

            $CargoDescription = $cargos['CargoDescription'] ?? ($cargo['CargoDescription'] ?? '');

            if (strlen($CargoDescription) > 200) {
                log_error('OpenBox Order Create : Cargo Description must set a value or default value.');
                return false;
            }

            $quantity = $cargo['Quantity'] ?? $cargos['Quantity'];
            if (empty($quantity) &&
                strlen($quantity) > 60
            ) {
                $quantity = 0;
            }

            $handlingUnit = $cargos['HandlingUnit'] ?? $cargo['HandlingUnit'];
            if (empty($handlingUnit) ||
                is_array($handlingUnit)
            ) {
                $handlingUnit = '';
            }

            $dangerousGoodsFlag = $cargos['DangerousGoodsFlag'] ?? $cargo['DangerousGoodsFlag'];

            if (empty($dangerousGoodsFlag) ||
                is_array($dangerousGoodsFlag)
            ) {
                $dangerousGoodsFlag = 0;
            }

            $stackable = $cargos['Stackable'] ?? $cargo['Stackable'];

            if (empty($stackable) ||
                is_array($stackable)
            ) {
                $stackable = 0;
            }

            $splittable = $cargos['Splittable'] ?? $cargo['Splittable'];

            if (empty($splittable) && is_array($splittable)) {
                $splittable = 0;
            }

            $grounded = $cargos['Grounded'] ?? $cargo['Grounded'];

            if (empty($grounded) && is_array($grounded)) {
                $grounded = 0;
            }

            $lengthValue = $cargos['Length']['Value'] ?? $cargo['Length']['Value'];

            if (empty($lengthValue) && is_array($lengthValue) || $lengthValue < 0) {
                $lengthValue = 0;
            }

            $lengthUom = $cargos['Length']['UOM'] ?? $cargo['Length']['UOM'];

            if (empty($lengthUom) && is_array($lengthUom)) {
                $lengthUom = 'M';
            }

            $widthValue = $cargos['Width']['Value'] ?? $cargo['Width']['Value'];

            if (empty($widthValue) && is_array($widthValue) || $widthValue < 0) {
                $widthValue = 0;
            }

            $widthhUom = $cargos['Width']['UOM'] ?? $cargo['Width']['UOM'];

            if (empty($widthhUom) && is_array($widthhUom)) {
                $widthhUom = 'M';
            }

            $heightValue = $cargos['Height']['Value'] ?? $cargo['Height']['Value'];

            if (empty($heightValue) && is_array($heightValue) || $heightValue < 0) {
                $heightValue = 0;
            }

            $heightUom = $cargos['Height']['UOM'] ?? $cargo['Height']['UOM'];

            if (empty($heightUom) && is_array($heightUom)) {
                $heightUom = 'M';
            }

            $weightValue = $cargos['Weight']['Value'] ?? $cargo['Weight']['Value'];

            if (empty($weightValue) && is_array($weightValue) || $weightValue < 0) {
                $weightValue = 0;
            }

            $weightUom = $cargos['Weight']['UOM'] ?? $cargo['Weight']['UOM'];

            if (empty($weightUom) && is_array($weightUom)) {
                $weightUom = 'Kg';
            }

            $actualWeightValue = $cargos['ActualWeight']['Value'] ?? $cargo['ActualWeight']['Value'];

            if (empty($actualWeightValue) && is_array($actualWeightValue) || $actualWeightValue < 0) {
                $actualWeightValue = 0;
            }

            $actualWeightUom = $cargos['ActualWeight']['UOM'] ?? $cargo['ActualWeight']['UOM'];
            if (empty($actualWeightUom) && is_array($actualWeightUom)) {
                $actualWeightUom = 'Kg';
            }

            $actualVolumeValue = $cargos['ActualVolume']['Value'] ?? $cargo['ActualVolume']['Value'];

            if (empty($actualVolumeValue) && is_array($actualVolumeValue) || $actualVolumeValue < 0) {
                $actualVolumeValue = 0;
            }

            $actualVolumeUom = $cargos['ActualVolume']['UOM'] ?? $cargo['ActualVolume']['UOM'];

            if (empty($actualVolumeUom) && is_array($actualVolumeUom)) {
                $actualVolumeUom = 'cmd';
            }

            $volumetricWeightValue = $cargos['VolumetricWeight']['Value'] ?? $cargo['VolumetricWeight']['Value'];

            if (empty($volumetricWeightValue) && is_array($volumetricWeightValue) || $volumetricWeightValue < 0) {
                $volumetricWeightValue = 0;
            }

            $volumetricWeightUom = $cargos['VolumetricWeight']['UOM'] ?? $cargo['VolumetricWeight']['UOM'];

            if (empty($volumetricWeightUom) && is_array($volumetricWeightUom)) {
                $volumetricWeightUom = 'cmd';
            }

            $ldm = $cargos['LDM'] ?? $cargo['LDM'];

            if ((empty($ldm) && $ldm != 0) || is_array($ldm)) {
                $ldm = "";
            }

            $cargoDetails = $this->db->query(
                'INSERT INTO tb_cargo_details(
                    cargo_type,
					goods_description,
					quantity,
					length,
					length_unit,
					width,
					width_unit,
					height,
					height_unit,
					weight,
					weight_unit,
					stackable,
					splittable,
                    grounded,
					createdby,
					createdon,
					ldm,
                    dg_goods,
					second_weight,
					secondweight_uom,
					second_volume,
					secondvolume_uom,
                    volumetric_weight,
                    volweight_uom
                         )
                          VALUES(?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?,?, ?)',
                [
                    $cargo['CargoType'] ?? $cargos['CargoType'],
                    trim( (string) $CargoDescription ),
                    $quantity,
                    $lengthValue,
                    $lengthUom,
                    $widthValue,
                    $widthhUom,
                    $heightValue,
                    $heightUom,
                    $actualWeightValue,
                    $actualWeightUom,
                    $stackable,
                    $splittable,
                    $grounded,
                    $userId,
                    $data['createdon'],
                    $ldm,
                    $dangerousGoodsFlag,
                    0, // second_weight is decimal
                    '',
                    $actualVolumeValue,
                    $actualVolumeUom,
                    $volumetricWeightValue,
                    $volumetricWeightUom,
                ]
            );

            if ($cargoDetails) {
                $cargoRowId = $this->db->insert_id();

                $orderCargoDetail = $this->db->query(
                    'INSERT INTO tb_order_cargodetails(
  						order_id,
						cargo_id,
						handling_unit,
						length,
						width,
						weight,
						second_weight,
						ldm,
                        quantity,
                        height,
						cargo_content,
						createdon,
                        volumetric_weight,
                        volweight_uom,
                        volume
                         )
                          VALUES(?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?,?,?,?)',
                    [
                        $data['bookingRowId'],
                        $cargoRowId,
                        $handlingUnit,
                        $lengthValue,
                        $widthValue,
                        $actualWeightValue,
                        $weightValue,
                        $ldm,
                        $quantity,
                        $heightValue,
                        $cargo['CargoType'] ?? $cargos['CargoType'],
                        $data['createdon'],
                        $volumetricWeightValue,
                        $volumetricWeightUom,
                        $actualVolumeValue,
                    ]
                );

                if (!$orderCargoDetail) {
                    return false;
                }
            } else {
                return false;
            }
            if (isset($cargos['CargoType'])) {
                break;
            }
        }

        return true;
    }

    private function saveReferences(array $data): bool
    {
        $references = $data['OrderDetails']['ManageReferences']['RefType'];

        if (empty($references)) {
            log_error('OpenBox Order Create : save References details data unavailable.');
            return false;
        }

        foreach ($references as $reference) {
            $referenceCode = $reference['Code'] ?? $references['Code'];
            $referenceValue = $reference['Value'] ?? $references['Value'];

            if (!empty($referenceCode) && !empty($referenceValue)) {
                $orderReference = $this->db->query(
                    'INSERT INTO tb_order_references(
  					order_id,
					reference_id,
					ref_value,
					status,
					createdon
                         )
                          VALUES(?, ?, ?, ?, ?)',
                    [
                        $data['bookingRowId'],
                        $referenceCode,
                        $referenceValue,
                        1,
                        $data['createdon'],
                    ]
                );

                if (!$orderReference) {
                    return false;
                }
            }
            if (isset($references['Code'])) {
                break;
            }
        }

        if (isset($data['OrderDetails']['OrderStartLocation']['DriverInstructions']) &&
            !empty($data['OrderDetails']['OrderStartLocation']['DriverInstructions']) &&
            !is_array($data['OrderDetails']['OrderStartLocation']['DriverInstructions'])
        ) {
            if ($data['Action'] ==  OpenboxMessageHandler::ACTION_CREATE) {
                $this->db->query(
                    'INSERT INTO tb_order_references(
  					order_id,
					reference_id,
					ref_value,
					status,
					createdon
                         )
                          VALUES(?, ?, ?, ?, ?)',
                    [
                        $data['bookingRowId'],
                        'ORD_PIKINST',
                        $data['OrderDetails']['OrderStartLocation']['DriverInstructions'],
                        1,
                        $data['createdon'],
                    ]
                );
            } else {
                $this->db->query(
                    'UPDATE tb_order_references set
					ref_value = ?,
					updatedon = ? where order_id = ? and reference_id = ?',
                    [
                        $data['OrderDetails']['OrderStartLocation']['DriverInstructions'],
                        $data['createdon'],
                        $data['bookingRowId'],
                        'ORD_PIKINST',
                    ]
                );
            }
        }

        if (isset($data['OrderDetails']['OrderEndLocation']['DriverInstructions']) &&
            !empty($data['OrderDetails']['OrderEndLocation']['DriverInstructions']) &&
            !is_array($data['OrderDetails']['OrderEndLocation']['DriverInstructions'])
        ) {
            if ($data['Action'] ==  OpenboxMessageHandler::ACTION_CREATE) {
                $this->db->query(
                    'INSERT INTO tb_order_references(
  					order_id,
					reference_id,
					ref_value,
					status,
					createdon
                         )
                          VALUES(?, ?, ?, ?, ?)',
                    [
                        $data['bookingRowId'],
                        'ORD_DLVINST',
                        $data['OrderDetails']['OrderEndLocation']['DriverInstructions'],
                        1,
                        $data['createdon'],
                    ]
                );
            } else {
                $this->db->query(
                    'UPDATE tb_order_references set
					ref_value = ?,
					updatedon = ? where order_id = ? and reference_id = ?',
                    [
                        $data['OrderDetails']['OrderEndLocation']['DriverInstructions'],
                        $data['createdon'],
                        $data['bookingRowId'],
                        'ORD_DLVINST',
                    ]
                );
            }
        }

        return true;
    }

    private function saveRemarks(array $data): void
    {
        if (!isset($data['OrderDetails']['ManageRemarks']['Remark']) ||
            empty($data['OrderDetails']['ManageRemarks']['Remark'])
        ) {
            log_error('OpenBox Order Create : save Remarks details data unavailable.');
            return;
        }

        foreach ($data['OrderDetails']['ManageRemarks']['Remark'] as $remark) {
            $remarkValue = $data['OrderDetails']['ManageRemarks']['Remark']['Value'] ?? $remark['Value'];

            if (strlen($remarkValue) > 1000) {
                log_error('OpenBox Order Create :  Exceeded length of remarks value.');
                return;
            }

            $this->db->query(
                'INSERT INTO tb_order_remarks(
  					order_id,
					remark_id,
					description
                         )
                          VALUES(?, ?, ?)',
                [
                    $data['bookingRowId'],
                    $this->openBoxAdminUseId(),
                    $remark['Value'] ?? '',
                ]
            );
        }
    }

    private function saveOrderPartys(array $data): bool
    {
        if (!isset($data['shipper_id'], $data['drop_partyid'])) {
            log_error('OpenBox Order Create : save OrderPartys details data unavailable.');
            return false;
        }

        $userId = $this->openBoxUserId();
        foreach ([$data['shipper_id'], $data['drop_partyid']] as $key => $id) {
            if ($key == 0) {
                $partyData = $this->getShipperConsineeId($id, 'SHIPPER');
            } else {
                $partyData = $this->getShipperConsineeId($id, 'CONSIGNEE');
            }

            $orderPartyAddress = $this->db->query(
                'INSERT INTO tbl_orderparty_address(
  					order_id,
					party_master_id,
					location_id,
					street,
					state,
					address,
					pincode,
					country,
					user_id,
					status
                         )
                          VALUES(?, ?, ?, ?, ?, ?, ?, ?, ?, ?)',
                [
                    $data['bookingRowId'],
                    $partyData['partyId'],
                    $data['OrderDetails']['OrderStartLocation']['Address']['City'],
                    $data['OrderDetails']['OrderStartLocation']['Address']['Street'],
                    $data['OrderDetails']['OrderStartLocation']['Address']['State'],
                    $data['OrderDetails']['OrderStartLocation']['Address']['Name'],
                    $data['OrderDetails']['OrderStartLocation']['Address']['Postal'],
                    $data['OrderDetails']['OrderStartLocation']['Address']['Country'],
                    $userId,
                    1,
                ]
            );

            if (!$orderPartyAddress) {
                return false;
            } else {
                $result = true;
            }
        }
        return $result;
    }

    private function saveInvolvedPartys(array $data): bool
    {
        if (!isset($data['OrderDetails']['OrderParties']['PartyType']) &&
            !empty($data['OrderDetails']['OrderParties']['PartyType'])) {
            log_error('OpenBox Order Create : save Involved Parties details data unavailable.');
            return false;
        }

        foreach ($data['OrderDetails']['OrderParties']['PartyType'] as $partyType) {
            $partyData = $this->getShipperConsineeId($partyType['PartyID']['ID'], $partyType['Type']);

            $orderParty = $this->db->query(
                'INSERT INTO tb_order_parties(
  				    order_id,
					order_number,
					party_type,
					party_id,
                    createdon,
					status
                         )
                          VALUES(?, ?, ?, ?, ?, ?)',
                [
                    $data['bookingRowId'],
                    $data['bookingId'],
                    $partyData['partyType'],
                    $partyData['partyId'],
                    $data['createdon'],
                    '1',
                ]
            );
            log_error('OpenBox Order Create : partyId - ' . $partyData['partyId']);
            if (!$orderParty) {
                return false;
            }
        }
        return true;
    }

    private function getServiceId(string $serviceType): ?int
    {
        $result = $this->db->query(
            'SELECT id from tb_service_master
                WHERE name = ? and  status = ?',
            [$serviceType, 1]
        );

        return $result->row()->id ?? null;
    }

    public function getCount(string $table, array $values): int
    {
        $this->db->from($table);
        foreach ($values as $key => $value) {
            $this->db->where($key, $value);
        }

        $db_results =  $this->db->get();
        return $db_results->num_rows();
    }

    private function getShipperConsineeId(string $orderShipperId, string $partyTypeName): array
    {
        $partyId = 0;
        $partyType = 0;
        $party_types = '';

        $result = $this->db->query(
            'SELECT id, party_types from tbl_party_master
                WHERE code = ? and  status = ?',
            [$orderShipperId, 1]
        );
        if (is_object($result->row())) {
            $partyId = $result->row()->id;
            $party_types = $result->row()->party_types;
        }
        switch ($partyTypeName) {
            case 'CONSIGNEE':
                $partyType = $this->getPartytypeId($party_types, 'Consignee');
                break;
            case 'SHIPPER':
                $partyType = $this->getPartytypeId($party_types, 'Shipper');
                break;
            case 'CUSTOMER':
                $partyType = $this->getPartytypeId($party_types, 'Customer');
                break;
            default:
                log_error('OpenBox Order Create : Unknown Party - ' . $partyTypeName);
        }

        return ['partyId' => $partyId, 'partyType' => $partyType];
    }

    public function getPartytypeId(string $ids, string $type): int
    {
        $result = $this->db->query(
            "SELECT id, name FROM tbl_party_types
                WHERE id IN ? AND name = ? AND status = ?",
            [explode(',', $ids), $type, 1]
        );

        return $result->row()->id ?? 0;
    }

    /**
     * @param array $data
     * @param string $companyCode
     * @param string $branchCode
     * @param int $partyMasterId
     * @param array $address
     * @return bool
     */
    public function createCustomer(array $data, string $companyCode, string $branchCode, int $partyMasterId, array $address) : bool
    {
        $toSave = array(
            'name' => $data['PartyID']['Name'],
            'phone' => (isset($data['ContactData']['Phone']) && !empty($data['ContactData']['Phone'])) ? implode('', $data['ContactData']['Phone']) : '0123456789',
            'code' => $data['PartyID']['ID'],
            'email_id' => $data['ContactData']['EmailAddress'] ?? null,
            'password' => password_hash($data['PartyID']['Name'].$data['PartyID']['ID'].time(),PASSWORD_DEFAULT),
            'user_id' => 0,
            'partner_id' => $partyMasterId,
            'status' => 1,
            'company_code' => $companyCode,
            'branch_code' => $branchCode
        );

        ;
        if (!$this->db->insert('tb_customers', $toSave)) {
            return false;
        }
        log_error('OpenBox Order Create : New customer created - ' . $this->db->insert_id());
        return true;

    }

    public function createParty(
        array $data,
        string $companyCode,
        string $branchCode,
        string $departmentCode,
        array $address
    ): bool {
        $partyTypeId = $this->getNewPartyTypeId($companyCode, $branchCode, $data['Type']);

        if ($partyTypeId == 0) {
            log_error('OpenBox Order Create : Party Type id not found in database.');
            return false;
        }

        switch ($data['Type']) {
            case 'CONSIGNEE':
                $tag = 'OrderStartLocation';
                break;
            case 'SHIPPER':
                $tag = 'OrderEndLocation';
                break;
            case 'CUSTOMER':
                $tag = 'expected_not_available';
                break;
            default:
                log_error('OpenBox Order Create : Unknown New Party - ' . $data['Type']);
                return false;
        }

        $saveParty = $this->db->query(
            'INSERT INTO tbl_party_master(
  				    party_type_id,
					name,
					email,
					mobile,
                    user_id,
                    code,
                    company_code,
                    branch_code,
                    customeridentifier,
                    department_code,
                    currency,
                    country,
                    state,
                    location_id,
                    street,
                    pincode,
                    kn_login_account,
                    party_types,
					status
                         )
                          VALUES(?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?)',
            [
                $partyTypeId,
                $data['PartyID']['Name'],
                $data['ContactData']['EmailAddress'],
                $data['ContactData']['Phone']['ContactNo'] ?? null,
                $this->openBoxAdminUseId(),
                $data['PartyID']['ID'],
                $companyCode,
                $branchCode,
                $data['PartyID']['ID'],
                $departmentCode,
                $address['GoodsValue']['Currency'],
                $address[$tag]['Address']['Country'] ?? null,
                $address[$tag]['Address']['State'] ?? null,
                $address[$tag]['Address']['City'] ?? null,
                $address[$tag]['Address']['Street'] ?? null,
                $address[$tag]['Address']['Postal'] ?? null,
                $data['PartyID']['ID'],
                $partyTypeId,
                '1',
            ]
        );

        if (!$saveParty) {
            return false;
        }
        log_error('OpenBox Order Create : New party created - ' . $data['PartyID']['ID']);
        return true;
    }

    public function getNewPartyTypeId(string $companyCode, string $branchCode, string $party): int
    {
        $result = $this->db->query(
            'SELECT id FROM tbl_party_types
                WHERE name = ? AND company_code = ? AND branch_code = ? AND user_id = ? AND status = ?',
            [$party, $companyCode, $branchCode, $this->openBoxAdminUseId(), 1]
        );

        if (null !== $result->row()) {
            return (int)$result->row()->id;
        }

        // Create party type when missing
        $this->db->insert(
            'tbl_party_types',
            ['name' => $party, 'company_code' => $companyCode, 'branch_code' => $branchCode, 'user_id' => $this->openBoxAdminUseId(), 'status' => 1]
        );

        return (int)$this->db->insert_id();
    }

    public function openBoxUserId(): int
    {
        $result = $this->db->query(
            'SELECT id from tb_customers
                WHERE name = ? and  company_code = ? and status = ?',
            ['COMMON WEALTH GAMES', 'UKKN-CG', 1]
        );

        return $result->row()->id ?? 0;
    }

    public function openBoxAdminUseId(): int
    {
        $result = $this->db->query(
            'SELECT id from tb_users
                WHERE company_code = ? and status = ?',
            ['UKKN-CG', 'Active']
        );

        return $result->row()->id ?? 0;
    }

    public function checkSequenceNumber(int $externalId): int
    {
        $result = $this->db->query(
            'SELECT sequence_number from tb_orders
                WHERE external_order_id = ?  and status <> ?',
            [$externalId, 0]
        );

        return  (int)$result->row()->sequence_number ?? 0;
    }

    public function updateOrder(array $data): bool
    {
        $this->db->trans_start();
        $order = $this->db->query(
            'UPDATE tb_orders SET
			goods_value = ?,
			currency = ?,
			shipper_id = ?,
			pickup_custid = ?,
			pickup_partyid = ?,
			drop_custid = ?,
			drop_partyid = ?,
			customer_id = ?,
			pickup_country = ?,
			pickup_city = ?,
			pickup_pincode = ?,
			pickup_company = ?,
			pickup_address1 = ?,
			pickup_address2 = ?,
			delivery_country = ?,
			delivery_city = ?,
			delivery_pincode = ?,
			delivery_company = ?,
			delivery_address1 = ?,
			delivery_address2 = ?,
			quantity = ?,
			volume = ?,
			weight = ?,
			pickup_datetime = ?,
			pickup_endtime = ?,
			delivery_datetime = ?,
			drop_endtime = ?,
			user_id = ?,
			product = ?,
			status = ?,
			is_created = ?,
			physicalreceiver = ?,
			LogicalReceiver = ?,
			physicalsender = ?,
			logicalsender = ?,
			transport_mode = ?,
			updatedon = ?,
			sequence_number = ? WHERE external_order_id = ?',
            [
                $data['goods_value'],
                $data['currency'],
                $data['shipper_id'],
                $data['pickup_custid'],
                $data['pickup_partyid'],
                $data['drop_custid'],
                $data['drop_partyid'],
                $data['customer_id'],
                $data['pickup_country'],
                $data['pickup_city'],
                $data['pickup_pincode'],
                $data['pickup_company'],
                $data['pickup_address1'],
                $data['pickup_address2'],
                $data['delivery_country'],
                $data['delivery_city'],
                $data['delivery_pincode'],
                $data['delivery_company'],
                $data['delivery_address1'],
                $data['dropState'],
                $data['quantity'],
                $data['volume'],
                $data['weight'],
                $data['pickup_datetime'],
                $data['pickup_endtime'],
                $data['delivery_datetime'],
                $data['drop_endtime'],
                $this->openBoxAdminUseId(),
                $data['product'],
                1,
                1,
                $data['PhysicalReceiver'],
                $data['LogicalReceiver'],
                $data['PhysicalSender'],
                $data['LogicalSender'],
                $data['OrderDetails']['ModeOfTransport'],
                $data['createdon'],
                $data['SequenceNumber'],
                $data['external_order_id'],
            ]
        );

        if (!$order) {
            return false;
        }

        $result = $this->db->query(
            'SELECT id, order_id, shipment_id  from tb_orders WHERE external_order_id = ?', [$data['external_order_id']]
        );

        $data['bookingRowId'] = $result->row()->id;
        $data['bookingId'] = $result->row()->order_id;
        $data['shipment_id'] = $result->row()->shipment_id;

        if (!$this->saveOrderDetails($data)) {
            return $this->rollback("OpenBox Order Create : Update order Details failed Trans status rollback");
        }

        $shipmentId = $this->saveOrUpdateShipment($data);

        if (!is_int($shipmentId) || $shipmentId <= 0) {
            return $this->rollback("OpenBox Order Create : Update  Shipment Details failed Trans status rollback");
        }

        $getCargoId = $this->db->query(
            'SELECT cargo_id from tb_order_cargodetails
                WHERE order_id = ?',
            [$data['bookingRowId']]
        );

        $cargo = false;
        if (count($getCargoId->result())) {
            foreach ($getCargoId->result() as $row) {
                $this->db->query(
                    'UPDATE tb_cargo_details SET
    		status = ?  where id = ?',
                    [0, $row->cargo_id]
                );
            }

            $cargo = $this->db->query(
                'UPDATE tb_order_cargodetails SET
    		status = ?  where order_id = ?',
                [0, $data['bookingRowId']]
            );
        }

        if (!$cargo || !$this->saveCargo($data)) {
            return $this->rollback("OpenBox Order Create : Update Cargo failed Trans status rollback");
        }

        $references = $this->db->query(
            'UPDATE tb_order_references SET
    		status = ?  where order_id = ? and reference_id <> ? and reference_id <> ?',
            [0, $data['bookingRowId'], 'ORD_DLVINST', 'ORD_PIKINST']
        );

        if (!$references || !$this->saveReferences($data)) {
            return $this->rollback("OpenBox Order Create : Update  References failed Trans status rollback");
        }

        $orderPartys = $this->db->query(
            'UPDATE tbl_orderparty_address SET
    		status = ?  where order_id = ?',
            [0, $data['bookingRowId']]
        );

        if (!$orderPartys || !$this->saveOrderPartys($data)) {
            return $this->rollback("OpenBox Order Create : Save Order party types failed Trans status rollback");
        }

        $involvedPartys = $this->db->query(
            'UPDATE tb_order_parties SET
    		status = ?  where order_id = ?',
            [0, $data['bookingRowId']]
        );

        if (!$involvedPartys || !$this->saveInvolvedPartys($data)) {
            return $this->rollback("OpenBox Order Create : Involved parties types failed Trans status rollback");
        }

        $remarks = $this->db->query(
            'UPDATE tb_order_remarks SET
    		status = ?  where order_id = ?',
            [0, $data['bookingRowId']]
        );

        if ($remarks) {
            $this->saveRemarks($data);
        }

        $this->db->trans_commit();
        return true;
    }

    public function getOrderId(int $externalId): int
    {
        $result = $this->db->query(
            'SELECT id from tb_orders
                WHERE external_order_id = ?  and status <> ?',
            [$externalId, 0]
        );

        return (int)$result->row()->id ?? 0;
    }
}
