<?php

class Transportwaybillforwardermodel extends CI_Model
{
    private const DATE_STYLE = 'd.m.Y';

    public function __construct()
    {
        $this->load->helper('log');
    }

    public function getOrderInfo(int $shift_id, int $customer_id): array
    {
        $result = [];
        $query = "SELECT
                    shift_id AS temp_shift_id
                    ,MIN(pickup_datetime) AS order_min_pickup_datetime
                    ,MAX(drop_endtime) AS order_max_drop_endtime
                    ,(SELECT MIN(pickup_datetime) AS trip_min_pickup_datetime FROM tb_orders WHERE shift_id=temp_shift_id) AS trip_min_pickup_datetime
                    ,(SELECT MAX(drop_endtime) AS trip_drop_endtime FROM tb_orders WHERE shift_id=temp_shift_id) AS trip_max_drop_endtime
                    ,GROUP_CONCAT(DISTINCT tb_orders.order_id) order_ids
                    ,GROUP_CONCAT(DISTINCT tb_orders.id) ids
                    ,sum(goods_value) as goods_value
                    ,tb_orders.customer_id
                    ,tb_orders.vendor_id
                    ,tb_orders.shipmentid
                    ,tb_shifts.created_on AS transport_order_date
                FROM tb_orders
                JOIN tb_shifts ON (tb_shifts.id=tb_orders.shift_id)
                WHERE tb_orders.shift_id= ? and tb_orders.customer_id= ? and tb_shifts.status =1
                GROUP BY customer_id";
        $this->db->trans_begin();
        $result_set = $this->db->query($query, [$shift_id, $customer_id]);
        if ($result_set && $result_set->num_rows() > 0) {
            $result['common_info'] = $result_set->row_array();
        } else {
            $this->db->trans_rollback();
            $result['transaction_status'] = false;
            log_error("Unable to execute query :" . json_encode($this->db->error()));
            return $result;
        }
        if (isset($result['common_info']['temp_shift_id']) && is_numeric($result['common_info']['temp_shift_id'])) {
            $result['common_info']['order_min_pickup_datetime'] = self::dateFormatter($result['common_info']['order_min_pickup_datetime']);
            $result['common_info']['order_max_drop_endtime'] = self::dateFormatter($result['common_info']['order_max_drop_endtime']);
            $result['common_info']['trip_min_pickup_datetime'] = self::dateFormatter($result['common_info']['trip_min_pickup_datetime']);
            $result['common_info']['trip_max_drop_endtime'] = self::dateFormatter($result['common_info']['trip_max_drop_endtime']);
            $result['common_info']['transport_order_date'] = self::dateFormatter($result['common_info']['transport_order_date']);
            $result['common_info']['trip_sequence_no'] = get_trip_sequence_no($shift_id, $result['common_info']['transport_order_date']);
            $order_ids_list = explode(',', $result['common_info']['ids']);
            $result['vehicle_and_driver_info'] = $this->getVehicleAndDriverInfo($shift_id);
            $orderDetailsList = [];
            foreach ($order_ids_list as $order_id) {
                $orderDetailsList[$order_id] = [
                    'order_parties' => $this->getOrderPartyTypes($order_id),
                    'order_cargos' => $this->getOrderCargoDetails($order_id),
                ];
            }
            $result['orderDetailsList'] = $orderDetailsList;
            $result['driver_accepted_date'] = $this->getAcceptStatusInfo($shift_id) ?? $result['common_info']['transport_order_date'];
        }
        if ($this->db->trans_status() === false) {
            $this->db->trans_rollback();
            $result['transaction_status'] = false;
        } else {
            $this->db->trans_commit();
            $result['transaction_status'] = true;
        }
        return $result;
    }

    public function getOrderPartyTypes(int $order_id): array
    {
        $result = [];
        $query = "SELECT
                        tbl_party_types.`name` as role,
                        tbl_party_master.name,tbl_party_master.email,tbl_party_master.tax_payer_no,
                        tbl_party_master.pincode,tbl_party_master.location_id AS city,tbl_party_master.state,
                        tbl_party_master.street,tbl_party_master.house_number,tbl_party_master.building,
                        tbl_party_master.mobile
                    FROM `tb_order_parties`
                    JOIN `tbl_party_types` ON (tb_order_parties.party_type= `tbl_party_types`.id)
                    JOIN `tbl_party_master` ON (tb_order_parties.party_id = tbl_party_master.id AND tbl_party_master.status = 1)
                    WHERE order_id= ? AND tb_order_parties.`status`=1
                    ORDER BY tbl_party_master.name";
        $result_set = $this->db->query($query, [$order_id]);
        if ($result_set && $result_set->num_rows() > 0) {
            $result = $result_set->result_array();
            foreach ($result as $index => $involvedparty) {
                $result[$index]['summary'] = ($involvedparty['name'] ?? ' ') . ' ' . ($involvedparty['tax_payer_no'] ?? ' ') . ' ' . ($involvedparty['pincode'] ?? ' ') . ' ' . ($involvedparty['state'] ?? ' ') . ' ' . ($involvedparty['city'] ?? ' ') . ' ' . ($involvedparty['street'] ?? ' ') . ' ' . ($involvedparty['house_number'] ?? ' ') . ' ' . ($involvedparty['building'] ?? ' ');
                $result[$index]['address_summary'] = ($involvedparty['pincode'] ?? ' ') . ' ' . ($involvedparty['state'] ?? ' ') . ' ' . ($involvedparty['city'] ?? ' ') . ' ' . ($involvedparty['street'] ?? ' ') . ' ' . ($involvedparty['house_number'] ?? ' ') . ' ' . ($involvedparty['building'] ?? ' ');
            }
        } else {
            log_error("Unable to execute query :" . json_encode($this->db->error()));
        }
        return $result;
    }

    public function getOrderCargoDetails(int $order_id): array
    {
        $result = [];
        $select = ['SUM(tb_order_cargodetails.quantity) quantity', 'SUM(tb_order_cargodetails.weight) as weight', 'SUM(tb_order_cargodetails.volume) volume', 'GROUP_CONCAT(DISTINCT tb_cargo_details.goods_description) goods_description', 'GROUP_CONCAT(DISTINCT tb_cargo_details.cargo_type) cargo_type'];
        $result_set = $this->db->select($select)
            ->from('tb_order_cargodetails')
            ->join('tb_cargo_details', 'tb_order_cargodetails.cargo_id = tb_cargo_details.id')
            ->where(['order_id' => $order_id, 'tb_order_cargodetails.status' => 1])
            ->limit(1)
            ->get();
        if ($result_set && $result_set->num_rows() > 0) {
            $result = $result_set->row_array();
        } else {
            log_error("Unable to execute query :" . json_encode($this->db->error()));
        }
        return $result;
    }

    public function getVehicleAndDriverInfo(int $shift_id): array
    {
        $vehicle_driver_info = [];
        $select = ['tb_truck_drivers.name driver_name', 'tb_truck_drivers.contact_num driver_contact_number', 'tb_trucks_data.truck_volume', 'tb_trucks_data.truck_number', 'tb_trucks_data.truck_brand', 'tb_trucks_data.truck_weight', 'tb_trucks_data.register_number as vehicle_register_number', 'tb_trucks_data.vehicle_owner', 'tb_trucks_data.description'];
        $this->db->select($select)
            ->from('tb_shft_veh')
            ->join('tbl_assigned_drivers', 'tb_shft_veh.vehicle_id =tbl_assigned_drivers.vehicle_id AND tbl_assigned_drivers.status=1')
            ->join('tb_truck_drivers', "tb_truck_drivers.id = tbl_assigned_drivers.driver_id AND `tb_truck_drivers`.`status`='active'")
            ->join('tb_trucks_data', "tb_trucks_data.id = tb_shft_veh.vehicle_id AND .tb_trucks_data.status='Active'")
            ->join('tb_trucktypes', 'tb_trucktypes.id = tb_trucks_data.truck_type', 'left')
            ->where(['tb_shft_veh.shft_id' => $shift_id, 'tb_shft_veh.status' => 1])->limit(1);
        $vehicle_driver_result_set = $this->db->get();
        if ($vehicle_driver_result_set && $vehicle_driver_result_set->num_rows() > 0) {
            $vehicle_driver_info = $vehicle_driver_result_set->row_array();
        } else {
            log_error("Unable to execute query :" . json_encode($this->db->error()));
        }
        return $vehicle_driver_info;
    }

    private static function dateFormatter(string $date, string $default = ''): string
    {
        if (empty($date)) {
            return $default;
        }
        $date = date_create($date);
        if ($date === false) {
            return $default;
        }
        return date_format($date, self::DATE_STYLE);
    }

    public function getAcceptStatusInfo(int $shift_id): ?string
    {
        $this->load->model('common');
        $result = $this->common->gettblrowdata(['shipment_id' => $shift_id, 'status_code' => '0212', 'status' => 1], 'createdon', 'tb_stop_status', 0, 0);
        return isset($result['createdon']) ? self::dateFormatter($result['createdon']) : null;
    }

    public function getDocumentSignature(int $is_main_user, int $id): string
    {
        $this->load->model('common');
        $table = $is_main_user ? 'tb_users' : 'tb_user_logins';
        $usersinfo = $this->common->gettblrowdata(["id" => $id], 'signpath', $table, 0, 0);
        return $usersinfo['signpath'] ?? '';
    }
}
