<?php

class CI_Envelope extends CI_Email
{
    private const LOG_FILEPATH = './application/logs/envelope.log';
    private bool $useLogFile = false;

    public function __construct(array $config = [])
    {
        parent::__construct(
            !empty($config) ? $config : [
                'protocol' => 'smtp',
                'smtp_host' => KN_SMTP_HOST,
                'smtp_port' => KN_SMTP_PORT,
                'crlf' => "\r\n",
                'newline' => "\r\n",
            ]
        );

        if ($this->shouldUseLogFileInsteadOfSendingEmail()) {
            $this->useLogFile = true;
        }
    }

    /**
     * Send email, or use Envelope Log File
     *
     * @param bool $auto_clear = TRUE
     * @return    bool
     */
    public function send($auto_clear = true): bool
    {
        if (!$this->useLogFile) {
            return parent::send($auto_clear);
        }

        try {
            $this->_build_headers();
            $this->_build_message();
            $status = 'ok';
        } catch (Exception $e) {
            $status = $e->getMessage();
        }

        $this->addLogLine([
            'status' => $status,
            'from' => $this->_headers['From'] . ' ' . $this->_headers['X-Sender'],
            'to' => implode('<br />', $this->_recipients),
            'bcc' => implode('<br />', $this->_cc_array),
            // 'content-type' => $this->_get_content_type(),
            'subject' => preg_replace(['/(=\?UTF-8\?Q\?)/', '/\?$/', '/\?/'], ['', ''], quoted_printable_decode($this->_subject)),
            'message' => $this->_body,
            'attachments' => $this->showAttachmentsAsList($this->_attachments ?: [])
        ]);
        return true;
    }

    private function showAttachmentsAsList(array $attachments): array
    {
        $list = [];
        foreach ($attachments as $row) {
            $url = str_replace('/var/www/html/', '', base_url($row['name'][0]));
            $name = str_replace('/var/www/html/', '', $row['name'][0]);
            $list [] = $row['type'] .': <a href="'.$url.'" target="_blank">'.$name.'</a>';
        }

        return $list;
    }

    public function clearEnvelopeLog(): void
    {
        unlink(self::LOG_FILEPATH);
    }

    public function getEnvelopeLogAsHtml(int $limit = 0): string
    {
        $logs = file_exists(self::LOG_FILEPATH)
            ? file(self::LOG_FILEPATH)
            : [];
        rsort($logs);
        $size = count($logs);

        if ($limit) {
            $logs = array_slice($logs, 0, $limit);
        }

        foreach ($logs as $k => $line) {
        try {
                $logs [$k] = json_decode($line, true, 512, JSON_THROW_ON_ERROR) ?? [];
            } catch (Exception $e) {
                $logs [$k] = ['error' => $e->getMessage()];
            }
        }

        return $this->transformArrayToHtmlString($logs, $size);
    }

    private function shouldUseLogFileInsteadOfSendingEmail(): bool
    {
        return SYS_TYPE !== 'TMS';
    }

    private function addLogLine(array $data): void
    {
        $data = array_merge(['time' => $timeMarker = date("Y-m-d H:i:s")], $data);

        try {
            $logFile = fopen(self::LOG_FILEPATH, 'ab+');
            fwrite($logFile, json_encode($data, JSON_THROW_ON_ERROR) . "\n");
            fclose($logFile);
        } catch (Exception $e) {
            log_message('error', $e->getMessage());
        }
    }

    private function transformArrayToHtmlString(array $rows, int $size = 0): string
    {
        $ci = &get_instance();

        $heads = [];
        foreach ($rows as $row) {
            $heads = array_unique(array_merge($heads, array_keys($row)));
        }

        return $ci->load->view('custom/envelope_log', [ 'heads' => $heads, 'rows' => $rows, 'size' => $size ], true);
    }
}
